# ABSTRACT: Perl 5 API wrapper for Trello
package API::Trello;

use API::Trello::Class;

extends 'API::Trello::Client';

use Carp ();
use Scalar::Util ();

our $VERSION = '0.01'; # VERSION

has key => (
    is       => 'rw',
    isa      => STRING,
    required => 1,
);

has token => (
    is       => 'rw',
    isa      => STRING,
    required => 0,
);

has identifier => (
    is       => 'rw',
    isa      => STRING,
    default  => 'API::Trello (Perl)',
);

has version => (
    is       => 'rw',
    isa      => INTEGER,
    default  => 1,
);

method AUTOLOAD () {
    my ($package, $method) = our $AUTOLOAD =~ /^(.+)::(.+)$/;
    Carp::croak "Undefined subroutine &${package}::$method called"
        unless Scalar::Util::blessed $self && $self->isa(__PACKAGE__);

    my @segments = @_;

    # camelize the even parts
    for (my $i = 0; $i < @segments; $i++) {
        $segments[$i] =~ s/_([a-z])/\U$1/g if $i % 2 != 0;
    }

    # return new resource instance dynamically
    return $self->resource($method, @segments);
}

method BUILD () {
    my $key        = $self->key;
    my $token      = $self->token;
    my $identifier = $self->identifier;
    my $version    = $self->version;
    my $agent      = $self->user_agent;
    my $url        = $self->url;

    $agent->transactor->name($identifier);

    $url->path("/$version");
    $url->query(key => $key, $token ? (token => $token) : ());

    return $self;
}

method PREPARE ($ua, $tx, %args) {
    my $headers = $tx->req->headers;
    my $url     = $tx->req->url;

    # default headers
    $headers->header('Content-Type' => 'application/json');
}

method action ($method, %args) {
    $method = uc($method || 'get');

    # execute transaction and return response
    return $self->$method(%args);
}

method create (%args) {
    # execute transaction and return response
    return $self->POST(%args);
}

method delete (%args) {
    # execute transaction and return response
    return $self->DELETE(%args);
}

method fetch (%args) {
    # execute transaction and return response
    return $self->GET(%args);
}

method resource (@segments) {
    # build new resource instance
    my $instance = __PACKAGE__->new(
        debug      => $self->debug,
        fatal      => $self->fatal,
        retries    => $self->retries,
        timeout    => $self->timeout,
        user_agent => $self->user_agent,
        key        => $self->key,
        token      => $self->token,
        identifier => $self->identifier,
        version    => $self->version,
    );

    # resource locator
    my $url = $instance->url;

    # modify resource locator if possible
    $url->path(join '/', $self->url->path, @segments);

    # return resource instance
    return $instance;
}

method update (%args) {
    # execute transaction and return response
    return $self->PUT(%args);
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

API::Trello - Perl 5 API wrapper for Trello

=head1 VERSION

version 0.01

=head1 SYNOPSIS

    use API::Trello;

    my $trello = API::Trello->new(
        key        => 'KEY',
        token      => 'TOKEN',
        identifier => 'APPLICATION (yourname@example.com)',
    );

    my $board = $trello->boards(1);
    my $results = $board->fetch;

    # after some introspection

    $board->delete;

=head1 DESCRIPTION

This distribution provides an object-oriented thin-client library for
interacting with the Trello (L<http://trello.com>) API. For usage and
documentation information visit L<https://trello.com/docs/gettingstarted/index.html>.

=head1 THIN CLIENT

A thin-client library is advantageous as it has complete coverage and can
easily adapt to changes in the API with minimal effort. As a thin-client
library, this module does not map specific HTTP requests to specific routines
nor does it provide parameter validation, pagination, or other conventions
found in typical API client implementations, instead, it simply provides a
simple and consistent mechanism for dynamically generating HTTP requests.
Additionally, this module has support for debugging and retrying API calls as
well as throwing exceptions when 4xx and 5xx server response codes are
received.

=head2 Building

    my $board = $trello->boards(1);
    my $checklists = $board->checklists;

    $checklists->action; # GET /boards/1/checklists
    $checklists->action('head'); # HEAD /boards/1/checklists

Building up an HTTP request object is extremely easy, simply call method names
which correspond to the API's path segments in the resource you wish to execute
a request against. This module uses autoloading and returns a new instance with
each method call. The following is the equivalent:

    my $board = $trello->resource(boards => 1);
    my $checklists = $board->resource('checklists');

    # or

    my $checklists = $trello->resource('boards', 1, 'checklists');

    # then

    $checklists->action('put', %args); # PUT /boards/1/checklists

Because each call returns a new API instance configured with a resource locator
based on the supplied parameters, reuse and request isolation are made simple,
i.e., you will only need to configure the client once in your application.

=head2 Fetching

    my $boards = $trello->boards;

    $boards->fetch(
        query => {
            # query-string parameters
        },
    );

    # equivalent to

    $trello->resource('boards')->action(
        get => ( query => { ... } )
    );

This example illustrates how you might fetch an API resource.

=head2 Creating

    my $boards = $trello->boards;

    $boards->create(
        data => {
            # content-body parameters
        },
        query => {
            # query-string parameters
        },
    );

    # or

    my $boards = $trello->boards->create(...);

    # equivalent to

    $trello->resource('boards')->action(
        post => ( query => { ... }, data => { ... } )
    );

This example illustrates how you might create a new API resource.

=head2 Updating

    my $boards = $trello->boards;
    my $board  = $boards->resource(1);

    $board->update(
        data => {
            # content-body parameters
        },
        query => {
            # query-string parameters
        },
    );

    # or

    my $board = $trello->boards(1);

    $board->update(...);

    # equivalent to

    $trello->resource('boards')->action(
        put => ( query => { ... }, data => { ... } )
    );

This example illustrates how you might update an new API resource.

=head2 Deleting

    my $boards = $trello->boards;
    my $board  = $boards->resource(1);

    $board->delete(
        data => {
            # content-body parameters
        },
        query => {
            # query-string parameters
        },
    );

    # or

    my $board = $trello->boards(1);

    $board->delete(...);

    # equivalent to

    $trello->resource('boards')->action(
        delete => ( query => { ... }, data => { ... } )
    );

This example illustrates how you might delete an API resource.

=head2 Transacting

    my $board = $trello->resource('boards', '1');

    my ($results, $transaction) = $board->fetch(...);

This example illustrates how you can access the transaction object used to
submit the HTTP request.

=head1 PARAMETERS

=head2 key

    $trello->key;
    $trello->key('KEY');

The key parameter should be set to the account holder's API key.

=head2 token

    $trello->token;
    $trello->token('TOKEN');

The token parameter should be set to the account holder's API access token.

=head2 identifier

    $trello->identifier;
    $trello->identifier('IDENTIFIER');

The identifier parameter should be set using a string to identify your app.

=head1 ATTRIBUTES

=head2 debug

    $trello->debug;
    $trello->debug(1);

The debug attribute if true prints HTTP requests and responses to standard out.

=head2 fatal

    $trello->fatal;
    $trello->fatal(1);

The fatal attribute if true promotes 4xx and 5xx server response codes to
exceptions, a L<API::Trello::Exception> object.

=head2 retries

    $trello->retries;
    $trello->retries(10);

The retries attribute determines how many times an HTTP request should be
retried if a 4xx or 5xx response is received. This attribute defaults to 0.

=head2 timeout

    $trello->timeout;
    $trello->timeout(5);

The timeout attribute determines how long an HTTP connection should be kept
alive. This attribute defaults to 10.

=head2 url

    $trello->url;
    $trello->url(Mojo::URL->new('https://api.trello.com'));

The url attribute set the base/pre-configured URL object that will be used in
all HTTP requests. This attribute expects a L<Mojo::URL> object.

=head2 user_agent

    $trello->user_agent;
    $trello->user_agent(Mojo::UserAgent->new);

The user_agent attribute set the pre-configured UserAgent object that will be
used in all HTTP requests. This attribute expects a L<Mojo::UserAgent> object.

=head1 METHODS

=head2 action

    my $result = $trello->action($verb, %args);

    # e.g.

    $trello->action('head', %args);   # HEAD request
    $trello->action('optons', %args); # OPTIONS request
    $trello->action('patch', %args);  # PATCH request

The action method issues a request to the API resource represented by the
object. The first parameter will be used as the HTTP request method. The
arguments, expected to be a list of key/value pairs, will be included in the
request if the key is either C<data> or C<query>.

=head2 create

    my $results = $trello->create(%args);

    # or

    $trello->POST(%args);

The create method issues a C<POST> request to the API resource represented by
the object. The arguments, expected to be a list of key/value pairs, will be
included in the request if the key is either C<data> or C<query>.

=head2 delete

    my $results = $trello->delete(%args);

    # or

    $trello->DELETE(%args);

The delete method issues a C<DELETE> request to the API resource represented by
the object. The arguments, expected to be a list of key/value pairs, will be
included in the request if the key is either C<data> or C<query>.

=head2 fetch

    my $results = $trello->fetch(%args);

    # or

    $trello->GET(%args);

The fetch method issues a C<GET> request to the API resource represented by the
object. The arguments, expected to be a list of key/value pairs, will be
included in the request if the key is either C<data> or C<query>.

=head2 update

    my $results = $trello->update(%args);

    # or

    $trello->PUT(%args);

The update method issues a C<PUT> request to the API resource represented by
the object. The arguments, expected to be a list of key/value pairs, will be
included in the request if the key is either C<data> or C<query>.

=head1 RESOURCES

=head2 actions

    $trello->actions;

The actions method returns a new instance representative of the API
resource requested. This method accepts a list of path segments which will be
used in the HTTP request. The following documentation can be used to find more
information. L<https://trello.com/docs/api/action/index.html>.

=head2 batch

    $trello->batch;

The batch method returns a new instance representative of the API
resource requested. This method accepts a list of path segments which will be
used in the HTTP request. The following documentation can be used to find more
information. L<https://trello.com/docs/api/batch/index.html>.

=head2 boards

    $trello->boards;

The boards method returns a new instance representative of the API
resource requested. This method accepts a list of path segments which will be
used in the HTTP request. The following documentation can be used to find more
information. L<https://trello.com/docs/api/board/index.html>.

=head2 cards

    $trello->cards;

The cards method returns a new instance representative of the API
resource requested. This method accepts a list of path segments which will be
used in the HTTP request. The following documentation can be used to find more
information. L<https://trello.com/docs/api/card/index.html>.

=head2 checklists

    $trello->checklists;

The checklists method returns a new instance representative of the API
resource requested. This method accepts a list of path segments which will be
used in the HTTP request. The following documentation can be used to find more
information. L<https://trello.com/docs/api/checklist/index.html>.

=head2 labels

    $trello->labels;

The labels method returns a new instance representative of the API
resource requested. This method accepts a list of path segments which will be
used in the HTTP request. The following documentation can be used to find more
information. L<https://trello.com/docs/api/label/index.html>.

=head2 lists

    $trello->lists;

The lists method returns a new instance representative of the API
resource requested. This method accepts a list of path segments which will be
used in the HTTP request. The following documentation can be used to find more
information. L<https://trello.com/docs/api/list/index.html>.

=head2 members

    $trello->members;

The members method returns a new instance representative of the API
resource requested. This method accepts a list of path segments which will be
used in the HTTP request. The following documentation can be used to find more
information. L<https://trello.com/docs/api/member/index.html>.

=head2 notifications

    $trello->notifications;

The notifications method returns a new instance representative of the API
resource requested. This method accepts a list of path segments which will be
used in the HTTP request. The following documentation can be used to find more
information. L<https://trello.com/docs/api/notification/index.html>.

=head2 organizations

    $trello->organizations;

The organizations method returns a new instance representative of the API
resource requested. This method accepts a list of path segments which will be
used in the HTTP request. The following documentation can be used to find more
information. L<https://trello.com/docs/api/organization/index.html>.

=head2 search

    $trello->search;

The search method returns a new instance representative of the API
resource requested. This method accepts a list of path segments which will be
used in the HTTP request. The following documentation can be used to find more
information. L<https://trello.com/docs/api/search/index.html>.

=head2 sessions

    $trello->sessions;

The sessions method returns a new instance representative of the API
resource requested. This method accepts a list of path segments which will be
used in the HTTP request. The following documentation can be used to find more
information. L<https://trello.com/docs/api/session/index.html>.

=head2 tokens

    $trello->tokens;

The tokens method returns a new instance representative of the API
resource requested. This method accepts a list of path segments which will be
used in the HTTP request. The following documentation can be used to find more
information. L<https://trello.com/docs/api/token/index.html>.

=head2 types

    $trello->types;

The types method returns a new instance representative of the API
resource requested. This method accepts a list of path segments which will be
used in the HTTP request. The following documentation can be used to find more
information. L<https://trello.com/docs/api/type/index.html>.

=head2 webhooks

    $trello->webhooks;

The webhooks method returns a new instance representative of the API
resource requested. This method accepts a list of path segments which will be
used in the HTTP request. The following documentation can be used to find more
information. L<https://trello.com/docs/api/webhook/index.html>.

=head1 AUTHOR

Al Newkirk <anewkirk@ana.io>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2014 by Al Newkirk.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
