
=head1 NAME

STANAG - Library for processing STANAG 4586 messages

=head1 SYNOPSIS

    use STANAG qw(Get_Vehicle_Name Get_Vehicle_Subtype);

    use strict;
    use warnings;
    use Data::Dumper;

    my $hashref = STANAG::Vehicle_ID(); #get blank hash

    $hashref->{Vehicle_Type} = 22; #stick in some sample data
    $hashref->{Vehicle_Subtype} = 101;

    print Dumper($hashref); #see what's in it

    my $flat = STANAG::Encode_Vehicle_ID($hashref); #package up for sending (computes xsums)
    my $hashref2 = STANAG::Decode_Vehicle_ID($flat); #unpackage into hash (checks xsums)

    my $hashref3 = STANAG::Vehicle_ID(); #grab a new hash with latest used values
                                         #so this one gets the values we put into $hashref

    print Dumper($hashref2); 
    print Dumper($hashref3); #these print the same thing

    print Get_Vehicle_Name($hashref2->{Vehicle_Type}),"\n"; #demo decoding some numeric codes.
    print Get_Vehicle_Subtype($hashref2->{Vehicle_Subtype}),"\n";

=head1 ABSTRACT

Library acts as a wrapper around Parse::Binary::FixedFormat for STANAG 4586 messages. Most of the internals were autogenerated from the 4586 specification.

The module also stores an internal state hash representing a history of the latest values encoded or decoded.
This way when you get a message and decode it, and then want to respond, the new template is prepopulated.

=head1 EXPORT

None by default.

Available are:
Get_Vehicle_Name 
Get_Vehicle_Subtype 
Get_Country_Name

=over
=cut

use Parse::Binary::FixedFormat;
use List::Util qw(sum);
use AutoLoader;
use strict;
use warnings;

package STANAG;

use STANAG::CountryAndVehicleIDs;
use STANAG::Messages;

my %state; # holds any data that's been encoded or decoded for filling future messages

use Exporter 'import';
our @EXPORT_OK = qw(Get_Vehicle_Name Get_Vehicle_Subtype Get_Country_Name);

our $VERSION = '0.01';

=item AUTOLOAD
This function handles all access to the messages. This is the main interface to this module.

A call of STANAG::<message> will return a hashref for that message

A call of STANAG::Encode_<message>(hashref) will return the message, formatted and with xsums added

A call of STANAG::Decode_<message>(message) will return the hashref corresponding the the message
and will issue warnings if the xsums don't match.
=cut
use Carp qw(croak);
sub AUTOLOAD {
    our $AUTOLOAD;
    no strict 'refs';

    if($AUTOLOAD =~ /.*::(?:Encode_)(.*)/) {
        croak "Undefined subroutine &$AUTOLOAD called" unless exists $messages{$1};
        my $element = $1;
        *$AUTOLOAD = sub { #Handles Encode_<message> calls
            #this could be maybe optimized by assuming the xsum will always be last.
            my $hashref = shift;
            my $flat = $messages{$element}->format($hashref); 
            $hashref->{Checksum} = Checksum($flat);
            for( keys %{ $hashref } ) { #auto-apply to state
                $state{$_} = $hashref->{$_};
            }
            return $messages{$element}->format($hashref);
        }
    } elsif($AUTOLOAD =~ /.*::(?:Decode_)(.*)/) {
        croak "Undefined subroutine &$AUTOLOAD called" unless exists $messages{$1};
        my $element = $1;
        *$AUTOLOAD = sub { #Handles Decode_<message> calls
            my $xsum = Checksum($_[0]);
            my $hashref = $messages{$element}->unformat($_[0]);
            if($xsum != $hashref->{Checksum}) {
                warn "unmatched xsum for $element message - expected $xsum got ".$hashref->{Checksum} 
            }
            for( keys %{ $hashref } ) { #auto-apply to state
                $state{$_} = $hashref->{$_};
            }
            return $hashref;
        }
    } elsif($AUTOLOAD =~ /.*::(.*)/) {
        return if $1 eq "DESTROY";
        unless (exists $messages{$1}) {
            $AutoLoader::AUTOLOAD = $AUTOLOAD;
            goto &AutoLoader::AUTOLOAD;
        }
        my $element = $1;
        *$AUTOLOAD = sub { #Handles <message> calls, gives blank hash
            my $hashref = $messages{$element}->blank(); 
            for( keys %{ $hashref } ) { #auto-fill from state
                $hashref->{$_} = $state{$_} if exists $state{$_};
            }
            return $hashref;
        }
    }    goto &$AUTOLOAD;
}

1;

__END__

=item Get_Country_Name($country_code)
This function returns a string representing the country which corresponds to the numberic code given.
=cut
sub Get_Country_Name {
    my $code = shift;
    return $Country_Name[$code]      if $code <= 27;
    return $Country_Name[$code - 72] if $code <= 119;
    return "Undefined";
}
=item Get_Vehicle_Subtype($code)
This function returns a string representing the vehicle subtype which corresponds to the numberic code given.
=cut
sub Get_Vehicle_Subtype {
    return $Vehicle_Subtype{$_[0]} if exists $Vehicle_Subtype{$_[0]};
    return "Undefined";
}
=item Checksum($datagram)
This function computes a checksum on the given record as the addition of all elements in the record when interpreted as 4 byte unsigned integers.

The modulus constrains the result to a 4 byte integer. Perl might allow it to grow past that.
=cut
sub Checksum {
    my @data = unpack("I*",$_[0]);
    (List::Util::sum( @data ) - $data[-1]) % 0xFFFFFFFF;
}
=item Get_Vehicle_Name($code)
This function returns a string representing the vehicle which corresponds to the numberic code given.
=cut
sub Get_Vehicle_Name {
    return $Vehicle_Name[ $_[0] ] if exists $Vehicle_Name[ $_[0] ];
    return "Undefined";
}


=back

=head1 SEE ALSO

Parse::Binary::FixedFormat - Documentation for the objects returned.

=head1 COPYRIGHT AND LICENSE

Copyright 2009 Chris Johnson

This library is free software; you can redistribute it and/or modify it under the same terms as Perl itself.

=cut


