package Lingua::StanfordCoreNLP;

use strict;
#use warnings;

our $Jar_Path;

BEGIN {
	use Exporter ();
	our @ISA       = qw(Exporter);
	our @EXPORT    = ();
	our $VERSION   = '0.03';
	    $VERSION   = eval $VERSION;
	my ($pkg_path) = __FILE__ =~ /(.*)\.pm/;

	$Jar_Path = join ':', map { $pkg_path . '/' . $_ } qw(
		stanford-corenlp.jar
		stanford-corenlp-models.jar
		jgrapht.jar
		xom.jar
	);
}

sub new {
	die 'Lingua::StanfordCoreNLP cannot be instantiated. Use Lingua::StanfordCoreNLP::Pipeline.';
}

use Inline (
	Java            => 'DATA',
	CLASSPATH       => $Jar_Path,
	EXTRA_JAVA_ARGS => '-Xmx2000m',
 	AUTOSTUDY       => 1,
);

1;

__DATA__
__Java__
import java.io.*;
import java.util.*;

import edu.stanford.nlp.dcoref.*;
import edu.stanford.nlp.dcoref.CorefChain.*;
import edu.stanford.nlp.ling.*;
import edu.stanford.nlp.ling.CoreAnnotations.*;
import edu.stanford.nlp.ling.CorefCoreAnnotations.*;
import edu.stanford.nlp.pipeline.*;
import edu.stanford.nlp.trees.*;
import edu.stanford.nlp.trees.semgraph.*;
import edu.stanford.nlp.trees.semgraph.SemanticGraphCoreAnnotations.*;
import edu.stanford.nlp.util.*;

class Pipeline {
	private StanfordCoreNLP   pipeline = null;
	private boolean           bidirectionalCorefs = false;
	private PipedInputStream  stdoutStream  = null;
	private PipedInputStream  stderrStream  = null;

	public StanfordCoreNLP getPipeline() {
		return pipeline;
	}

	public Pipeline() {
		this(false, false);
	}

	public Pipeline(boolean silent) {
		this(silent, false);
	}

	public Pipeline(boolean silent, boolean bidirectionalCorefs) {
		if(silent)
			redirectStreams();
		this.bidirectionalCorefs = bidirectionalCorefs;
	}

	public Pipeline(boolean silent, boolean bidirectionalCorefs, long startingID) {
		this(silent, bidirectionalCorefs);

		PipelineItem.initializeCounters(startingID, Long.MIN_VALUE);
	}

	public String getAnnotatorLog() throws IOException {
		return readStream(stdoutStream) + readStream(stderrStream);
	}

	public void initPipeline() {
		Properties props = new Properties();
		props.put("annotators", "tokenize, ssplit, pos, parse, lemma, ner, dcoref");
		pipeline = new StanfordCoreNLP(props, false);
	}

	private String readStream(PipedInputStream stream) throws IOException {
		String output = "";
		if(stream != null) {
			if(stream.available() > 0) {
				byte errBytes[] = new byte[stream.available()];
				stderrStream.read(errBytes, 0, stream.available());
				output += new String(errBytes);
			}
		}
		return output;
	}

	private void redirectStreams() {
		try {
			stdoutStream = new PipedInputStream();
			stderrStream = new PipedInputStream();
			System.setOut(new PrintStream(new PipedOutputStream(stdoutStream)));
			System.setErr(new PrintStream(new PipedOutputStream(stderrStream)));
		} catch(Exception e) {
			System.err.println(e);
		}
	}

	public PipelineSentenceList process(String text) {
		if(pipeline == null)
			initPipeline();

		PipelineSentenceList outList  = new PipelineSentenceList();
		Annotation           document = pipeline.process(text);

		if(document == null)
			return null;

		for(CoreMap sentence: document.get(SentencesAnnotation.class)) {
			String                  str = sentence.get(TextAnnotation.class);
			PipelineTokenList       ptl = new PipelineTokenList();
			PipelineDependencyList  pel = new PipelineDependencyList();

			for(CoreLabel token: sentence.get(TokensAnnotation.class)) {
				String word = token.get(TextAnnotation.class);
				String pos  = token.get(PartOfSpeechAnnotation.class);
				String ner  = token.get(NamedEntityTagAnnotation.class);
				String lemma= token.get(LemmaAnnotation.class);

				ptl.add(new PipelineToken(word, pos, ner, lemma));
			}

			SemanticGraph dependencies = sentence.get(CollapsedCCProcessedDependenciesAnnotation.class);

			if(dependencies != null) {
				for(SemanticGraphEdge edge: dependencies.edgeList()) {
					//String govStr = edge.getGovernor().toString(IndexedWord.WORD_TAG_INDEX_FORMAT);
					//String depStr = edge.getDependent().toString(IndexedWord.WORD_TAG_INDEX_FORMAT);
					GrammaticalRelation rel = edge.getRelation();

					int govTokenIndex = edge.getGovernor().index()  - 1; //getIndexFromWTI(govStr);
					int depTokenIndex = edge.getDependent().index() - 1; //getIndexFromWTI(depStr);

					if(govTokenIndex >= 0 && depTokenIndex >= 0 &&
					   govTokenIndex < ptl.size() && depTokenIndex < ptl.size()
					) {
						pel.add(new PipelineDependency(
							ptl.get(govTokenIndex),
							ptl.get(depTokenIndex),
							govTokenIndex,
							depTokenIndex,
							rel
						));
					} else {
						System.err.println("Index of " + edge.toString() + " out of range!");
					}
				}
			}
			outList.add(new PipelineSentence(str, ptl, pel));
		}//for

		Map<Integer, CorefChain> graph = document.get(CorefChainAnnotation.class);

		if(graph != null) {
			for(CorefChain crc: graph.values()) {
				List<CorefMention> crms = crc.getCorefMentions();
				ArrayList<int[]> corefs = new ArrayList<int[]>();

				for(CorefMention crm: crms) {
					corefs.add(new int[]{crm.sentNum, crm.headIndex});
				}

				if(corefs.size() > 1) {
					int fromSentence = (corefs.get(0))[0] - 1;
					int fromHead     = (corefs.get(0))[1] - 1;
					int toSentence   = (corefs.get(1))[0] - 1;
					int toHead       = (corefs.get(1))[1] - 1;

					if(fromSentence >= 0 && toSentence >= 0 &&
					   fromHead >= 0 && toHead >= 0 &&
					   fromSentence < outList.size() && toSentence < outList.size() &&
					   fromHead < outList.get(fromSentence).getTokens().size() &&
					   toHead   < outList.get(toSentence).getTokens().size()
					) {
						outList.get(fromSentence).addCoreference(
							new PipelineCoreference(
								fromSentence, toSentence, fromHead, toHead,
								outList.get(fromSentence).getTokens().get(fromHead),
								outList.get(toSentence).getTokens().get(toHead)
							)
						);
						if(bidirectionalCorefs && toSentence != fromSentence) {
							outList.get(toSentence).addCoreference(
								new PipelineCoreference(
									toSentence, fromSentence, toHead, fromHead,
									outList.get(toSentence).getTokens().get(toHead),
									outList.get(fromSentence).getTokens().get(fromHead)
								)
							);
						}
					}//if(fromSentence
				}//if(corefs
			}//for
		}//if(graph

		return outList;
	}//process
}


abstract class PipelineItem {
	protected UUID   id    = null;
	protected String idStr = "";
	protected static long idLeast = Long.MIN_VALUE;
	protected static long idMost  = Long.MIN_VALUE;

	public void setIDFromString(String str) {
		id = UUID.nameUUIDFromBytes(str.getBytes());
	}

	public static void initializeCounters(long least, long most) {
		idLeast = least;
		idMost  = most;
	}

	public static void randomizeCounters() {
		UUID t = UUID.randomUUID();

		PipelineItem.initializeCounters(
			t.getLeastSignificantBits(),
			t.getMostSignificantBits()
		);
	}

	public static UUID generateID() {
		if(idLeast <= Long.MAX_VALUE)
			idLeast++;
		else if(idMost <= Long.MAX_VALUE)
			idMost++;
		else // if this happens, you've got big problems
			System.out.println("PipelineItem.generateID(): ran out of IDs!");

		return new UUID(idMost, idLeast);
	}

	public UUID getID() {
		if(id == null)
			id = PipelineItem.generateID();
		return id;
	}

	public String getIDString() {
		if(idStr == "")
			idStr = getID().toString();
		return idStr;
	}

	public boolean identicalTo(PipelineItem b) {
		return getID().equals(b.getID());
	}

	abstract public String toCompactString();
}


class PipelineCoreference extends PipelineItem {
	private int fromSentence;
	private int toSentence;
	private int fromHead;
	private int toHead;
	private PipelineToken fromToken;
	private PipelineToken toToken;

	public int getSourceSentence()        { return fromSentence; }
	public int getTargetSentence()        { return toSentence; }
	public int getSourceHead()            { return fromHead; }
	public int getTargetHead()            { return toHead; }
	public PipelineToken getSourceToken() { return fromToken; }
	public PipelineToken getTargetToken() { return toToken;   }

	public PipelineCoreference(
		int fromSentence, int toSentence, int fromHead, int toHead,
		PipelineToken fromToken, PipelineToken toToken
	) {
		this.fromSentence = fromSentence;
		this.toSentence   = toSentence;
		this.fromHead     = fromHead;
		this.toHead       = toHead;
		this.fromToken    = fromToken;
		this.toToken      = toToken;
		//this.setIDFromString(fromToken.getIDString() + toToken.getIDString());
	}

	public boolean equals(PipelineCoreference b) {
		if((
			fromToken.identicalTo(b.fromToken)
				&&
			toToken.identicalTo(b.toToken)
		) || (
			fromToken.identicalTo(b.toToken)
				&&
			toToken.identicalTo(b.fromToken)
		))
			return true;
		else
			return false;
	}

	public String toCompactString() {
		return fromToken.getWord() + "/" + fromSentence + ":" + fromHead + " <=> "
		     + toToken.getWord()   + "/" + toSentence   + ":" + toHead;
	}

	@Override public String toString() {
		return fromToken.toCompactString() + " [" + fromSentence + "," + fromHead + "] <=> " +
		       toToken.toCompactString()   + " [" + toSentence   + "," + toHead   + "]";
	}
}


class PipelineDependency extends PipelineItem {
	private PipelineToken governor;
	private PipelineToken dependent;
	private int           govIndex;
	private int           depIndex;
	private String        relation;
	private String        longRelation;

	public PipelineToken getGovernor()  { return governor;  }
	public PipelineToken getDependent() { return dependent; }
	public int getGovernorIndex()       { return govIndex; }
	public int getDependentIndex()      { return depIndex; }
	public String getRelation()         { return relation; }
	public String getLongRelation()     { return longRelation; }

	public PipelineDependency(
		PipelineToken governor,
		PipelineToken dependent,
		int           govIndex,
		int           depIndex,
		GrammaticalRelation relation
	) {
		this.governor     = governor;
		this.dependent    = dependent;
		this.govIndex     = govIndex;
		this.depIndex     = depIndex;
		this.relation     = relation.toString();
		this.longRelation = relation.getLongName();
	}

	public String toCompactString() { return toCompactString(false); }

	public String toCompactString(boolean includeIndices) {
		return relation + "("
		     + governor.getWord()  + (includeIndices ? "-" + govIndex : "")
		     + ", "
		     + dependent.getWord() + (includeIndices ? "-" + depIndex : "")
		     + ")";
	}

	public String toString(boolean includeIndices) {
		return toCompactString(includeIndices) + " [" + longRelation + "]";
	}

	@Override public String toString() {
		return toString(true);
	}
}


class PipelineSentence extends PipelineItem {
	private String                  sentence;
	private PipelineTokenList       tokens;
	private PipelineDependencyList  dependencies;
	private PipelineCoreferenceList coreferences;

	public String                  getSentence()     { return sentence; }
	public PipelineTokenList       getTokens()       { return tokens; }
	public PipelineDependencyList  getDependencies() { return dependencies; }
	public PipelineCoreferenceList getCoreferences() { return coreferences; }

	public PipelineSentence() {
		sentence     = "";
		tokens       = new PipelineTokenList();
		dependencies = new PipelineDependencyList();
		coreferences = new PipelineCoreferenceList();
	}

	public PipelineSentence(
		String                 sentence,
		PipelineTokenList      tokens,
		PipelineDependencyList dependencies
	) {
		this.sentence     = sentence;
		this.tokens       = tokens;
		this.dependencies = dependencies;
		this.coreferences = new PipelineCoreferenceList();
	}

	public void addCoreference(PipelineCoreference cr) {
		coreferences.add(cr);
	}

	public String toCompactString() { return join("\n"); }

	@Override public String toString() {
		return join("\n\n");
	}

	public String join(String sep) {
		return sentence + sep + tokens.toString() + sep + dependencies.toString() + sep
					+ coreferences.toString();
	}
}


class PipelineToken extends PipelineItem {
	private String word;
	private String posTag;
	private String nerTag;
	private String lemma;

	public String getWord()   { return word; }
	public String getPOSTag() { return posTag; }
	public String getNERTag() { return nerTag; }
	public String getLemma()  { return lemma; }

	public PipelineToken(String word, String posTag, String nerTag, String lemma) {
		this.word   = word;
		this.posTag = posTag;
		this.nerTag = nerTag;
		this.lemma  = lemma;
	}

	@Override public String toString() {
		return word + "/" + lemma + "/" + posTag + "/" + nerTag;
	}

	public String toCompactString() { return toCompactString(false); }

	public String toCompactString(boolean lemmaize) {
		return (lemmaize ? word : lemma) + "/" + posTag;
	}
}

class PipelineList<T> extends ArrayList<T> {
	public String joinList(String sep) {
		String output = "";
		for(T elem: this) {
			output += (output.length() > 0 ? sep : "") + elem.toString();
		}
		return output;
	}

	public String joinListCompact(String sep) {
		String output = "";
		for(T elem: this) {
			output += (output.length() > 0 ? sep : "") + ((PipelineItem)elem).toCompactString();
		}
		return output;
	}

	public HashMap<String,T> toHashMap() {
		HashMap<String,T> hm = new HashMap<String,T>();

		for(T elem: this) {
			hm.put( ((PipelineItem)elem).getIDString(), elem);
		}

		return hm;
	}

	public String toCompactString()    { return joinListCompact("\n"); }
	@Override public String toString() { return joinList("\n"); }
}

class PipelineCoreferenceList extends PipelineList<PipelineCoreference> { }

class PipelineDependencyList extends PipelineList<PipelineDependency> { }

class PipelineSentenceList extends PipelineList<PipelineSentence> { }

class PipelineTokenList extends PipelineList<PipelineToken>  {
	@Override public String toCompactString() {
		return joinListCompact(" ");
	}

	@Override public String toString() {
		return joinList(" ");
	}
}
__END__

=head1 NAME

Lingua::StanfordCoreNLP - A Perl interface to Stanford's CoreNLP tool set.

=head1 SYNOPSIS
   
 # Note that Lingua::StanfordCoreNLP can't be instantiated.
 use Lingua::StanfordCoreNLP;

 # Create a new NLP pipeline (don't silence messages, do make corefs bidirectional)
 my $pipeline = new Lingua::StanfordCoreNLP::Pipeline(0, 1);

 # Process text
 # (Will output lots of debug info from the Java classes to STDERR.)
 my $result = $pipeline->process(
    'Jane looked at the IBM computer. She turned it off.'
 );

 my @seen_corefs;

 # Print results
 for my $sentence (@{$result->toArray}) {
    print "\n[Sentence ID: ", $sentence->getIDString, "]:\n";
    print "Original sentence:\n\t", $sentence->getSentence, "\n";

    print "Tagged text:\n";
    for my $token (@{$sentence->getTokens->toArray}) {
       printf "\t%s/%s/%s [%s]\n",
              $token->getWord,
              $token->getPOSTag,
              $token->getNERTag,
              $token->getLemma;
    }

    print "Dependencies:\n";
    for my $dep (@{$sentence->getDependencies->toArray}) {
       printf "\t%s(%s-%d, %s-%d) [%s]\n",
              $dep->getRelation,
              $dep->getGovernor->getWord,
              $dep->getGovernorIndex,
              $dep->getDependent->getWord,
              $dep->getDependentIndex,
              $dep->getLongRelation;
    }

    print "Coreferences:\n";
    for my $coref (@{$sentence->getCoreferences->toArray}) {
       printf "\t%s [%d, %d] <=> %s [%d, %d]\n",
              $coref->getSourceToken->getWord,
              $coref->getSourceSentence,
              $coref->getSourceHead,
              $coref->getTargetToken->getWord,
              $coref->getTargetSentence,
              $coref->getTargetHead;

       print "\t\t(Duplicate)\n"
          if(grep { $_->equals($coref) } @seen_corefs);

       push @seen_corefs, $coref;
    }
 }


=head1 DESCRIPTION

This module implements a C<StanfordCoreNLP> pipeline for annotating
text with part-of-speech tags, dependencies, lemmas, named-entity tags, and coreferences.

(Note that the archive contains the CoreNLP annotation models, which is why
it's so darn big.)


=head1 INSTALLATION

The following should do the job:

 $ perl Build.PL
 $ ./Build test
 $ sudo ./Build install


=head1 PREREQUISITES

Lingua::StanfordCoreNLP consists mainly of Java code, and thus needs L<Inline::Java> installed
to function.


=head1 EXPORTED CLASSES

Lingua::StanfordCoreNLP exports the following Java-classes via L<Inline::Java>:


=head2 Lingua::StanfordCoreNLP::Pipeline

The main interface to C<StanfordCoreNLP>. This class is the only one you
should need to instantiate yourself.

=over

=item new

=item new($silent)

=item new($silent, $bidirectionalCorefs)

Creates a new C<Lingua::StanfordCoreNLP::Pipeline> object. The optional
boolean parameter C<$silent> silences the output from annotators if true,
while the optional parameter C<$bidirectionalCorefs> makes coreferences bidirectional;
that is to say, the coreference is added to both the source and the target
sentence of all coreferences (if the source and target sentence are different).
C<$silent> and C<$bidirectionalCorefs> default to false.

=item getAnnotatorLog

If the pipeline was created to be C<$silent>, return logged messages as a string.
Otherwise, or if no output has been logged, returns an empty string.

=item getPipeline

Returns a reference to the C<StanfordCoreNLP> pipeline used for annotation.
You probably won't want to touch this.

=item process($str)

Process a string. Returns a C<Lingua::StanfordCoreNLP::PipelineSentenceList>.

=back


=head2 Lingua::StanfordCoreNLP::PipelineItem

Abstract superclass of C<Pipeline{Coreference,Dependency,Sentence,Token}>. Contains ID
and methods for getting and comparing it.

=over

=item getID

Returns a C<java.util.UUID> object which represents the item's ID.

=item getIDString

Returns the ID as a string.

=item identicalTo($b)

Returns true if C<$b> has an identical ID to this item.

=back


=head2 Lingua::StanfordCoreNLP::PipelineCoreference

An object representing a coreference between head-word W1 in sentence S1 and head-word W2 in sentence S2.
Note that both sentences and words are zero-indexed, unlike the default outputs of Stanford's tools.

=over

=item getSourceSentence

Index of sentence S1.

=item getTargetSentence

Index of sentence S2.

=item getSourceHead

Index of word W1 (in S1).

=item getTargetHead

Index of word W2 (in S2).

=item getSourceToken

The C<Lingua::StanfordCoreNLP::PipelineToken> representing W1.

=item getTargetToken

The C<Lingua::StanfordCoreNLP::PipelineToken> representing W2.

=item equals($b)

Returns true if this C<PipelineCoreference> matches C<$b> --- if
their C<getSourceToken> and C<getTargetToken> have the same ID.
Note that it returns true even if the orders of the
coreferences are reversed (if C<< $a->getSourceToken->getID == $b->getTargetToken->getID >>
and C<< $a->getTargetToken->getID == $b->getSourceToken->getID >>).

=item toCompactString

A compact String representation of the coreference ---
"Word/Sentence:Head E<lt>=E<gt> Word/Sentence:Head".

=item toString

A String representation of the coreference ---
"Word/POS-tag [sentence, head] E<lt>=E<gt> Word/POS-tag [sentence, head]".

=back


=head2 Lingua::StanfordCoreNLP::PipelineDependency

Represents a dependency in the Stanford Typed Dependency format.
For example, in the fragment "Walk hard", "Walk" is the governor and "hard"
is the dependent in the relationship "advmod" ("hard" is an adverbial modifier
of "Walk").

=over

=item getGovernor

The governor in the relation as a C<Lingua::StanfordCoreNLP::PipelineToken>.

=item getGovernorIndex

The index of the governor within the sentence.

=item getDependent

The dependent in the relation as a C<Lingua::StanfordCoreNLP::PipelineToken>.

=item getDependentIndex

The index of the dependent within the sentence.

=item getRelation

Short name of the relation.

=item getLongRelation

Long description of the relation.

=item toCompactString

=item toCompactString($includeIndices)

=item toString

=item toString($includeIndices)

Returns a String representation of the dependency --- "relation(governor-N, dependent-N) [description]".
C<toCompactString> does not include description. The optional parameter C<$includeIndices> controls
whether governor and dependent indices are included, and defaults to true.
(Note that unlike those of, e.g., the Stanford Parser, these indices start at zero, not one.)

=back


=head2 Lingua::StanfordCoreNLP::PipelineSentence

An annotated sentence, containing the sentence itself, its dependencies,
pos- and ner-tagged tokens, and coreferences.

=over

=item getSentence

Returns a string containing the original sentence

=item getTokens

A C<Lingua::StanfordCoreNLP::PipelineTokenList> containing the POS- and
NER-tagged and lemmaized tokens of the sentence.

=item getDependencies

A C<Lingua::StanfordCoreNLP::PipelineDependencyList> containing the dependencies
found in the sentence.

=item getCoreferences

A C<Lingua::StanfordCoreNLP::PipelineCoreferenceList> of the coreferences between
this and other sentences.

=item toCompactString

=item toString

A String representation of the sentence, its coreferences, dependencies, and tokens.
C<toCompactString> separates fields by "\n", whereas C<toString> separates them by
"\n\n".

=back


=head2 Lingua::StanfordCoreNLP::PipelineToken

A token, with POS- and NER-tag and lemma.

=over

=item getWord

The textual representation of the token (i.e. the word).

=item getPOSTag

The token's Part-of-Speech tag.

=item getNERTag

The token's Named-Entity tag.

=item getLemma

The lemma of the the token.

=item toCompactString

=item toCompactString($lemmaize)

A compact String representation of the token --- "word/POS-tag". If the
optional argument C<$lemmaize> is true, returns "lemma/POS-tag".

=item toString

A String representation of the token --- "word/POS-tag/NER-tag [lemma]".

=back


=head2 Lingua::StanfordCoreNLP::PipelineList

=head2 Lingua::StanfordCoreNLP::PipelineCoreferenceList

=head2 Lingua::StanfordCoreNLP::PipelineDependencyList

=head2 Lingua::StanfordCoreNLP::PipelineSentenceList

=head2 Lingua::StanfordCoreNLP::PipelineTokenList

C<Lingua::StanfordCoreNLP::PipelineList> is a generic list class which
extends C<java.Util.ArrayList>. It is in turn extended by
C<Pipeline{Coreference,Dependency,Sentence,Token}List> (which are the
list-types that C<Pipeline> returns). Note that all lists are zero-indexed.

=over

=item joinList($sep)

=item joinListCompact($sep)

Returns a string containing the output of either the C<toString> or
C<toCompactString> methods of the elements in C<PipelineList>, separated
by C<$sep>.

=item toArray

Return the elements of the list as an array-reference.

=item toHashMap

Return the list as a C<< java.util.HashMap<String,PipelineItem> >>, with
items' stringified ID:s as keys.

=item toCompactString

=item toString

Returns the elements of the C<PipelineList> as a string containing the output
of either their C<toCompactString> or C<toString> methods, separated by the
default separator (which is "\n" for all lists except C<PipelineTokenList>
which uses " ").

=back


=head1 TODO

=over

=item *

Custom annotator-combinations, so you won't have to load up six different annotator
models just to POSTag som text.

=back


=head1 REQUESTS & BUGS

Mail any bug-reports or feature-requests to E<lt>StanfordCoreNLP@fivebyfive.beE<gt>.


=head1 AUTHORS

Kalle RE<auml>isE<auml>nen E<lt>kal@cpan.orgE<gt>.


=head1 COPYRIGHT

=head2 Lingua::StanfordCoreNLP (Perl bindings)

Copyright E<copy> 2011 Kalle RE<auml>isE<auml>nen.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU Affero General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Affero General Public License for more details.

You should have received a copy of the GNU Affero General Public License
along with this program.  If not, see L<http://www.gnu.org/licenses/>.


=head2 Stanford CoreNLP tool set

Copyright E<copy> 2010-2011 The Board of Trustees of The Leland Stanford
Junior University.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, see L<http://www.gnu.org/licenses/>.


=head1 SEE ALSO

L<http://nlp.stanford.edu/software/corenlp.shtml>,
L<Text::NLP::Stanford::EntityExtract>,
L<NLP::StanfordParser>,
L<Inline::Java>.

=cut
