package Net::CDP::Packet;

#
# $Id: Packet.pm,v 1.3 2004/09/02 04:25:04 mchapman Exp $
#

use 5.00503;
use strict;
use Carp::Clan qw(^Net::CDP);

use vars qw($VERSION @ISA $AUTOLOAD @EXPORT @EXPORT_OK %EXPORT_TAGS);

$VERSION = (qw$Revision: 1.3 $)[1];;

require Exporter;
@ISA = qw(Exporter);

my @EXPORT_CAPS = qw(
	CDP_CAP_ROUTER CDP_CAP_TRANSPARENT_BRIDGE CDP_CAP_SOURCE_BRIDGE
	CDP_CAP_SWITCH CDP_CAP_HOST CDP_CAP_IGMP CDP_CAP_REPEATER
);

@EXPORT = qw();
@EXPORT_OK = (@EXPORT_CAPS, );
%EXPORT_TAGS = (
	caps => [ @EXPORT_CAPS, ],
);

sub AUTOLOAD {
	my $constname;
	($constname = $AUTOLOAD) =~ s/.*:://;
	croak '&Net::CDP::constant not defined' if $constname eq 'constant';
	my ($error, $val) = Net::CDP::Constants::constant($constname);
	croak $error if $error;

	no strict 'refs';
	*$AUTOLOAD = sub { $val };
	goto &$AUTOLOAD;
}

=head1 NAME

Net::CDP::Packet - Cisco Discovery Protocol (CDP) packet

=head1 SYNOPSIS

  use Net::CDP::Packet qw(:caps);
  
  # Constructors
  $packet = new Net::CDP::Packet;
  $packet = new Net::CDP::Packet($cdp);
  $cloned = clone $packet;
  
  # Object methods
  $version                     = $packet->version;
  $ttl                         = $packet->ttl;
  $checksum                    = $packet->checksum;
  $device                      = $packet->device;
  @addresses                   = $packet->addresses;
  $port                        = $packet->port;
  $capabilities                = $packet->capabilities;
  $ios_version                 = $packet->ios_version;
  $platform                    = $packet->platform;
  @ip_prefixes                 = $packet->ip_prefixes;
  $vtp_management_domain       = $packet->vtp_management_domain;
  $native_vlan                 = $packet->native_vlan;
  $duplex                      = $packet->duplex;
  ($voice_vlan, $appliance_id) = $packet->voice_vlan;
  $mtu                         = $packet->mtu;
  $trusted                     = $packet->trusted;
  $untrusted_cos               = $packet->untrusted_cos

=head1 DESCRIPTION

A Net::CDP::Packet object represents a single CDP packet.

=head1 CONSTRUCTORS

=over

=item B<new>

    $packet = new Net::CDP::Packet()
    $packet = new Net::CDP::Packet($cdp)

Returns a new Net::CDP::Packet object.

The returned object will have the following fields set:

    Version: 1
    TTL: 180
    Device ID: `hostname`
    Capabilities: CDP_CAP_HOST
    IOS Version: `uname -a`
    Platform: `uname -s`

If supplied, C<$cdp> must be a L<Net::CDP> object. C<new> will use this to
generate the following fields:

    Port ID: $cdp->port()
    Addresses: $cdp->addresses()
    Duplex

=item B<clone>

    $cloned = clone $packet;

Returns a deep copy of the supplied Net::CDP::Packet object.

=back

=head1 OBJECT METHODS

=over

=item B<version>

    $version = $packet->version()

Returns this packet's Version field. For packets generated by L</new>, this
is always 1.

=item B<ttl>

    $ttl = $packet->ttl()
    $ttl = $packet->ttl($new_ttl)

Returns this packet's Time-to-live field. If C<$new_ttl> is supplied and
positive, the field will be updated first.

The Time-to-live field specifies how long a receiving CDP implementation may
cache the data in this packet.

=item B<checksum>

    $checksum = $packet->checksum()

Returns this packet's Checksum field. This checksum will be automatically
updated when other fields are modified.

=item B<device>

    $device = $packet->device()
    $device = $packet->device($new_device)

Returns this packet's Device ID field if present, C<undef> otherwise. If
C<$new_device> is supplied, the field will be updated first. If C<$new_device>
is undefined, the Device ID field is removed from the packet.

For packets received from other devices, the Device ID field specifies the name
of the device that generated the packet.

=item B<addresses>

    @addresses = $packet->addresses()
    @addresses = $packet->addresses($new_addresses)

Returns this packet's Addresses field as a list of L<Net::CDP::Address>
objects. In scalar context, the number of addresses is returned, or C<undef> if
no Addresses field is present.

If C<$new_addresses> is supplied, the field will be updated first. If
C<$new_addresses> is C<undef>, the Addresses field is removed from the packet.
Otherwise C<$new_addresses> must be a reference to an array of
L<Net::CDP::Address> objects. The array may be empty.

For packets received from other devices, the Addresses field specifies the
addresses of the network port upon which the packet was sent.

=item B<port>

    $port = $packet->port()
    $port = $packet->port($new_port)

Returns this packet's Port ID field if present, C<undef> otherwise. If
C<$new_port> is supplied, the field will be updated first. If C<$new_device> is
undefined, the Port ID field is removed from the packet.

For packets received from other devices, the Port ID field specifies the network
port upon which the packet was sent.

=item B<capabilities>

    $capabilities = $packet->capabilities()
    $capabilities = $packet->capabilities($new_capabilities)

Returns this packet's Capabilities field if present, C<undef> otherwise. If
$new_capabilities is supplied, the field will be updated first. If
$new_capabilities is undefined, the Capabilities field is removed from the
packet.

The Capabilities field is a bitmask specifying one or more of the following
constants:

    CDP_CAP_ROUTER
    CDP_CAP_TRANSPARENT_BRIDGE
    CDP_CAP_SOURCE_BRIDGE
    CDP_CAP_SWITCH
    CDP_CAP_HOST
    CDP_CAP_IGMP
    CDP_CAP_REPEATER

These constants can be exported using the tag C<:caps>. See L<Exporter>.

=item B<ios_version>

    $ios_version = $packet->ios_version()
    $ios_version = $packet->ios_version($new_ios_version)

Returns this packet's IOS Version field if present, C<undef> otherwise. If
$new_ios_version is supplied, the field will be updated first. If
$new_ios_version is undefined, the IOS Version field is removed from the packet.

=item B<platform>

    $platform = $packet->platform()
    $platform = $packet->platform($new_platform)

Returns this packet's Platform field if present, C<undef> otherwise. If
$new_platform is supplied, the field will be updated first. If $new_platform is
undefined, the Platform field is removed from the packet.

=item B<ip_prefixes>

    @ip_prefixes = $packet->ip_prefixes()
    @ip_prefixes = $packet->ip_prefixes($new_ip_prefixes)

Returns this packet's IP Prefixes field as a list of L<Net::CDP::IPPrefix>
objects.  In scalar context, the number of prefixes is returned, or C<undef> if
no IP Prefixes field is present.

If C<$new_ip_prefix> is supplied, the field will be updated first. If
C<$new_ip_prefix> is C<undef>, the IP Prefixes field is removed from the packet.
Otherwise C<$new_ip_prefix> must be a reference to an array of
L<Net::CDP::IPPrefix> objects. The array may be empty.

=item B<vtp_management_domain>

    $vtp_management_domain = $packet->vtp_management_domain()
    $vtp_management_domain =
        $packet->vtp_management_domain($new_vtp_management_domain)

Returns this packet's VTP Management Domain field if present, C<undef>
otherwise. If C<$new_vtp_management_domain> is supplied, the field will be
updated first. If C<$new_vtp_management_domain> is undefined, the VTP ManagementDomain field is removed from the packet.

=item B<native_vlan>

    $native_vlan = $packet->native_vlan()
    $native_vlan = $packet->native_vlan($new_native_vlan)

Returns this packet's Native VLAN field if present, C<undef> otherwise. If
C<$new_native_vlan> is supplied, the field will be updated first. If
C<$new_native_vlan> is undefined, the Native VLAN field is removed from the
packet.

=item B<duplex>

    $duplex = $packet->duplex()
    $duplex = $packet->duplex($new_duplex)

Returns this packet's Duplex field if present, C<undef> otherwise. If
C<$new_duplex> is supplied, the field will be updated first. If C<$new_duplex>
is undefined, the Duplex field is removed from the packet.

The Duplex field contains a boolean value. If it is true, the port supplied in
the Port ID field supports full-duplex communication. Otherwise, only
half-duplex communication is supported.

=item B<voice_vlan>

    $voice_vlan                  = $packet->voice_vlan()
    ($voice_vlan, $appliance_id) = $packet->voice_vlan()
    ($voice_vlan, $appliance_id)
        = $packet->voice_vlan($new_voice_vlan)
    ($voice_vlan, $appliance_id)
        = $packet->voice_vlan($new_voice_vlan, $new_appliance_id)

Returns the Voice VLAN from this packet's Appliance VLAN-ID field if present,
C<undef> otherwise. In list context, the Appliance ID is also returned.

If C<$new_voice_vlan> or C<$new_appliance_id> is supplied, the field will be
updated first:

=over

=item $packet->voice_vlan($new_voice_vlan)

Updates the Voice VLAN only. If no Appliance VLAN-ID field currently exists in
the packet, the Appliance ID is set to 1.

=item $packet->voice_vlan(undef, $new_appliance_id)

Updates the Appliance ID only. If no Appliance VLAN-ID field currently exists
in the packet, the method croaks.

=item $packet->voice_vlan($new_voice_vlan, $new_appliance_id)

Updates both the Voice VLAN and the Appliance ID.

=item $packet->voice_vlan(undef) or $packet->voice_vlan(undef, undef)

Removes the Appliance VLAN-ID field completely.

=back

=item B<mtu>

    $mtu = $packet->mtu()
    $mtu = $packet->mtu($new_mtu)

Returns this packet's MTU field if present, C<undef> otherwise. If C<$new_mtu>
is supplied, the field will be updated first. If C<$new_mtu> is undefined, the
MTU field is removed from the packet.

=item B<trusted>

    $trusted = $packet->trusted()
    $trusted = $packet->trusted($new_trusted)

Returns this packet's Extended Trust field if present, C<undef> otherwise.
If C<$new_trusted> is supplied, the field will be updated first. If
C<$new_trusted> is undefined, the Extended Trust field is removed from the
packet.

The Extended Trust field contains a boolean value. If it is true, the port
trusts the CoS (Class of Service) values in incoming packets. Otherwise, the
port will assign its own CoS value to the traffic. See also the
L</untrusted_cos> method.

=item B<untrusted_cos>

    $untrusted_cos = $packet->untrusted_cos()
    $untrusted_cos = $packet->untrusted_cos($new_untrusted_cos)

Returns this packet's CoS for Untrusted Ports field if present, C<undef>
otherwise. If C<$new_untrusted_cos> is supplied, the field will be updated
first. If C<$new_untrusted_cos> is undefined, the field is removed from the
packet.

The CoS for Untrusted Ports field contains the CoS (Class of Service) that will
be applied when the port does not trust the CoS of incoming packets. See also
the L</trusted> method.

=back

=head1 SEE ALSO

L<Net::CDP>, L<Net::CDP::Address>, L<Net::CDP::IPPrefix>

=head1 AUTHOR

Michael Chapman, E<lt>cpan@very.puzzling.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2004 by Michael Chapman

libcdp is released under the terms and conditions of the GNU Library General
Public License version 2. Net::CDP may be redistributed and/or modified under
the same terms as Perl itself.

=cut

1;
