package Class::Type::Enum;
# ABSTRACT: Build Enum-like classes
$Class::Type::Enum::VERSION = '0.001';

use strict;
use warnings;

use Function::Parameters ':strict';
use List::Util ();
use Scalar::Util qw(blessed);
use Class::Method::Modifiers qw(install_modifier);

use namespace::clean;

use overload (
  '""'     => 'stringify',
  '0+'     => 'numify',
  fallback => 1,
);



fun import ($class, %params) {
  # import is inherited, but we don't want to do all this to everything that
  # uses a subclass of Class::Type::Enum.
  return unless $class eq __PACKAGE__;
  # If there's a use case for it, we can still allow extending CTE subclasses.

  my $target = caller;

  my %values;

  if (ref $params{values} eq 'ARRAY') {
    my $i = $params{init} // 0;

    %values = map {
      $_ => ($params{bits} ? 2**($i++) : $i++)
    } @{$params{values}};
  }
  elsif (ref $params{values} eq 'HASH') {
    %values = %{$params{values}};
  }
  else {
    die "Enum values must be provided either as an array or hash ref.";
  }

  ## the bits that are installed into the target class, plus @ISA
  {
    no strict 'refs';
    push @{"${target}::ISA"}, $class;
  }
  install_modifier $target, 'fresh', values_ord => sub { \%values };
  install_modifier $target, 'fresh', ord_values => sub { +{ reverse(%values) } };

  install_modifier $target, 'fresh', values => method {
    my $ord = $self->values_ord;
    [ sort { $ord->{$a} <=> $ord->{$b} } keys %values ];
  };

  for my $value (keys %values) {
    install_modifier $target, 'fresh', "is_$value" => method { $self->is($value) };
  }
}



method new ($class: $value) {
  $class->inflate_value($value);
}


method inflate_value ($class: $value) {
  bless {
    ord => $class->values_ord->{$value}
        // die "Value [$value] is not valid for enum $class"
  }, $class;
}


method inflate ($class: $ord) {
  die "Ordinal $ord is not valid for enum $class"
    if !exists $class->ord_values->{$ord};
  bless { ord => $ord }, $class;
}


method get_test ($class:) {
  return fun ($value) {
    exists($class->values_ord->{$value})
      or die "Value [$value] is not valid for enum $class"
  }
}



method test ($class: $value) {
  $class->get_test->($value)
}



method get_coerce ($class:) {
  return fun ($value) {
    eval { $value->isa($class) }
      ? $value
      : $class->new($value);
  }
}



method is ($value) {
  $self->{ord} == ($self->values_ord->{$value} // die "Value [$value] is not valid for enum ". blessed($self))
}



method stringify {
  $self->ord_values->{$self->{ord}};
}


method numify {
  $self->{ord}
}



method any (@cases) {
  List::Util::any { $self->is($_) } @cases;
}


method none (@cases) {
  List::Util::none { $self->is($_) } @cases;
}



1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Class::Type::Enum - Build Enum-like classes

=head1 VERSION

version 0.001

=head1 SYNOPSIS

  package Toast::Status {
    use Class::Type::Enum values => ['bread', 'toasting', 'toast', 'burnt'];
  }

  package Toast {
    use Moo;

    has status => (
      is     => 'rw',
      isa    => Toast::Status->get_test,
      coerce => Toast::Status->get_coerce,
    );
  }

  my @toast = map { Toast->new(status => $_) } qw( toast burnt bread bread toasting toast );

  my @trashcan = grep { $_->status->is_burnt } @toast;
  my @plate    = grep { $_->status->is_toast } @toast;

  my $ready_status   = Toast::Status->new('toast');
  my @eventual_toast = grep { $_->status < $ready_status } @toast;

  # or:

  @eventual_toast = grep { $_->status->none('toast', 'burnt') } @toast;

=head1 DESCRIPTION

Class::Type::Enum is a class builder for type-like classes to represent your
enumerated values.  In particular, it was built to scratch an itch with
L<DBIx::Class> value inflation.

I wouldn't consider the interface stable yet; I'd love feedback on this dist.

When C<use>ing Class::Type::Enum:

=over 4

=item *

Required:

=over 4

=item values => [@symbols]

The list of symbolic values in your enum, in ascending order if relevant.

=back

=item *

Optional

=over 4

=item init => $integer // 0

If provided, the ordinal values of your enum will begin with the init value.

=item bits => $bool // !!0

If true, each ordinal will be an increasing power of two, rather than
increments.  That is, with this set, your enum's ordinal values will be 1, 2,
4, 8, 16, ... which can be handy for bit fields.

=back

=back

=head1 METHODS

=head2 $class->import(values => [], ...)

Sets up the consuming class as a subclass of Class::Type::Enum and installs
functions that are unique to the class.

=head2 $class->new($value)

Your basic constructor, expects only a value corresponding to a symbol in the
enum type.

=head2 $class->inflate_value($value)

Does the actual work of L<$class-E<gt>new($value)>, also used when inflating values for
L<DBIx::Class::InflateColumn::ClassTypeEnum>.

=head2 $class->inflate($ord)

Used when inflating ordinal values for
L<DBIx::Class::InflateColumn::ClassTypeEnum> or if you need to work with
ordinals directly.

=head2 $class->values_ord

Returns a hashref with symbols as keys and ordinals as values.

=head2 $class->ord_values

Returns a hashref with ordinals as keys and symbols as values.

=head2 $class->values

Returns an arrayref of valid symbolic values in order.

=head2 $class->get_test

Returns a function which either returns true if it's passed a valid value for
the enum, or throws an exception.

=head2 $class->test($value)

A helper for directly using L<$class-E<gt>get_test>.

  Toast::Status->test('deleted')   # throws an exception

=head2 $class->get_coerce

Returns a function which returns an enum if given an enum, or tries to create an enum from the given value using L<$class-E<gt>new($value)>.

TODO: test and coerce don't work with ordinals

=head2 $o->is($value)

Given a test value, returns true or false if the enum instance's value is equal
to the test value.

An exception is thrown if an invalid test value is given.

=head2 $o->is_$value

Shortcut for L<$o-E<gt>is($value)>

=head2 $o->stringify

Returns the symbolic value.

=head2 $o->numify

Returns the ordinal value.

=head2 $o->any(@cases)

True if C<$o-E<gt>is(..)> for any of the given cases.

=head2 $o->none(@cases)

True if C<$o-E<gt>is(..)> for none of the given cases.

=head1 SEE ALSO

=over 4

=item *

L<Object::Enum>

=item *

L<Class::Enum>

=item *

L<Enumeration>

=back

=head1 AUTHOR

Meredith Howard <mhoward@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Meredith Howard.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
