package Spp::Tools;

=head1 NAME

Spp::Tools - The perl interface for Spp

=head1 VERSION

Version 0.01

=cut

our $VERSION = '0.01';

=head1 SYNOPSIS

Tools gather some small reused function

    use Spp::Tools;

    my $first_element = first([1,2,3]);
    # 1

=head1 EXPORT

add_exprs all_is_array all_is_hash all_is_int all_is_str
all_is_sym apply_char array_index bool error fill_array
first format get_atoms_type get_token_name in is_array
is_atom is_atoms is_case is_dot is_else is_elsif is_fail
is_false is_hash is_if is_in is_int is_lambda is_list
is_match is_nil is_perl_array is_perl_func is_perl_hash
is_perl_int is_perl_str is_ref is_rule is_same is_str
is_sym is_context is_true is_when len load_file perl_fill
perl_join perl_max perl_split perl_zip perl_substr read_file
rest second see sublist tail to_str trim type uuid value
write_file zip

=cut

use Exporter;
our @ISA = qw(Exporter);
our @EXPORT = qw(
add_exprs
all_is_array
all_is_hash
all_is_int
all_is_str
all_is_sym
apply_char
array_index
bool
error
fill_array
first
format
get_atoms_type
get_token_name
in
is_array
is_atom
is_atoms
is_case
is_dot
is_else
is_elsif
is_fail
is_false
is_hash
is_if
is_in
is_int
is_lambda
is_list
is_match
is_nil
is_perl_array
is_perl_func
is_perl_hash
is_perl_int
is_perl_str
is_ref
is_rule
is_same
is_str
is_sym
is_context
is_true
is_when
len
load_file
perl_fill
perl_join
perl_max
perl_split
perl_zip
perl_substr
read_file
rest
second
see
sublist
tail
to_str
trim
type
uuid
value
write_file
zip
);

use 5.020;
use Carp qw(croak);
use JSON qw(encode_json decode_json);
use List::Util qw(max);
use experimental qw(switch autoderef);
use List::MoreUtils qw(pairwise firstidx);

sub error { return croak(@_) }

sub bool {
  my $x = shift;
  return ['true'] if $x;
  return ['false'];
}

=head2 uuid
  
   say uuid(); # => 0.693987008257867

=cut

sub uuid { return scalar(rand()) }

=head2 is_perl_str

    say 'aa is str' if is_perl_str('aa');
    say '[1,2] not str' unless is_perl_str([1,2]);

=cut

sub is_perl_str {
   my $x = shift;
   return 1 if ref($x) eq ref('');
   return 0;
}

# say '[1,2] is array' if is_perl_array([1,2]);
# say '"aa" is not array' unless is_perl_array('aa');
sub is_perl_array {
   my $x = shift;
   return 1 if ref($x) eq ref([]);
   return 0;
}
#say '{} is perl hash' if is_hash({});
#say '[] is not perl hash' unless is_hash([]);
#say '"aa" is not perl hash' unless is_hash('aa');
sub is_perl_hash {
   my $x = shift;
   return 1 if ref($x) eq ref({});
   return 0;
}

#for my $x (1, '1', 'a', [], {}, //) {
#  if (is_int($x)) { say "$x is int"; next }
#  if (is_str($x)) { say "'$x' is str"; next }
#  if (is_perl_array($x)) {
#    say to_str($x) . " is array"; next;
#  }
#  if (is_hash($x)) { say to_str($x) . " is hash"; next }
#  if (is_perl_regex($x)) { say to_str($x) . " is regex"; next }
#  say "could not known $x type";
#}

sub is_perl_int {
  my $x = shift;
  if (is_perl_str($x)) {
    return 0 if $x ^ $x;
    return 0 if $x eq '';
    return 1;
  }
  return 0;
}

sub is_perl_func {
  my $x = shift;
  return 1 if ref($x) eq ref(sub {});
  return 0;
}

sub to_str {
  my $data = shift;
  return $data if is_perl_str($data);
  return encode_json($data);
}

#see(1);
# see([1,2]);
sub see {
  my $data = shift;
  say to_str($data);
  return ['true'];
}

# say read_file('Spp-implement-function.md');
# don't support utf8
sub read_file {
  my $file = shift;
  # say $file;
  local $/;
  open my ($fh), '<', $file or die $!;
  return <$fh>;
}

# say to_str(load_file('spp.rule'));
# failed rule is not compitiable with JSON
sub load_file {
  my $file_name = shift;
  my $file_txt = read_file($file_name);
  return decode_json($file_txt);
}

# write_file('test.file', 'hello');
sub write_file {
  my ($file, $str) = @_;
  open my ($fh), '>', $file or die $!;
  print {$fh} $str;
  return ['true'];
}

# say len('str'); => 3
# say len([1,2,3]); => 3
sub len {
  my $data = shift;
  # see $data;
  return scalar( @{$data} ) if is_perl_array($data);
  return $data if is_perl_int($data);
  return length( $data ) if is_perl_str($data);
}

# say '<' . trim(' aa ') . '>'; #  <aa>
sub trim {
  my $str = shift;
  if (is_perl_str($str)) {
    $str =~ s/^\s+|\s+$//g;
    return $str;
  }
  my $str_json = to_str($str);
  croak("trim only could make string, not $str_json");
}

# say to_str(sublist([1,2,3], 1, -1)); # [2,3]
# say to_str(sublist([1,2,3,4], 1, -2)); # [2,3]
# see sublist([0,1,2,3], 0, -1);
# exit;
sub sublist {
  my ($array, $from, $to) = @_;
  if (is_perl_array($array)) {
    # splice would change value, so need make a copy
    my $list = [ @{$array} ];
    if ($to < 0) {
      my $len = len($list) + $to - $from + 1;
      my $sub_list = [ splice $list, $from, $len ];
      return $sub_list;
    }
    return [ splice $list, $from, $to ];
  }
  my $array_str = to_str($array);
  croak("sublist only could process array: not $array_str");
}

#my $x = ['nil'];
#my $y = fill_array($x, 3);
#say to_str($y); # [["nil"],["nil"],["nil"]]
sub fill_array {
  my ($value, $len) = @_;
  my @fill_array;
  for my $x (1 .. $len) {
    push @fill_array, $value;
  }
  return [ @fill_array ];
}

# say 'it same' if is_same([1,2], [1,2]);
# say 'it not same' unless is_same([1,2], [2,3]);
sub is_same {
  my ($data_one, $data_two) = @_;
  if (is_perl_int($data_one) and is_perl_int($data_two)) {
    return ($data_one eq $data_two);
  }
  return 1 if to_str($data_one) eq to_str($data_two);
  return 0;
}

# say first('123'); # 1
# say first([1,2,3]); # 1
sub first {
  my $data = shift;
  return $data->[0] if is_perl_array($data);
  return substr($data, 0, 1) if is_perl_str($data);
}

sub second {
  my $data = shift;
  return $data->[1] if is_perl_array($data);
  return substr($data, 1, 1) if is_perl_str($data);
}

# say tail('123'); # 3
# say tail([1,2,3]); # 3
sub tail {
  my $data = shift;
  return $data->[-1] if is_perl_array($data);
  return substr($data, -1) if is_perl_str($data);
}

#say to_str(rest('1234')); # 234
#my $x = [1,2,3];
#say to_str([ rest($x)]); # [2,3,4]
#see $x;
#exit;
# rest return list
sub rest {
   my $data = shift;
   my $len_data = len($data);
   # splice would change data, make an copy of $data
   return [ splice( [ @{$data} ], 1, $len_data ) ] if is_perl_array($data);
   return substr($data, 1) if is_perl_str($data);
}

#say 'int in int array' if in(1,[1,2]);
#say 'str in str array' if in('a', [1,'a']);
#say 'array in array of array' if ([1,2], [1,[1,2]]);
sub in {
  my ($element, $array) = @_;
  my $element_str = to_str($element);
  for my $x (@{$array}) {
    return 1 if to_str($x) eq $element_str;
  }
  return 0;
}

##################################
# my $a = [1,2,3];
# my $b = [4,5,6];
# say to_str(zip($a, $b));
# [[1,4],[2,5],[3,6]]
# #################################

sub zip {
  my ($a_one, $a_two) = @_;
  return [ pairwise { [$a, $b] } @$a_one, @$a_two ];
}

############################################
# Spp data is
############################################

#say '["a",1] is atom' if is_atom(['a',1]);
#my $data = ['a',[1,2,3],3];
#say '["a",[1,2,3]] is atom' if is_atom($data);
#say 'its is perl array' if is_perl_array($data);
sub is_atom {
   my $x = shift;
   return 0 unless is_perl_array($x);
   return 0 unless len($x) > 0;
   return 0 unless is_perl_str(first($x));
   return 1;
}

#my $data = ['a',[1,2,3],3];
#say to_str($data) . ' is not atoms' unless is_atoms($data);
#my $atoms = [['a',1],['b',2]];
#say to_str($atoms) . " is atoms" if is_atoms($atoms);
#my $no_atoms = [['a',1],['b',2], 3];
#say to_str($no_atoms) . " is not atoms" unless is_atoms($data);

sub is_atoms {
  my $pairs = shift;
  if ( is_perl_array($pairs) ) {
    for my $pair (values $pairs) {
      return 0 unless is_atom($pair);
    }
  }
  return 1;
}

#say type(['str', 'str']);
#say type(['list', ['hello', 'dd']]);
sub type {
  my $x = shift;
  return first($x) if is_atom($x);
  return '';
}

# say value(['str', 'hello']);
sub value {
  my $x = shift;
  return $x->[1] if is_atom($x);
  my $x_str = to_str($x);
  croak "Could not get $x_str value";
}

#say '["false"] is fail' if is_fail(['false']);
#say '["true"] is not fail' unless is_fail(['true']);
sub is_fail {
  my $x = shift;
  return 1 if is_false($x) or is_nil($x);
  return 0;
}

sub is_false {
  my $x = shift;
  return 1 if type($x) eq 'false';
  return 0;
}

# say "['true'] is true" if is_true(['true']);
sub is_true {
  my $x = shift;
  return 1 if type($x) eq 'true';
  return 0;
}

sub is_nil {
  my $x = shift;
  return 1 if type($x) eq 'nil';
  return 0;
}

sub is_dot {
  my $x = shift;
  return 1 if type($x) eq 'dot';
  return 0;
}

sub is_lambda {
  my $x = shift;
  return 1 if type($x) eq 'lambda';
  return 0;
}

sub is_ref {
  my $x = shift;
  return 1 if type($x) eq 'ref';
  return 0;
}

sub is_int {
  my $x = shift;
  return 1 if type($x) eq 'int';
  return 0;
}

sub is_array {
  my $x = shift;
  return 1 if type($x) eq 'array';
  return 0;
}

sub is_str {
  my $x = shift;
  return 1 if type($x) eq 'str';
}

sub is_hash {
  my $x = shift;
  return 1 if type($x) eq 'hash';
  return 0;
}

#say "'str' is match" if is_match('str');
#say "['true'] is match" if is_match(['true']);
#say "['false'] is not match" unless is_match(['false']);
#say "['str', 'ss'] is match" if is_match(['str', 'ss']);
#say "[['int', 1],['int',2]] is match" if is_match([['int', 1],['int',2]]);
sub is_match {
  my $x = shift;
  return 1 if is_true($x) or is_perl_str($x);
  return 0 if is_fail($x);
  return 1 if is_atom($x) or is_atoms($x);
  croak("croak spp data");
}

#say "['sym', 'name'] is sym" if is_sym(['sym', 'name']);
sub is_sym {
  my $x = shift;
  return 1 if type($x) eq 'sym';
  return 0;
}

sub is_int {
  my $x = shift;
  return 1 if type($x) eq 'int';
  return 0;
}

#say "['list', 'name'] is list" if is_list(['list', 'name']);
sub is_list {
  my $x = shift;
  return 1 if type($x) eq 'list';
  return 0;
}

sub is_rule {
  my $x = shift;
  return 1 if type($x) eq 'rule';
  return 0;
}

sub is_context {
  my $x = shift;
  return 1 if type($x) eq 'context';
  return 0;
}

###############################
# keyword define
###############################

#say "['sym', 'if'] is if" if is_if(['sym','if']);
sub is_if {
  my $x = shift;
  return 1 if is_sym($x) and value($x) eq 'if';
  return 0;
}

#say "['sym', 'when'] is when" if is_when(['sym','when']);
sub is_when {
  my $x = shift;
  return 1 if is_sym($x) and value($x) eq 'when';
  return 0;
}

#say "['sym', 'else'] is else" if is_else(['sym','else']);
sub is_else {
  my $x = shift;
  return 1 if is_sym($x) and value($x) eq 'else';
  return 0;
}

#say "['sym', 'elsif'] is elsif" if is_elsif(['sym','elsif']);
sub is_elsif {
  my $x = shift;
  return 1 if is_sym($x) and value($x) eq 'elsif';
  return 0;
}

#say "['sym', 'case'] is case" if is_if(['sym','case']);
sub is_case {
  my $x = shift;
  return 1 if is_sym($x) and value($x) eq 'case';
  return 0;
}

#say "['sym', 'in'] is in" if is_in(['sym','in']);
sub is_in {
  my $x = shift;
  return 1 if is_sym($x) and value($x) eq 'in';
  return 0;
}

#see add_exprs([1,2,3]);
#see add_exprs([[[1,2,3]]]);
sub add_exprs {
  my $atoms = shift;
  return first($atoms) if len($atoms) == 1;
  return ['exprs', $atoms];
}

# used for spp
sub perl_zip { my $args = shift; return zip(@{$args}) }

# used for spp
sub format { my $args = @_; return sprintf(@{$args}) }

# used for spp
sub perl_fill { my $args = shift; return fill_array(@{$args}) }

# say array_index(3, [1,2,3]);
sub array_index {
  my ($value, $array) = @_;
  return firstidx { is_same($_, $value) } @{$array};
}

#see perl_split('abc', 'b');
sub perl_split {
  my ($str, $sep) = @_;
  if (defined $sep) {
    return [ split($sep, $str) ];
  }
  return [ split('', $str) ];
}

# say perl_join([1,2,3], 'a');
sub perl_join {
  my ($array, $sep) = @_;
  if (defined $sep) {
    return join($sep, @$array);
  }
  return join('', @$array);
}

sub perl_max {
  my $args = shift;
  return max(@{$args});
}

sub perl_substr {
  my ($from, $to, $len) = @_;
  return substr($from, $to, $len);
}

sub apply_char {
  my ($len, $cursor) = @_;
  my $pos = $cursor->{POS};
  my $str = $cursor->{STR};
  return '' if $pos >= $cursor->{LEN};
  return substr($str, $pos, $len) if $len > 0;
  return substr($str, $pos + $len, abs($len)) if $len < 0;
}

# test reference for push
# my $x = [];
# push $x, 1;
# see $x;
sub get_token_name {
   my $token_name = shift;
   if ( $token_name =~ /^\./ ) {
      return substr($token_name, 1);
   }
   return $token_name;
}


#my $data = [["array",[["int",1],["int",2]]],["array",[["int",3],["int",4]]]];
#see $data;
#rest($data);
#get_atoms_type($data);
#see $data;
#exit;
sub get_atoms_type {
  my $atoms = shift;
  # see $atoms;
  my $type = type(first($atoms));
  # see $atoms;
  my $types = [];
  for my $atom (values $atoms) {
    # see $atom;
    my $atom_type = type($atom);
    next if $atom_type eq $type;
    push $types, $atom_type;
  }
  return $types if len($types) > 0;
  return $type;
}

sub all_is_sym {
  my $args = shift;
  return 1 if get_atoms_type($args) eq 'sym';
  return 0;
}

sub all_is_int {
  my $args = shift;
  return 1 if get_atoms_type($args) eq 'int';
  return 0;
}

sub all_is_str {
  my $args = shift;
  return 1 if get_atoms_type($args) eq 'str';
  return 0;
}

sub all_is_array {
  my $args = shift;
  return 1 if get_atoms_type($args) eq 'array';
  return 0;
}

sub all_is_hash {
  my $args = shift;
  return 1 if get_atoms_type($args) eq 'hash';
  return 0;
}

=head1 AUTHOR

Michael Song, C<< <10435916 at qq.com> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-spp at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Spp>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Spp::Tools

You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Spp>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Spp>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Spp>

=item * Search CPAN

L<http://search.cpan.org/dist/Spp/>

=back

=head1 ACKNOWLEDGEMENTS

=head1 LICENSE AND COPYRIGHT

Copyright 2015 Michael Song.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=cut

1; # End of Spp::Tools
