package App::EventStreamr::Ingest;
use Method::Signatures;
use Scalar::Util::Reftype;
use Carp 'croak';
use Module::Load;
use Moo;
use namespace::clean;

# ABSTRACT: A Ingest object

our $VERSION = '0.2'; # VERSION: Generated by DZP::OurPkg:Version


my $ConfigRef = sub {
  croak "config isn't a 'App::EventStreamr::Config' object!" unless reftype( $_[0] )->class eq "App::EventStreamr::Config";
};

my $StatusRef = sub {
  croak "config isn't a 'App::EventStreamr::Status' object!" unless reftype( $_[0] )->class eq "App::EventStreamr::Status";
};

has 'config'      => ( is => 'rw', required => 1, isa => $ConfigRef );
has 'status'      => ( is => 'rw', required => 1, isa => $StatusRef );
has 'backend'     => ( is => 'ro', default => sub { 'DVswitch' } );
has 'type'     => ( is => 'ro', default => sub { 'ingest' } );
has 'id'     => ( is => 'ro', default => sub { 'ingest' } );
has '_devices'    => ( is => 'ro', default => sub { { } } );

method _load_package($device) {
  my $pkg = "App::EventStreamr::".$self->backend."::Ingest::".$device->{type};
  load $pkg;
  $self->_devices->{$device->{id}} = $pkg->new(
    device => $device->{device},
    id => $device->{id},
    config => $self->config,
    status => $self->status,
  );
}


method start() {
  foreach my $device (@{$self->config->devices}) {
    if (! defined $self->_devices->{$device->{id}}) {
      $self->_load_package($device);
    }
    $self->_devices->{$device->{id}}->start();
  }
}


method run_stop() {
  foreach my $device (@{$self->config->devices}) {
    if (! defined $self->_devices->{$device->{id}}) {
      $self->_load_package($device);
    }
    $self->_devices->{$device->{id}}->run_stop();
  }
}


method stop() {
  foreach my $device (@{$self->config->devices}) {
    if (! defined $self->_devices->{$device->{id}}) {
      $self->_load_package($device);
    }
    $self->_devices->{$device->{id}}->stop();
  }
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

App::EventStreamr::Ingest - A Ingest object

=head1 VERSION

version 0.2

=head1 SYNOPSIS

This package provides an extendable class for starting/stopping
ingest devices.

=head1 DESCRIPTION

This package provides the core run/stop logic for Ingest Devices. A 
backend specific package will extend this and provide any extra logic 
specific to its needs.

'backend' is expected to be overridden by the consuming role.

=head1 METHODS

=head2 start

  $ingest->start()

Will start all configured devices.

=head2 run_stop

  $ingest->run_stop()

Will start all configured devices if they're intended to be running 
or stop them if they're not.

=head2 stop

  $ingest->stop()

Will stop all configured devices.

=head1 AUTHOR

Leon Wright < techman@cpan.org >

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2014 by Leon Wright.

This is free software, licensed under:

  The GNU Affero General Public License, Version 3, November 2007

=cut
