package App::DataPackUtils;

our $DATE = '2016-02-06'; # DATE
our $VERSION = '0.03'; # VERSION

use 5.010001;
use strict;
use warnings;

use ExtUtils::MakeMaker;

our %SPEC;

$SPEC{dump_datapack_script} = {
    v => 1.1,
    summary => 'Show the content of datapacked script (list of included modules and their versions)',
    args => {
        script => {
            schema=>'str*',
            'x.schema.entity' => 'filename',
            pos => 0,
            completion => sub {
                require Complete::Program;

                my %args = @_;
                my $word = $args{word};

                Complete::Program::complete_program(word => $word);
            },
        },
    },
};
sub dump_datapack_script {
    require File::Slurper;
    require File::Temp;
    require File::Which;

    my %args = @_;

    my $script = $args{script};
    unless (-f $script) {
        $script = File::Which::which($script);
        return [400, "No such script '$script'"] unless $script;
    }
    open my($fh), "<", $script
        or return [500, "Can't open script '$script': $!"];

    my $found;
    while (<$fh>) {
        chomp;
        do {$found++; last} if /\A__DATA__\z/;
    }
    return [412, "No __DATA__ found in script"] unless $found;

    require Data::Section::Seekable::Reader;
    my $reader = Data::Section::Seekable::Reader->new(handle=>$fh);

    my (undef, $temp_filename) = File::Temp::tempfile();

    my @res;
    for my $part ($reader->parts) {
        my $ct = $reader->read_part($part);
        $ct =~ s/^#//gm;
        File::Slurper::write_text($temp_filename, $ct);
        my $mod = $part; $mod =~ s!/!::!g; $mod =~ s/\.pm\z//;
        my $ver = MM->parse_version($temp_filename);
        push @res, {
            module => $mod,
            version => $ver,
            filesize => (-s $temp_filename),
        };
    }

    my %resmeta;
    $resmeta{'table.fields'} = [qw/module version filesize/];
    [200, "OK", \@res, \%resmeta];
}

1;
# ABSTRACT: Collection of CLI utilities related to Module::DataPack

__END__

=pod

=encoding UTF-8

=head1 NAME

App::DataPackUtils - Collection of CLI utilities related to Module::DataPack

=head1 VERSION

This document describes version 0.03 of App::DataPackUtils (from Perl distribution App-DataPackUtils), released on 2016-02-06.

=head1 DESCRIPTION

This distribution provides the following command-line utilities related to
L<Module::DataPack>:

=over

=back

=head1 FUNCTIONS


=head2 dump_datapack_script(%args) -> [status, msg, result, meta]

Show the content of datapacked script (list of included modules and their versions).

This function is not exportable.

Arguments ('*' denotes required arguments):

=over 4

=item * B<script> => I<str>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/App-DataPackUtils>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-App-DataPackkUtils>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=App-DataPackUtils>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<Module::DataPack>

L<App::DataSectionSeekableUtils>

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
