package Bencher::Scenario::IODParsingModules;

our $DATE = '2016-01-05'; # DATE
our $VERSION = '0.03'; # VERSION

use 5.010001;
use strict;
use warnings;

use File::ShareDir::Tarball qw(dist_dir);

our $scenario = {
    summary => 'Benchmark IOD/INI parsing modules',
    participants => [
        {
            module => 'Config::IOD::Reader',
            code_template => 'state $iod = Config::IOD::Reader->new; $iod->read_file(<filename>)',
        },
        {
            module => 'Config::IOD',
            code_template => 'state $iod = Config::IOD->new; $iod->read_file(<filename>)',
        },

        {
            module => 'Config::INI::Reader',
            code_template => 'Config::INI::Reader->read_file(<filename>)',
            tags => ['ini'],
        },
        {
            module => 'Config::IniFiles',
            code_template => 'Config::IniFiles->new(-file => <filename>)',
            tags => ['ini'],
        },
    ],

    datasets => [
    ],
};

my $dir = dist_dir('IOD-Examples')
    or die "Can't find share dir for IOD-Examples";
for my $filename (glob "$dir/examples/extra-bench-*.iod") {
    my $basename = $filename; $basename =~ s!.+/!!;
    push @{ $scenario->{datasets} }, {
        name => $basename,
        args => {filename => $filename},
        ( exclude_participant_tags => ['ini'] ) x ($basename =~ /basic\.iod/ ? 1:0), # these files are not parseable by INI parsers
    };
}

1;
# ABSTRACT: Benchmark IOD/INI parsing modules

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::IODParsingModules

=head1 VERSION

This document describes version 0.03 of Bencher::Scenario::IODParsingModules (from Perl distribution Bencher-Scenario-IODParsingModules), released on 2016-01-05.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m IODParsingModules

To run module startup overhead benchmark:

 % bencher --module-startup -m IODParsingModules

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

L<Config::IOD::Reader> 0.19

L<Config::IOD> 0.23

L<Config::INI::Reader> 0.025

L<Config::IniFiles> 2.88

=head1 BENCHMARK PARTICIPANTS

=over

=item * Config::IOD::Reader (perl_code)

Code template:

 state $iod = Config::IOD::Reader->new; $iod->read_file(<filename>)



=item * Config::IOD (perl_code)

Code template:

 state $iod = Config::IOD->new; $iod->read_file(<filename>)



=item * Config::INI::Reader (perl_code)

Code template:

 Config::INI::Reader->read_file(<filename>)



=item * Config::IniFiles (perl_code)

Code template:

 Config::IniFiles->new(-file => <filename>)



=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i5-2400 CPU @ 3.10GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default option:

 +-----+------------------------------------------------------------------------------+----------+---------+---------+---------+
 | seq | name                                                                         | rate     | time    | errors  | samples |
 +-----+------------------------------------------------------------------------------+----------+---------+---------+---------+
 | 8   | {dataset=>"extra-bench-basic-compat.iod",participant=>"Config::IniFiles"}    | 181      | 5.54ms  | 6.5e-06 | 20      |
 | 6   | {dataset=>"extra-bench-basic-compat.iod",participant=>"Config::INI::Reader"} | 5.8e+02  | 1.7ms   | 5.5e-06 | 21      |
 | 3   | {dataset=>"extra-bench-basic-compat.iod",participant=>"Config::IOD"}         | 635      | 1.58ms  | 9.1e-07 | 20      |
 | 4   | {dataset=>"extra-bench-basic.iod",participant=>"Config::IOD"}                | 654      | 1.53ms  | 2.7e-06 | 20      |
 | 0   | {dataset=>"extra-bench-basic-compat.iod",participant=>"Config::IOD::Reader"} | 727      | 1.38ms  | 8.5e-07 | 20      |
 | 1   | {dataset=>"extra-bench-basic.iod",participant=>"Config::IOD::Reader"}        | 740      | 1.35ms  | 2.8e-06 | 21      |
 | 9   | {dataset=>"extra-bench-typical1.iod",participant=>"Config::IniFiles"}        | 893      | 1.12ms  | 1.5e-06 | 20      |
 | 7   | {dataset=>"extra-bench-typical1.iod",participant=>"Config::INI::Reader"}     | 3.03e+03 | 0.33ms  | 2.1e-07 | 21      |
 | 5   | {dataset=>"extra-bench-typical1.iod",participant=>"Config::IOD"}             | 3.92e+03 | 0.255ms | 4.3e-07 | 20      |
 | 2   | {dataset=>"extra-bench-typical1.iod",participant=>"Config::IOD::Reader"}     | 6.62e+03 | 0.151ms | 2.1e-07 | 20      |
 +-----+------------------------------------------------------------------------------+----------+---------+---------+---------+


Benchmark module startup overhead:

 +-----+---------------------+--------+-------------------+---------+---------+
 | seq | name                | time   | mod_overhead_time | errors  | samples |
 +-----+---------------------+--------+-------------------+---------+---------+
 | 3   | Config::INI::Reader | 41ms   | 36ms              | 0.00058 | 21      |
 | 4   | Config::IniFiles    | 38.5ms | 33.7ms            | 6.3e-05 | 20      |
 | 1   | Config::IOD::Reader | 12ms   | 7.1ms             | 4.9e-05 | 20      |
 | 2   | Config::IOD         | 12ms   | 6.9ms             | 4.4e-05 | 22      |
 | 0   | perl -e1 (baseline) | 4.8ms  | 0ms               | 2e-05   | 23      |
 +-----+---------------------+--------+-------------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-IODParsingModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-IODParsingModules>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-IODParsingModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
