package Module::CoreList::More;

our $DATE = '2015-05-07'; # DATE
our $VERSION = '0.07'; # VERSION

use 5.010001;
use strict;
use warnings;

use Module::CoreList;

sub _firstidx {
    my ($item, $ary) = @_;
    for (0..@$ary-1) {
       return $_ if $ary->[$_] eq $item;
    }
    -1;
}

# construct our own %delta from Module::CoreList's %delta. our version is a
# linear "linked list" (e.g. %delta{5.017} is a delta against %delta{5.016003}
# instead of %delta{5.016}. also, version numbers are cleaned (some versions in
# Module::CoreList has trailing whitespaces or alphas)

# the same for our own %released (version numbers in keys are canonicalized)

our @releases; # list of perl release versions, sorted by version
our @releases_by_date; # list of perl release versions, sorted by release date
our %delta;
our %released;
my %rel_orig_formats;
{
    # first let's only stored the canonical format of release versions
    # (Module::Core stores "5.01" as well as "5.010000"), for less headache
    # let's just store "5.010000"
    my %releases;
    for (sort keys %Module::CoreList::delta) {
        my $canonical = sprintf "%.6f", $_;
        next if $releases{$canonical};
        $releases{$canonical} = $Module::CoreList::delta{$_};
        $released{$canonical} = $Module::CoreList::released{$_};
        $rel_orig_formats{$canonical} = $_;
    }
    @releases = sort keys %releases;
    @releases_by_date = sort {$released{$a} cmp $released{$b}} keys %releases;

    for my $i (0..@releases-1) {
        my $reldelta = $releases{$releases[$i]};
        my $delta_from = $reldelta->{delta_from};
        my $changed = {};
        my $removed = {};
        # make sure that %delta will be linear "linked list" by release versions
        if ($delta_from && $delta_from != $releases[$i-1]) {
            $delta_from = sprintf "%.6f", $delta_from;
            my $i0 = _firstidx($delta_from, \@releases);
            #say "D: delta_from jumps from $delta_from (#$i0) -> $releases[$i] (#$i)";
            # accumulate changes between delta at releases #($i0+1) and #($i-1),
            # subtract them from delta at #($i)
            my $changed_between = {};
            my $removed_between = {};
            for my $j ($i0+1 .. $i-1) {
                my $reldelta_between = $releases{$releases[$j]};
                for (keys %{$reldelta_between->{changed}}) {
                    $changed_between->{$_} = $reldelta_between->{changed}{$_};
                    delete $removed_between->{$_};
                }
                for (keys %{$reldelta_between->{removed}}) {
                    $removed_between->{$_} = $reldelta_between->{removed}{$_};
                }
            }
            for (keys %{$reldelta->{changed}}) {
                next if exists($changed_between->{$_}) &&
                    !defined($changed_between->{$_}) && !defined($reldelta->{changed}{$_}) || # both undef
                    defined ($changed_between->{$_}) && defined ($reldelta->{changed}{$_}) && $changed_between->{$_} eq $reldelta->{changed}{$_}; # both defined & equal
                $changed->{$_} = $reldelta->{changed}{$_};
            }
            for (keys %{$reldelta->{removed}}) {
                next if $removed_between->{$_};
                $removed->{$_} = $reldelta->{removed}{$_};
            }
        } else {
            $changed = { %{$reldelta->{changed}} };
            $removed = { %{$reldelta->{removed} // {}} };
        }

        # clean version numbers
        for my $k (keys %$changed) {
            for ($changed->{$k}) {
                next unless defined;
                s/\s+$//; # eliminate trailing space
                # for "alpha" version, turn trailing junk such as letters to _
                # plus a number based on the first junk char
                s/([^.0-9_])[^.0-9_]*$/'_'.sprintf('%03d',ord $1)/e;
            }
        }
        $delta{$releases[$i]} = {
            changed => $changed,
            removed => $removed,
        };
    }
}

my $removed_from = sub {
    my ($order, $module) = splice @_,0,2;
    $module = shift if eval { $module->isa(__PACKAGE__) } && @_ > 0 && defined($_[0]) && $_[0] =~ /^\w/;

    my $ans;
    for my $rel ($order eq 'date' ? @releases_by_date : @releases) {
        my $delta = $delta{$rel};
        if ($delta->{removed}{$module}) {
            $ans = $rel_orig_formats{$rel};
            last;
        }
    }

    return wantarray ? ($ans ? ($ans) : ()) : $ans;
};

sub removed_from {
    $removed_from->('', @_);
}

sub removed_from_by_date {
    $removed_from->('date', @_);
}

my $first_release = sub {
    my ($order, $module) = splice @_,0,2;
    $module = shift if eval { $module->isa(__PACKAGE__) } && @_ > 0 && defined($_[0]) && $_[0] =~ /^\w/;

    my $ans;
    for my $rel ($order eq 'date' ? @releases_by_date : @releases) {
        my $delta = $delta{$rel};
        if (exists $delta->{changed}{$module}) {
            $ans = $rel_orig_formats{$rel};
            last;
        }
    }

    return wantarray ? ($ans ? ($ans) : ()) : $ans;
};

sub first_release {
    $first_release->('', @_);
}

sub first_release_by_date {
    $first_release->('date', @_);
}

my $is_core = sub {
    my $all = pop;
    my $module = shift;
    $module = shift if eval { $module->isa(__PACKAGE__) } && @_ > 0 && defined($_[0]) && $_[0] =~ /^\w/;
    my ($module_version, $perl_version);

    $module_version = shift if @_ > 0;
    $perl_version   = @_ > 0 ? shift : $];

    my $mod_exists = 0;
    my $mod_ver; # module version at each perl release, -1 means doesn't exist

  RELEASE:
    for my $rel (sort keys %delta) {
        last if $all && $rel > $perl_version; # this is the difference with is_still_core()

        my $reldelta = $delta{$rel};

        if ($rel > $perl_version) {
            if ($reldelta->{removed}{$module}) {
                $mod_exists = 0;
            } else {
                next;
            }
        }

        if (exists $reldelta->{changed}{$module}) {
            $mod_exists = 1;
            $mod_ver = $reldelta->{changed}{$module};
        } elsif ($reldelta->{removed}{$module}) {
            $mod_exists = 0;
        }
    }

    if ($mod_exists) {
        if (defined $module_version) {
            return 0 unless defined $mod_ver;
            return version->parse($mod_ver) >= version->parse($module_version) ? 1:0;
        }
        return 1;
    }
    return 0;
};


my $list_core_modules = sub {
    my $all = pop;
    my $class = shift if @_ && eval { $_[0]->isa(__PACKAGE__) };
    my $perl_version = @_ ? shift : $];

    my %added;
    my %removed;

  RELEASE:
    for my $rel (sort keys %delta) {
        last if $all && $rel > $perl_version; # this is the difference with list_still_core_modules()

        my $delta = $delta{$rel};

        next unless $delta->{changed};
        for my $mod (keys %{$delta->{changed}}) {
            # module has been removed between perl_version..latest, skip
            next if $removed{$mod};

            if (exists $added{$mod}) {
                # module has been added in a previous version, update first
                # version
                $added{$mod} = $delta->{changed}{$mod} if $rel <= $perl_version;
            } else {
                # module is first added after perl_version, skip
                next if $rel > $perl_version;

                $added{$mod} = $delta->{changed}{$mod};
            }
        }
        next unless $delta->{removed};
        for my $mod (keys %{$delta->{removed}}) {
            delete $added{$mod};
            # module has been removed between perl_version..latest, mark it
            $removed{$mod}++ if $rel >= $perl_version;
        }

    }
    %added;
};

sub is_core { $is_core->(@_,1) }

sub is_still_core { $is_core->(@_,0) }

sub list_core_modules { $list_core_modules->(@_,1) }

sub list_still_core_modules { $list_core_modules->(@_,0) }

1;

# ABSTRACT: More functions for Module::CoreList

__END__

=pod

=encoding UTF-8

=head1 NAME

Module::CoreList::More - More functions for Module::CoreList

=head1 VERSION

This document describes version 0.07 of Module::CoreList::More (from Perl distribution Module-CoreList-More), released on 2015-05-07.

=head1 SYNOPSIS

 use Module::CoreList::More;

 # true, this module has always been in core since specified perl release
 Module::CoreList::More->is_still_core("Benchmark", 5.010001);

 # false, since CGI is removed in perl 5.021000
 Module::CoreList::More->is_still_core("CGI");

 # false, never been in core
 Module::CoreList::More->is_still_core("Foo");

 my %modules = list_still_core_modules(5.010001);

=head1 DESCRIPTION

This module is my experiment for providing more functionality to (or related to)
L<Module::CoreList>. Some ideas include: faster functions, more querying
functions, more convenience functions. When I've got something stable and useful
to show for, I'll most probably suggest the appropriate additions to
Module::CoreList.

Below are random notes:

=head1 FUNCTIONS

These functions are not exported. They can be called as function (e.g.
C<Module::CoreList::More::is_still_core($name)> or as class method (e.g. C<<
Module::CoreList::More->is_still_core($name) >>.

=head2 first_release( MODULE )

Like Module::CoreList's version, but faster (see L</"BENCHMARK">).

=head2 first_release_by_date( MODULE )

Like Module::CoreList's version, but faster (see L</"BENCHMARK">).

=head2 removed_from( MODULE )

Like Module::CoreList's version, but faster (see L</"BENCHMARK">).

=head2 removed_from_by_date( MODULE )

Like Module::CoreList's version, but faster (see L</"BENCHMARK">).

=head2 is_core( MODULE, [ MODULE_VERSION, [ PERL_VERSION ] ] )

Like Module::CoreList's version, but faster (see L</"BENCHMARK">).

=head2 is_still_core( MODULE, [ MODULE_VERSION, [ PERL_VERSION ] ] )

Like C<is_core>, but will also check that from PERL_VERSION up to the latest
known version, MODULE has never been removed from core.

Note/idea: could also be implemented by adding a fourth argument
MAX_PERL_VERSION to C<is_core>, defaulting to the latest known version.

=head2 list_core_modules([ PERL_VERSION ]) => %modules

List modules that are in core at specified perl release.

=head2 list_still_core_modules([ PERL_VERSION ]) => %modules

List modules that are (still) in core from specified perl release to the latest.
Keys are module names, while values are versions of said modules in specified
perl release.

=head1 BENCHMARK

                                    Rate MC->removed_from(Foo) MC->removed_from(CGI) MCM->removed_from(Foo) MCM->removed_from(CGI)
 MC->removed_from(Foo)  131.695+-0.074/s                    --                -89.7%                 -99.5%                 -99.5%
 MC->removed_from(CGI)   1281.91+-0.85/s                873.4%                    --                 -95.1%                 -95.4%
 MCM->removed_from(Foo)  26037.8+-0.06/s              19671.2%               1931.2%                     --                  -6.3%
 MCM->removed_from(CGI) 27802.4+-0.071/s              21011.1%               2068.8%                   6.8%                     --
 
                                            Rate MC->removed_from_by_date(Foo) MC->removed_from_by_date(CGI) MCM->removed_from_by_date(Foo) MCM->removed_from_by_date(CGI)
 MC->removed_from_by_date(Foo)  132.265+-0.027/s                            --                        -89.5%                         -99.5%                         -99.5%
 MC->removed_from_by_date(CGI)       1262.7+-1/s                        854.7%                            --                         -95.1%                         -95.7%
 MCM->removed_from_by_date(Foo)   25980.5+-7.5/s                      19542.8%                       1957.6%                             --                         -10.9%
 MCM->removed_from_by_date(CGI)      29164+-11/s                      21949.5%                       2209.7%                          12.3%                             --
 
                                    Rate MC->first_release(Foo) MC->first_release(CGI) MCM->first_release(Foo) MCM->first_release(CGI)
 MC->first_release(Foo)  131.55+-0.077/s                     --                 -86.1%                  -99.4%                 -100.0%
 MC->first_release(CGI)   946.91+-0.62/s          619.81+-0.63%                     --                  -95.8%                  -99.6%
 MCM->first_release(Foo)  22448.6+-6.7/s               16964.7%                2270.7%                      --                  -91.6%
 MCM->first_release(CGI)   266300+-470/s           202340+-380%             28023+-53%            1086.3+-2.1%                      --
 
                                             Rate MC->first_release_by_date(Foo) MC->first_release_by_date(CGI) MCM->first_release_by_date(Foo) MCM->first_release_by_date(CGI)
 MC->first_release_by_date(Foo)  131.813+-0.052/s                             --                         -81.4%                          -99.4%                         -100.0%
 MC->first_release_by_date(CGI)    708.88+-0.32/s                         437.8%                             --                          -96.8%                          -99.7%
 MCM->first_release_by_date(Foo)      22444+-20/s                       16927.2%                   3066.1+-3.1%                              --                          -91.5%
 MCM->first_release_by_date(CGI)    264160+-110/s                      200305.2%                       37164.5%                      1077+-1.1%                              --
 
                              Rate MC->is_core(Foo) is_still_core(Foo) MCM->is_core(Foo)
 MC->is_core(Foo)   131.8+-0.089/s               --             -98.6%            -99.3%
 is_still_core(Foo)  9450.8+-4.8/s          7070.6%                 --            -50.3%
 MCM->is_core(Foo)     19020+-39/s       14331+-31%      101.25+-0.42%                --
 
                                    Rate MC->is_core(Benchmark) is_still_core(Benchmark) MCM->is_core(Benchmark)
 MC->is_core(Benchmark)   459.83+-0.33/s                     --                   -94.9%                  -97.4%
 is_still_core(Benchmark)    9066+-4.4/s                1871.6%                       --                  -49.6%
 MCM->is_core(Benchmark)     18003+-26/s           3815.2+-6.3%              98.58+-0.3%                      --
 
                              Rate MC->is_core(CGI) is_still_core(CGI) MCM->is_core(CGI)
 MC->is_core(CGI)   674.18+-0.41/s               --             -92.4%            -96.2%
 is_still_core(CGI)     8826+-14/s     1209.1+-2.2%                 --            -50.2%
 MCM->is_core(CGI)  17738.7+-8.4/s          2531.1%      100.99+-0.33%                --
 
                                             Rate list_still_core_modules(5.020002) list_core_modules(5.020002) list_still_core_modules(5.010001) list_core_modules(5.010001)
 list_still_core_modules(5.020002) 196.35+-0.18/s                                --                       -7.7%                            -23.9%                      -65.0%
 list_core_modules(5.020002)       212.71+-0.24/s                       8.33+-0.16%                          --                            -17.6%                      -62.1%
 list_still_core_modules(5.010001) 258.17+-0.14/s                      31.48+-0.14%                21.37+-0.15%                                --                      -54.0%
 list_core_modules(5.010001)       560.86+-0.29/s                      185.65+-0.3%               163.68+-0.33%                     117.25+-0.16%                          --

=head1 SEE ALSO

L<Module::CoreList>

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Module-CoreList-More>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Module-CoreList-More>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Module-CoreList-More>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
