package Bencher::Scenario::BigInt;

our $DATE = '2016-01-31'; # DATE
our $VERSION = '0.01'; # VERSION

use 5.010001;
use strict;
use warnings;

our $scenario = {
    summary => 'Benchmark arbitrary size integer arithmetics',
    participants => [
        {
            name => '1k-Math::BigInt',
            module=>'Math::BigInt',
            code_template => 'my $val; for (1..1000) { $val = Math::BigInt->new(q(<num1>))+Math::BigInt->new(q(<num2>)) + Math::BigInt->new(q(<num1>)) * Math::BigInt->new(q(<num2>)) } "$val"'
        },
        {
            name => '1k-Math::GMP',
            module=>'Math::GMP',
            code_template => 'my $val; for (1..1000) { $val = Math::GMP->new(q(<num1>))+Math::GMP->new(q(<num2>)) + Math::GMP->new(q(<num1>)) * Math::GMP->new(q(<num2>)) } "$val"'
        },
        {
            name => '1k-native',
            tags => ['no-big'],
            code_template => 'my $val; for (1..1000) { $val = <num1>+<num2> + <num1> * <num2> } $val'
        },
    ],
    datasets => [
        {name=>'1e1', args=>{num1=>"10", num2=>"20"}, result=>"230" },
        {name=>'1e100', args=>{num1=>"1".("0"x100), num2=>"2".("0"x100)}, result=>("2".("0"x99)."3".("0"x100)), exclude_participant_tags=>['no-big'] },
    ],
};

1;
# ABSTRACT: Benchmark arbitrary size integer arithmetics

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::BigInt - Benchmark arbitrary size integer arithmetics

=head1 VERSION

This document describes version 0.01 of Bencher::Scenario::BigInt (from Perl distribution Bencher-Scenario-BigInt), released on 2016-01-31.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m BigInt

To run module startup overhead benchmark:

 % bencher --module-startup -m BigInt

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<Math::BigInt> 1.9997

L<Math::GMP> 2.11

=head1 BENCHMARK PARTICIPANTS

=over

=item * 1k-Math::BigInt (perl_code)

Code template:

 my $val; for (1..1000) { $val = Math::BigInt->new(q(<num1>))+Math::BigInt->new(q(<num2>)) + Math::BigInt->new(q(<num1>)) * Math::BigInt->new(q(<num2>)) } "$val"



=item * 1k-Math::GMP (perl_code)

Code template:

 my $val; for (1..1000) { $val = Math::GMP->new(q(<num1>))+Math::GMP->new(q(<num2>)) + Math::GMP->new(q(<num1>)) * Math::GMP->new(q(<num2>)) } "$val"



=item * 1k-native (perl_code) [no-big]

Code template:

 my $val; for (1..1000) { $val = <num1>+<num2> + <num1> * <num2> } $val



=back

=head1 BENCHMARK DATASETS

=over

=item * 1e1

=item * 1e100

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default options (C<< bencher -m BigInt >>):

 +-----------------+---------+-----------+-----------+---------+---------+
 | participant     | dataset | rate (/s) | time (ms) | errors  | samples |
 +-----------------+---------+-----------+-----------+---------+---------+
 | 1k-Math::BigInt | 1e100   | 22.8      | 43.9      | 2.3e-05 | 20      |
 | 1k-Math::BigInt | 1e1     | 42.8      | 23.3      | 8.2e-06 | 20      |
 | 1k-Math::GMP    | 1e100   | 117       | 8.51      | 3.2e-06 | 20      |
 | 1k-Math::GMP    | 1e1     | 202       | 4.95      | 1.1e-06 | 20      |
 | 1k-native       | 1e1     | 31775     | 0.031471  | 3.8e-11 | 20      |
 +-----------------+---------+-----------+-----------+---------+---------+


Benchmark module startup overhead (C<< bencher -m BigInt --module-startup >>):

 +---------------------+-----------+------------------------+---------+---------+
 | participant         | time (ms) | mod_overhead_time (ms) | errors  | samples |
 +---------------------+-----------+------------------------+---------+---------+
 | Math::BigInt        | 19        | 16.6                   | 4.9e-05 | 20      |
 | Math::GMP           | 9.5       | 7.1                    | 3.3e-05 | 20      |
 | perl -e1 (baseline) | 2.4       | 0                      | 2.9e-06 | 20      |
 +---------------------+-----------+------------------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-BigInt>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-BigInt>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-BigInt>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
