# Copyright 2015 Kevin Ryde
#
# This file is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 3, or (at your option) any later
# version.
#
# This file is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
# for more details.
#
# You should have received a copy of the GNU General Public License along
# with this file.  See the file COPYING.  If not, see
# <http://www.gnu.org/licenses/>.

package Graph::Maker::StarChain;
use 5.004;
use strict;
use base 'Graph::Maker';

use vars '$VERSION';
$VERSION = 4;

# uncomment this to run the ### lines
# use Smart::Comments;

sub _default_graph_maker {
  require Graph;
  Graph->new(@_);
}

sub init {
  my ($self, %params) = @_;

  my @N_list = grep {$_ != 0} @{delete($params{'N_list'})};
  if (@N_list <= 1) {
    require Graph::Maker::Star;
    my $N = $N_list[0] || 0;
    my $graph = Graph::Maker->new('star', N => $N, %params);

    # workaround for Graph::Maker::Star 0.01 giving empty graph for N=1
    if ($N == 1 && $graph->vertices == 0) { $graph->add_vertex(1); }

    return $graph;
  }

  my $graph_maker = delete($params{'graph_maker'});
  $graph_maker ||= \&_default_graph_maker;
  my $graph = $graph_maker->(%params);

  my $directed = $graph->is_directed;
  $graph->add_path(1 .. scalar(@N_list));
  if ($directed) { $graph->add_path(reverse 1 .. scalar(@N_list)); }

  my $v = scalar(@N_list) + 1;
  foreach my $i (0 .. $#N_list) {
    my $from = $i + 1;
    foreach (2 .. $N_list[$i]) {
      my $to = $v++;
      $graph->add_edge($from, $to);
      if ($directed) { $graph->add_edge($to, $from); }
    }
  };
  $graph->set_graph_attribute (name => "Star Chain ".join(',',@N_list));
  return $graph;
}

Graph::Maker->add_factory_type('star_chain' => __PACKAGE__);
1;

__END__

=for stopwords Ryde

=head1 NAME

Graph::Maker::StarChain - create chain of star graphs

=for test_synopsis my ($graph)

=head1 SYNOPSIS

 use Graph::Maker::StarChain;
 $graph = Graph::Maker->new ('star_chain', N_list => [3,1,4]);

=head1 DESCRIPTION

C<Graph::Maker::StarChain> creates C<Graph.pm> graphs of chains of stars.
Stars of a given number of vertices are connected at centres to make a
chain.  For example

     *   *         *
      \ /          |         N_list => [6,1,1,4]
   *---@---@---@---@---*
      / \          |
     *   *         *

The vertices marked "@" are the centres of stars of 6, 1, 1 and 4 vertices
respectively.  Those counts include the centre vertex, the same as the N in
C<Graph::Maker::Star> includes the centre.

One or both ends of the chain can have a 1-star and reduce the second last
if desired.  This changes the numbering of the vertices, but the structure
is the same (isomorphic).  For example

     *   *         *
      \ /          |         N_list => [1,5,1,1,4]
   @---@---@---@---@---*
      / \          |
     *   *         *

If C<N_list> is a single element then the graph is single star the same as
L<Graph::Maker::Star>.  An C<N_list=E<gt>[0]> or empty C<N_list=E<gt>[]>
gives an empty graph.

In the current implementation a 0-star in the chain is treated as no star at
all at that point and is discarded.  But don't rely on that yet (only a
single 0 or empty).

=head1 FUNCTIONS

=over

=item C<$graph = Graph::Maker-E<gt>new('star_chain', key =E<gt> value, ...)>

The key/value parameters are

    N_list => arrayref of integers

Other parameters are passed to C<Graph-E<gt>new()>.

If the graph is directed (the default) then edges are added in both
directions.  Option C<undirected =E<gt> 1> creates an undirected graph and
for it there is a single edge each.

=back

=head1 SEE ALSO

L<Graph::Maker>,
L<Graph::Maker::Star>

=head1 LICENSE

Copyright 2015 Kevin Ryde

This file is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 3, or (at your option) any later
version.

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
more details.

You should have received a copy of the GNU General Public License along with
This file.  If not, see L<http://www.gnu.org/licenses/>.

=cut
