//
//      tardy - a tar post-processor
//      Copyright (C) 1998, 1999, 2003, 2008, 2009 Peter Miller
//
//      This program is free software; you can redistribute it and/or modify
//      it under the terms of the GNU General Public License as published by
//      the Free Software Foundation; either version 3 of the License, or
//      (at your option) any later version.
//
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU General Public License for more details.
//
//      You should have received a copy of the GNU General Public License
//      along with this program. If not, see
//      <http://www.gnu.org/licenses/>.
//

#include <libtardy/ac/string.h>

#include <libtardy/error.h>
#include <libtardy/tar/format.h>
#include <libtardy/tar/output/tar/v7.h>


tar_output_tar_v7::~tar_output_tar_v7()
{
}


tar_output_tar_v7::tar_output_tar_v7(file_output *arg) :
    tar_output_tar_base(arg)
{
}


void
tar_output_tar_v7::write_header(const tar_header &h)
{
    char block[TBLOCK];
    memset(block, 0, sizeof(block));
    header_ty *hp = (header_ty *)block;

    if (h.name.length() < 1)
        fatal("filename \"%s\" too short", h.name.to_c_string());
    if (h.name.length() > sizeof(hp->name))
    {
        fatal
        (
            "filename \"%s\" too long (by %d)",
            h.name.to_c_string(),
            (int)(h.name.length() - sizeof(hp->name))
        );
    }
    hp->name_set(h.name);

    hp->mode_set(h.mode);
    hp->uid_set(h.user_id);
    hp->gid_set(h.group_id);
    hp->size_set(h.size);
    hp->mtime_set(h.mtime);

    hp->linkflag_set(LF_NORMAL);
    switch (h.type)
    {
    case tar_header::type_normal:
    case tar_header::type_normal_contiguous:
        hp->linkflag_set(LF_OLDNORMAL);
        break;

    case tar_header::type_directory:
        hp->linkflag_set(LF_NORMAL);
        hp->name_set(h.name + "/");
        break;

    case tar_header::type_link_hard:
        hp->linkflag_set(LF_LINK);
        break;

    case tar_header::type_link_symbolic:
        hp->linkflag_set(LF_SYMLINK);
        break;

    case tar_header::type_fifo:
        fatal
        (
                "\"%s\" fifo files not supported by this format",
                h.name.to_c_string()
        );
        break;

    case tar_header::type_device_block:
        fatal
        (
            "\"%s\" block special devices not supported by this format",
                h.name.to_c_string()
        );
        break;

    case tar_header::type_device_character:
        fatal
        (
        "\"%s\" character special devices not supported by this format",
                h.name.to_c_string()
        );
        break;

    case tar_header::type_socket:
        fatal
        (
                "\"%s\" named sockets not supported by this format",
                h.name.to_c_string()
        );
        break;

    case tar_header::type_normal_gzipped:
        fatal
        (
                "\"%s\" compressed file supported by this format",
                h.name.to_c_string()
        );
        break;
    }

    if
    (
        h.type == tar_header::type_link_hard
    ||
        h.type == tar_header::type_link_symbolic
    )
    {
        if (h.linkname.length() < 1)
        {
            fatal
            (
                "linkname \"%s\" too short",
                h.linkname.to_c_string()
            );
        }
        if (h.linkname.length() > sizeof(hp->linkname))
        {
            fatal
            (
                "linkname \"%s\" too long (by %d)",
                h.linkname.to_c_string(),
                (int)(h.linkname.length() - sizeof(hp->linkname))
            );
        }
        hp->linkname_set(h.linkname);
    }

    hp->chksum_set(hp->calculate_checksum());

    write_data(block, sizeof(block));
}
