import numpy

class ReadOnlyMemoryError(Exception): pass
class MemoryNotMappedError(Exception): pass

class Register(object):
  def __init__(self, width=8):
    self.value = 0
    self.width = width
    self.mask = (1 << width) - 1
  
  def set(self, value):
    self.value = value & self.mask
  
  def get(self):
    return self.value
  
  def inc(self):
    self.set(self.get() + 1)
  
  def dec(self):
    self.set(self.get() - 1)
  
  def setbit(self, bit, value):
    if value:
      self.set(self.get() | (1 << bit))
    else:
      self.set(self.get() & (~(1 << bit)))
  
  def getbit(self, bit):
    return (self.get() >> (7 - bit)) & 1

class Stack(object):
  def __init__(self, size, width=8):
    self.size = size
    self.width = width
    self.data = []
    self.mask = (1 << width) - 1
  
  def __str__(self):
    return ", ".join(map(lambda n: "%d (0x%02X)" % (n, n), self.data))
  
  def push(self, value):
    if len(self.data) == self.size:
      raise CPUError(E_SOU)
    
    self.data.append(value & self.mask)
  
  def pop(self):
    if len(self.data) == 0:
      raise CPUError(E_SOU)
    return self.data.pop(-1)
  
  def get(self, n=0):
    return self.data[-(n + 1)]

class ExtendedMemoryBase(object):
  "Defines memory utility methods"
  
  def get_word(self, addr):
    return self.get(addr) + (self.get(addr+1) << 8)
  
  def set_word(self, addr, value):
    self.set(addr, value)
    self.set(addr + 1, value >> 8)
  
  def __getitem__(self, addr):
    return self.get(addr)
  
  def __setitem__(self, addr, value):
    self.set(addr, value)

class ROMemory(ExtendedMemoryBase):
  def __init__(self, size):
    self.size = size
    self.clear()
  
  def clear(self):
    self.data = numpy.zeros(shape=(self.size,), dtype=numpy.uint8)
  
  def load(self, data, start=0):
    end = start + len(data)
    #~ self.data[start:end] = data
    self.data[start:end] = map(ord, data)
  
  def dump(self):
    return self.data
  
  def get(self, addr):
    return self.data[addr]
  
  def set(self, addr, value):
    raise ReadOnlyMemoryError

class RWMemory(ROMemory):
  def set(self, addr, value):
    self.data[addr] = value
    #~ self.data = self.data[:addr] + chr(value & 0xFF) + self.data[addr+1:]

class MemoryMap(ExtendedMemoryBase):
  def __init__(self):
    self.maps = {}
  
  def map(self, start, end, rw=True):
    cls = None
    if rw:
      cls = RWMemory
    else:
      cls = ROMemory
    
    size = end - start
    mem = cls(size)
    self.maps[start] = mem
    return mem
  
  def get_map(self, addr):
    offsets = self.maps.keys()
    minoffset = None
    
    for offset in sorted(offsets):
      if offset > addr:
        break
      minoffset = offset
    
    if minoffset is None:
      raise MemoryNotMappedError
    
    map = self.maps[minoffset]
    if minoffset + map.size < addr:
      raise MemoryNotMappedError
    
    return map, addr - minoffset
  
  def clear(self):
    for map in self.maps.values():
      map.clear()
  
  #~ def load(self):
  #~ def dump(self):
  
  def get(self, addr):
    map, addr = self.get_map(addr)
    return map.get(addr)
  
  def set(self, addr, value):
    map, addr = self.get_map(addr)
    map.set(addr, value)
