import sys
import time

DEFAULT_FORMAT_STR = "[%(time)s] %(channel)s:%(logger)s %(level)s: %(msg)s"

class Output(object):
  def __init__(self, formatstr=DEFAULT_FORMAT_STR):
    self.formatstr = formatstr
  
  def format_msg(self, logger, channel, level, msg):
    d = {
      "logger": logger.name,
      "channel": channel.name,
      "level": level.name,
      "msg": msg,
      "time": time.asctime(),
    }
    
    return self.formatstr % d

class StreamOutput(Output):
  def __init__(self, stream=None, formatstr=DEFAULT_FORMAT_STR):
    if stream is None:
      stream = sys.stderr
    
    self.formatstr = formatstr
    self.stream = stream
  
  def log(self, logger, channel, level, msg):
    formatted = self.format_msg(logger, channel, level, msg)
    
    self.stream.write(formatted + "\n")

class FileOutput(StreamOutput):
  def __init__(self, fname, formatstr=DEFAULT_FORMAT_STR):
    self.formatstr = formatstr
    self.stream = open(fname, "w")
  
  def __del__(self):
    self.stream.close()

class Channel(object):
  def __init__(self, name, outputs=None, filter=None):
    if outputs is None:
      outputs = [StreamOutput()]
    
    self.name = name
    self.outputs = outputs
  
  def log(self, logger, level, msg):
    for output in self.outputs:
      output.log(logger, self, level, msg)

class Level(object):
  def __init__(self, name, priority):
    self.name = name.upper()
    self.priority = priority

class ObjectManagerClass(object):
  def __init__(self):
    self.objects = {}
  
  def add(self, obj):
    self.objects[obj.name] = obj
  
  def get(self, name):
    return self.objects.get(name, None)

ChannelManager = ObjectManagerClass()
ChannelManager.add(Channel("main"))

LevelManager = ObjectManagerClass()
LevelManager.add(Level("DEBUG", 0.1))
LevelManager.add(Level("INFO", 0.3))
LevelManager.add(Level("WARNING", 0.5))
LevelManager.add(Level("ERROR", 0.8))
LevelManager.add(Level("FATAL", 1.0))

class Logger(object):
  def __init__(self, name):
    self.name = name
  
  def log(self, channel_name, level_name, msg):
    channel = ChannelManager.get(channel_name)
    if channel is None:
      raise ValueError("Invalid channel '%s'" % channel_name)
    
    level = LevelManager.get(level_name)
    if level is None:
      raise ValueError("Invalid level '%s'" % level_name)
    
    channel.log(self, level, msg)
