from __future__ import unicode_literals

import platform
import time
import socket
import os
import heapq

from PyProto.eventloop import Exceptions

EVENT_IN = 1
EVENT_OUT = 2
EVENT_EXCEPT = 4

class EventLoop:
    """Base eventloop class.

    This implements the eventloop of PyProto, selecting a given module for
    interfacing with the operating system's FD notification system.

    It also implements timers, e.g. events that fire at a given time.

    Errors in this class raise EventLoopException or another appropriate
    exception.
    """
    def __init__(self, evlooptype=None):
        """Initalise the eventloop.

        evlooptype should be a string with the FD notification system you want
        to use. Supported types are select (all platforms), epoll (Linux), and
        poll (windows, BSD, etc.)
        """
        self.evlooptype = evlooptype

        # Initalise some internal state
        if self.evlooptype == '' or self.evlooptype is None:
            system = platform.system()
            if system == 'Linux':
                self.evlooptype = 'epoll'
            # TODO: kqueue backend, but the kqueue python module sux
            elif system.endswith('BSD') or system == 'Windows':
                self.evlooptype = 'poll'
            else:
                self.evlooptype = 'select'

        if self.evlooptype == 'select':
            from PyProto.eventloop.engines import _select
            self.engine = select.EventEngine()
        elif self.evlooptype == 'poll':
            from PyProto.eventloop.engines import poll
            self.engine = poll.EventEngine()
        elif self.evlooptype == 'epoll':
            from PyProto.eventloop.engines import epoll
            self.engine = epoll.EventEngine()
        else:
            raise EventLoopException("Unsupported eventloop type")

        # FD:Event map
        self.fdmap = dict()

        self.timerlist = list()
        self.timercount = 0

        self.evloop_deadline = None

    # Set an FDEvent object or derivative as an event.
    def set_event(self, fdevent, evtype, removemask=False):
        """Set an FDEvent to be used for evtype. When the event
        happens the appropriate callback will be called in FDEvent.

        Set removemask to remove the mask from the event.
        """
        if evtype & EVENT_IN:
            if fdevent.readfd is None:
                raise EventLoopException("Cannot set an event with EVENT_IN with no readfd!")
            self.set_fd(fdevent.readfd, EVENT_IN, fdevent, removemask)

        if evtype & EVENT_OUT:
            if fdevent.writefd is None:
                raise EventLoopException("Cannot set an event with EVENT_OUT and no writefd!")
            self.set_fd(fdevent.writefd, EVENT_OUT, fdevent, removemask)

        # XXX -- is this the right thing to do? :/
        if evtype & EVENT_EXCEPT:
            raise EventLoopException("Cannot use set_event for setting EVENT_EXCEPT; use EventLoop.set_fd instead")

    # Remove an event by fdevent object
    def remove_event(self, fdevent):
        """Remove event by FDEvent."""
        self.remove_fd(fdevent.readfd)
        self.remove_fd(fdevent.writefd)

    # evtype should have read_callback, write_callback, and except_data events
    # depending on what you are listening for.
    def set_fd(self, fd, evtype, fdevent=None, removemask=False):
        """Set event by fd, specifying the fdevent callback.
        
        If an fd has already been added to the event pool, fdevent may be
        omitted. Otherwise fdevent is mandatory. removemask specifies the
        given eventmask in evtype should be removed
        """
        if hasattr(fd, 'fileno'):
            fd = fd.fileno()

        if fd not in self.fdmap:
            if fdevent is None:
                raise EventLoopException("Trying to add an empty event!")
            self.fdmap[fd] = [fdevent, evtype]
            fdevent.eventmask = evtype
            self.engine.register_fd(fd, self.fdmap[fd][1])
        else:
            if removemask:
                self.fdmap[fd][1] &= ~evtype
            else:
                self.fdmap[fd][1] |= evtype
            self.fdmap[fd][0].eventmask = self.fdmap[fd][1]
            self.engine.modify_fd(fd, self.fdmap[fd][1])

    def remove_fd(self, fd):
        """Remove an fd from the set"""
        if type(fd) != int:
            fd = fd.fileno()

        if fd not in self.fdmap:
            return

        del self.fdmap[fd]
        self.engine.unregister_fd(fd)

    def set_timer(self, when, timerev, recur=False):
        """Create a timer to go off every when seconds
        
        timerev is the timer callback.
        Set recur to true for a recurring event, false for oneshot
        """
        timer = (when, self.timercount, timerev, recur)

        timerev.last_ran = None

        heapq.heappush(self.timerlist, timer)
        self.timercount += 1

    def remove_timer(self, when=None, timercount=None, timerev=None, recur=None):
        """Remove a timer matching the given criteria"""
        for item in self.timerlist:
            ewhen, etimercount, etimerev, erecur = item
            if ewhen is not None and when != ewhen:
                continue

            if etimercount is not None and etimercount != timercount:
                continue

            if etimerev is not None and etimerev != timerev:
                continue

            if erecur is not None and erecur != recur:
                continue

            self.timerlist.remove(item)

    def process_timer(self):
        """Process each timer and return the time needed to sleep
        until the next event.

        Do not call this directly.
        """
        if len(self.timerlist) == 0:
            return None

        # First compute when the next timer runs
        curtime = time.time() # Cached
        next_time = None
        run_list = list()

        # Make a copy of the list for atomicity purposes
        timerlist = list(self.timerlist)

        for item in timerlist:
            when, timercount, timerev, recur = item
            # Starting line (list is sorted -- so this will always be set
            # With the soonest timer)
            if next_time is None:
                next_time = when

            # Not run? pretend it was run now for timing purposes
            if timerev.last_ran is None:
                timerev.last_ran = curtime

            # Calculate if it's eligible to be run yet
            # If not, the value we get will tell us how long to sleep
            # Disregard if curtime == last ran time, this means it was
            # just set or (unlikely) just run
            evinterval = curtime - timerev.last_ran
            if evinterval < when and evinterval > 0:
                # See if sleep interval must be adjusted
                if evinterval < next_time:
                    next_time = evinterval
                
            # Event eligible for running? Let's do this
            elif evinterval >= when:
                run_list.append(item)

        # If no events, return time to wait.
        if len(run_list) == 0:
            return next_time

        # Run pending events
        time_taken = 0
        for item in run_list:
            when, timercount, timerev, recur = item
            time_start = time.time()
            
            timerev.run_timer() # Run the timer
            
            time_fin = time.time()
            timerev.last_ran = time_fin # Update the event.
            time_taken += time_fin - time_start
            
            # Remove this if we have to
            if not recur:
                self.timerlist.remove(item)

            # Oh shit. We're too late.
            if time_taken >= next_time:
                return self.process_timer()

        # Check if our event list has been changed
        # If it hasn't, let's return, updating the time to the next event.
        # Otherwise, the wait time is possibly invalid -- recompute.
        if timerlist == self.timerlist:
            return next_time - time_taken

        # Recompute wait time 
        return self.process_timer()

    def run_once(self):
        """Run the eventloop once, firing timers also."""
        timeout = self.process_timer()
        starttime = time.time()
        # Run until we reach the timeout value...
        while True:
            events = self.engine.run_once(timeout)
            for fd, event in events:
                if event & EVENT_IN != 0:
                    if self.fdmap[fd][0].read_callback() == False:
                        self.set_fd(fd, EVENT_IN, removemask=True)
                if event & EVENT_OUT != 0:
                    if self.fdmap[fd][0].write_callback() == False:
                        self.set_fd(fd, EVENT_OUT, removemask=True)
                if event & EVENT_EXCEPT != 0:
                    if self.fdmap[fd][0].except_callback() == False:
                        self.set_fd(fd, EVENT_EXCEPT, removemask=True)

            if timeout is None:
                break

            curtime = time.time()
            if curtime - starttime >= timeout:
                break

            timeout -= (curtime - starttime)

    def run_forever(self):
        """Run the eventloop forever."""
        while True:
            self.run_once()

