import os
import sqlite3


def sqlite3_row_dict_factory(cur, row):
    """convert sqlite row to dict

    con = sqlite3.connect(...)
    con.row_factory = sqlite3_row_dict_factory
    cur = con.cursor()
    cur.execute("select * from sometable")
    print cur.fetchone()["somecol"]
    """
    return dict( (d[0],row[i]) for i,d in enumerate(cur.description) )


class UTA0(object):
    def __init__(self,db_path):
        if not os.path.exists(db_path):
            raise IOError(db_path + ': Non-existent database file')
        self._db_path = db_path
        self._con = sqlite3.connect(db_path)
        self._con.row_factory = sqlite3_row_dict_factory
        self.__check_schema_version('0.0')


    def __check_schema_version(self,required_version):
        obs_Mm = self.schema_version().split('.')[:2]
        req_Mm = required_version.split('.')[:2]
        if ( obs_Mm != req_Mm ):
            raise RuntimeError("Version mismatch: Version {req_Mm} required, but {db_path} is version {obs_Mm}".format(
                req_Mm = '.'.join(req_Mm), db_path = self._db_path, obs_Mm = '.'.join(obs_Mm)))
        # else no error


    def schema_version(self):
        cur = self._con.cursor()
        cur.execute("select * from meta where key = 'schema_version'")
        return cur.fetchone()['value']


    def get_tx_exons(self,ac,ref):    
        """
        return transcript info for supplied accession (ac), or None if not found
        
        :param ac: transcript accession with version (e.g., 'NM_000051.3')
        :type ac: str
        :param ref: reference genome ('GRCh37.p10' is the only valid value at this time)
        :type ref: str
        
        # tx_exons = db.get_tx_exons('NM_000051.3','GRCh37.p10')
        # len(tx_exons)
        63
        
        tx_exons have the following attributes::
        
          {'ac': 'NM_000051.3',               # transcript accession
          'ref': 'GRCh37.p10',
          'g_start_i': 108093558,             # genomic start coordinate
          'g_end_i': 108093913,               # genome end coordinate
          'name': '1',
          'ord': 1,
          't_start_i': 0
          't_end_i': 355,
          'g_cigar': '355M',                  # CIGAR string, relative to genome
          'g_seq_a': None,
          't_seq_a': None,
          }
        
        .. note:: chromosome and strand are in the tx_info record
        
        For example:
        
        # tx_exons[0]['ac']
        'NM_000051.3'
        
        """
        tx_exons_sql = 'select * from tx_exons where ac=? and ref=? order by g_start_i'
        cur = self._con.cursor()
        cur.execute(tx_exons_sql,[ac,ref])
        return cur.fetchall()


    def get_tx_info(self,ac):
        """return transcript info for supplied accession (ac), or None if not found

        :param ac: transcript accession with version (e.g., 'NM_000051.3')
        :type ac: str
        """
        tx_info_sql = 'select * from tx_info where ac=?'
        cur = self._con.cursor()
        cur.execute(tx_info_sql,[ac])
        return cur.fetchone()


    def get_tx_seq(self,ac):
        """return transcript sequence for supplied accession (ac), or None if not found

        :param ac: transcript accession with version (e.g., 'NM_000051.3')
        :type ac: str
        """
        tx_seq_sql = 'select gene,ac,seq from transcript where ac=?'
        cur = self._con.cursor()
        cur.execute(tx_seq_sql,[ac])
        return cur.fetchone()['seq']


    def get_tx_for_gene(self,gene):
        """
        return transcript info records for supplied gene, in order of decreasing length

        :param gene: HGNC gene name
        :type gene: str
        """
        tx_for_gene_sql = 'select * from tx_info where gene=? order by cds_end_i-cds_start_i desc'
        cur = self._con.cursor()
        cur.execute(tx_for_gene_sql,[gene])
        return cur.fetchall()

## <LICENSE>
## Copyright 2014 BDI Contributors (https://bitbucket.org/invitae/bdi)
## 
## Licensed under the Apache License, Version 2.0 (the "License");
## you may not use this file except in compliance with the License.
## You may obtain a copy of the License at
## 
##     http://www.apache.org/licenses/LICENSE-2.0
## 
## Unless required by applicable law or agreed to in writing, software
## distributed under the License is distributed on an "AS IS" BASIS,
## WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
## See the License for the specific language governing permissions and
## limitations under the License.
## </LICENSE>
