"""MyProxy Client interface

Developed for the NERC DataGrid Project: http://ndg.nerc.ac.uk/

Major re-write of an original class.   This updated version implements methods
with SSL calls with M2Crypto rather use calls to myproxy client executables as
in the original.  This version is adapted and extended from an original 
program myproxy_logon by Tom Uram <turam@mcs.anl.gov>
"""
__author__ = "P J Kershaw"
__date__ = "02/06/05"
__copyright__ = "(C) 2009 Science and Technology Facilities Council"
__license__ = """BSD - See LICENSE file in top-level directory

For myproxy_logon see Access Grid Toolkit Public License (AGTPL)

This product includes software developed by and/or derived from the Access 
Grid Project (http://www.accessgrid.org) to which the U.S. Government retains 
certain rights."""
__contact__ = "Philip.Kershaw@stfc.ac.uk"
__revision__ = '$Id$'
import logging
log = logging.getLogger(__name__)

import sys
import os
import socket
from M2Crypto import X509, RSA, EVP, m2, BIO, SSL
import base64
from ConfigParser import SafeConfigParser

from myproxy.utils.openssl import OpenSSLConfig, OpenSSLConfigError


class CaseSensitiveConfigParser(SafeConfigParser):
    '''Subclass the SafeConfigParser - to preserve the original string case of 
    config section names
    '''   
    def optionxform(self, optionstr):
        '''Extend SafeConfigParser.optionxform to preserve case of option names
        '''
        return optionstr


class _HostCheck(SSL.Checker.Checker):
    """Override SSL.Checker.Checker to allow additional check of MyProxy 
    server identity.  If hostname doesn't match, allow match of host's  
    Distinguished Name against MYPROXY_SERVER_DN setting"""

    def __init__(self, 
                 myProxyServerDN=os.environ.get('MYPROXY_SERVER_DN'),
                 cnHostPfx='',
                 **kw):
        """Override parent class __init__ to enable setting of myProxyServerDN
        setting
        
        @type myProxyServerDN: string
        @param myProxyServerDN: Set the expected Distinguished Name of the
        MyProxy server to avoid errors matching hostnames.  This is useful
        where the hostname is not fully qualified
        
        @type cnHostPfx: string
        @param cnHostPfx: globus host certificates are 
        generated by default with a 'host/' prefix to the host name.  Set
        this keyword to '' or None to override and omit the prefix"""
        
        SSL.Checker.Checker.__init__(self, **kw)
        
        self.myProxyServerDN = myProxyServerDN
        self.cnHostPfx = cnHostPfx
        
        
    def __call__(self, peerCert, host=None):
        """Carry out checks on server ID
        @type peerCert: basestring
        @param peerCert: MyProxy server host certificate as M2Crypto.X509.X509
        instance
        @type host: basestring
        @param host: name of host to check
        """
        
        # Globus host certificate has a "host/" prefix - see explanation in
        # __init__.__doc__
        cnHostPfx = isinstance(self.cnHostPfx, basestring) \
                    and self.cnHostPfx or ''
        host = None or cnHostPfx + self.host
        
        try:
            SSL.Checker.Checker.__call__(self, peerCert, host=host)
            
        except SSL.Checker.WrongHost, e:
            # Try match against DN set from MYPROXY_SERVER_DN / config
            # file setting
            peerCertDN = '/'+peerCert.get_subject().as_text().replace(', ','/')
            
            # If they match drop the exception and return all OK instead
            if peerCertDN != self.myProxyServerDN:
                raise
            
        return True
    
    
class MyProxyClientError(Exception):
    """Base exception class for MyProxyClient exceptions"""

class MyProxyClientConfigError(MyProxyClientError):
    """Error with configuration"""
     
class MyProxyClientGetError(MyProxyClientError):
    """Exceptions arising from get request to server"""
    
class MyProxyClientRetrieveError(MyProxyClientError):
    """Error recovering a response from MyProxy"""
            
        
class MyProxyClient(object):
    """MyProxy client interface 
    
    Based on protocol definitions in: 
    
    http://grid.ncsa.uiuc.edu/myproxy/protocol/
    
    @type getCmd: string
    @cvar getCmd: get command string
    
    @type putCmd: string
    @cvar putCmd: put command string
    
    @type infoCmd: string
    @cvar infoCmd: info command string
    
    @type destroyCmd: string
    @cvar destroyCmd: destroy command string
    
    @type changePassphraseCmd: string
    @cvar changePassphraseCmd: command string to change cred pass-phrase
    
    @type storeCmd: string
    @cvar storeCmd: store command string
    
    @type _hostCertSubDirPath: string
    @cvar _hostCertSubDirPath: sub-directory path host certificate (as tuple)
    
    @type _hostKeySubDirPath: string
    @cvar _hostKeySubDirPath: sub-directory path to host key (as tuple)
    
    @type propertyDefaults: tuple
    @cvar propertyDefaults: sets permissable element names for MyProxy config 
    file
    """
      
    getCmd="""VERSION=MYPROXYv2
COMMAND=0
USERNAME=%s
PASSPHRASE=%s
LIFETIME=%d"""

    putCmd = """VERSION=MYPROXYv2
COMMAND=1
USERNAME=%s
PASSPHRASE=%s
LIFETIME=%s"""

    infoCmd="""VERSION=MYPROXYv2
COMMAND=2
USERNAME=%s
PASSPHRASE=PASSPHRASE
LIFETIME=0"""
 
    destroyCmd="""VERSION=MYPROXYv2
COMMAND=3
USERNAME=%s
PASSPHRASE=PASSPHRASE
LIFETIME=0"""

    changePassphraseCmd="""VERSION=MYPROXYv2
 COMMAND=4
 USERNAME=%s
 PASSPHRASE=%s
 NEW_PHRASE=%s
 LIFETIME=0"""
   
    storeCmd="""VERSION=MYPROXYv2
COMMAND=5
USERNAME=%s
PASSPHRASE=
LIFETIME=%d"""

    _hostCertSubDirPath = ('etc', 'hostcert.pem')
    _hostKeySubDirPath = ('etc', 'hostkey.pem')
    
    # Work out default location of proxy file if it exists.  This is set if a
    # call has been made previously to logon / get-delegation
    defProxyFile = sys.platform == 'win32' and 'proxy' or \
    sys.platform in ('linux2', 'darwin') and '/tmp/x509up_u%s'%(os.getuid()) \
    or None     
    
    # valid configuration property keywords
    propertyDefaults = {
       'hostname':              'localhost',
       'port':                  7512,
       'serverDN':              '',
       'serverCNPrefix':        '',
       'openSSLConfFilePath':   '',
       'proxyCertMaxLifetime':  43200,
       'proxyCertLifetime':     43200,
       'caCertFilePath':        None,
       'caCertDir':             '/etc/grid-security/certificates',
    }
    
    # Restrict attributes to the above properties, their equivalent 
    # protected values + extra OpenSSL config object.
    __slots__ = propertyDefaults.copy()
    __slots__.update(dict([('_'+k, v) for k,v in propertyDefaults.items()] +
                          [('_openSSLConfig', None),
                           ('_cfg',           None)]
                          )
                     )
        
    def __init__(self, cfgFilePath=None, **prop):
        """Make any initial settings for client connections to MyProxy
        
        Settings are held in a dictionary which can be set from **prop,
        a call to setProperties() or by passing settings in an XML file
        given by cfgFilePath
        
        @param cfgFilePath:   set properties via a configuration file
        @param **prop:         set properties via keywords - see 
        propertyDefaults class variable for a list of these
        """
        
        # Default settings.  Nb. '_' - override property methods in order to 
        # set defaults
        for opt, val in MyProxyClient.propertyDefaults.items():
            setattr(self, '_'+opt, val)

        # Configuration file used to get default subject when generating a
        # new proxy certificate request
        self._openSSLConfig = OpenSSLConfig()
        
        # Server host name - take from environment variable if available
        self.hostname = os.environ.get('MYPROXY_SERVER',
                                    MyProxyClient.propertyDefaults['hostname'])
            
        # ... and port number
        self.port = int(os.environ.get('MYPROXY_SERVER_PORT', 
                                       MyProxyClient.propertyDefaults['port']))

        # Server Distinguished Name
        self.serverDN = os.environ.get('MYPROXY_SERVER_DN',
                                    MyProxyClient.propertyDefaults['serverDN'])
            
        # keyword settings
        for opt, val in prop.items():
            setattr(self, opt, val)
        
        # If properties file is set any parameters settings in file will
        # override those set by input keyword
        if cfgFilePath is not None:
            self.parseConfig(cfg=cfgFilePath)


    def parseConfig(self, cfg, section='DEFAULT'):
        '''Extract parameters from _cfg config object'''
        
        if isinstance(cfg, basestring):
            cfgFilePath = os.path.expandvars(cfg)
            self._cfg = CaseSensitiveConfigParser()
            self._cfg.read(cfgFilePath)
        else:
            cfgFilePath = None
            self._cfg = cfg
        
        for key, val in self._cfg.items(section):
            setattr(self, key, val)
        
    # Get/Set Property methods
    def _getHostname(self):
        return self._hostname
    
    def _setHostname(self, val):
        if not isinstance(val, basestring):
            raise AttributeError("Expecting string type for hostname "
                                 "attribute")
        self._hostname = val
        
    hostname = property(fget=_getHostname,
                        fset=_setHostname,
                        doc="hostname of MyProxy server")
    
    def _getPort(self):
        return self._port
    
    def _setPort(self, val):
        if isinstance(val, basestring):
            self._port = int(val)
        elif isinstance(val, int):
            self._port = val
        else:
            raise AttributeError("Expecting int type for port attribute")
    
    port = property(fget=_getPort,
                    fset=_setPort,
                    doc="Port number for MyProxy server")
    
    def _getServerDN(self):
        return self._serverDN
    
    def _setServerDN(self, val):
        if not isinstance(val, basestring):
            raise AttributeError("Expecting string type for serverDN "
                                 "attribute")
        self._serverDN = val
    
    serverDN = property(fget=_getServerDN,
                        fset=_setServerDN,
                        doc="Distinguished Name for MyProxy Server "
                            "Certificate")
    
    def _getServerCNPrefix(self):
        return self._serverCNPrefix
    
    def _setServerCNPrefix(self, val):
        if not isinstance(val, basestring):
            raise AttributeError("Expecting string type for serverCNPrefix "
                                 "attribute")
        self._serverCNPrefix = val
    
    serverCNPrefix = property(fget=_getServerCNPrefix,
                              fset=_setServerCNPrefix,
                              doc="Prefix if any for Server Certificate DN "
                                  "Common Name e.g. 'host/'")
    
    def _getOpenSSLConfFilePath(self):
        return self._openSSLConfFilePath
    
    def _setOpenSSLConfFilePath(self, val):
        if not isinstance(val, basestring):
            raise AttributeError("Expecting string type for "
                                 "openSSLConfFilePath attribute")
        self._openSSLConfFilePath = os.path.expandvars(val)
        self._openSSLConfig.filePath = self._openSSLConfFilePath
        self._openSSLConfig.read()  
    
    openSSLConfFilePath = property(fget=_getOpenSSLConfFilePath,
                                   fset=_setOpenSSLConfFilePath,
                                   doc="file path for OpenSSL config file")
    
    def _getProxyCertMaxLifetime(self):
        return self._proxyCertMaxLifetime
    
    def _setProxyCertMaxLifetime(self, val):
        if isinstance(val, basestring):
            self._proxyCertMaxLifetime = int(val)
            
        elif isinstance(val, int):
            self._proxyCertMaxLifetime = val
        else:
            raise AttributeError("Expecting int type for proxyCertMaxLifetime "
                                 "attribute")
    
    proxyCertMaxLifetime = property(fget=_getProxyCertMaxLifetime,
                                    fset=_setProxyCertMaxLifetime,
                                    doc="Default max. lifetime allowed for "
                                        "Proxy Certificate retrieved - used "
                                        "by store method")
    
    def _getProxyCertLifetime(self):
        return self._proxyCertLifetime
    
    def _setProxyCertLifetime(self, val):
        if isinstance(val, basestring):
            self._proxyCertLifetime = int(val)
        elif isinstance(val, int):
            self._proxyCertLifetime = val
        else:
            raise AttributeError("Expecting int type for proxyCertLifetime "
                                 "attribute")
    
    proxyCertLifetime = property(fget=_getProxyCertLifetime,
                                 fset=_setProxyCertLifetime,
                                 doc="Default proxy cert. lifetime used in "
                                     "logon request")
    
    def _getCACertFilePath(self):
        return self._caCertFilePath
    
    def _setCACertFilePath(self, val):
        '''@type val: basestring
        @param val: file path for CA certificate to be used to verify 
        MyProxy server certificate'''
        
        if isinstance(val, basestring):
            if val == '':
                self._caCertFilePath = None
            else:
                self._caCertFilePath = os.path.expandvars(val)
                
        elif isinstance(val, None):
            raise AttributeError("Expecting string type for caCertFilePath "
                                 "attribute")       
        
    caCertFilePath = property(fget=_getCACertFilePath,
                              fset=_setCACertFilePath,
                              doc="CA certificate file path - MyProxy server "
                                  "certificate must validate against it and/"
                                  "or any present in caCertDir")

    def _getCACertDir(self):
        return self._caCertDir

    def _setCACertDir(self, val):
        '''Specify a directory containing PEM encoded CA certs. used for 
        validation of MyProxy server certificate.
        
        Set to None to make M2Crypto.SSL.Context.load_verify_locations ignore
        this parameter
        
        @type val: basestring/None
        @param val: directory path'''
        
        if isinstance(val, basestring):
            if val == '':
                self._caCertDir = None
            else:
                self._caCertDir = os.path.expandvars(val)
                
        elif isinstance(val, None):
            self._caCertDir = val    
        else:
            raise AttributeError("Expecting string or None type for caCertDir "
                                 "attribute")
        
    caCertDir = property(fget=_getCACertDir,
                         fset=_setCACertDir,
                         doc="directory containing PEM encoded CA "
                             "certificates.  Use along with caCertFilePath "
                             "setting to validate MyProxy server certificate")


    def _getOpenSSLConfig(self):
        "Get OpenSSLConfig object property method"
        return self._openSSLConfig
    
    openSSLConfig = property(fget=_getOpenSSLConfig,
                             doc="OpenSSLConfig object")

          
    def _initConnection(self, 
                        ownerCertFile=None, 
                        ownerKeyFile=None,
                        ownerPassphrase=None):
        """Initialise connection setting up SSL context and client and
        server side identity checks
        
        @type ownerCertFile: basestring
        @param ownerCertFile: client certificate and owner of credential
        to be acted on.  Can be a proxy cert + proxy's signing cert.  Cert
        and private key are not necessary for getDelegation / logon calls
        @type ownerKeyFile: basestring
        @param ownerKeyFile: client private key file
        @type ownerPassphrase: basestring
        @param ownerPassphrase: pass-phrase protecting private key if set - 
        not needed in the case of a proxy private key
        """

        # Must be version 3 for MyProxy
        context = SSL.Context(protocol='sslv3')

        if self.caCertFilePath or self.caCertDir:
            context.load_verify_locations(cafile=self.caCertFilePath,
                                          capath=self.caCertDir)
                            
            # Stop if peer's certificate can't be verified
            context.set_allow_unknown_ca(False)
        else:
            context.set_allow_unknown_ca(True)

        from arcs.gsi import Certificate
        if ownerCertFile:
            try:
                if isinstance(ownerCertFile, Certificate):
                    m2.ssl_ctx_passphrase_callback(context.ctx, lambda *ar: ownerPassphrase)
                    m2.ssl_ctx_use_x509(context.ctx, ownerCertFile._certificate.x509)
                    m2.ssl_ctx_use_rsa_privkey(context.ctx, ownerKeyFile.rsa)
                    if not m2.ssl_ctx_check_privkey(context.ctx):
                        raise ValueError, 'public/private key mismatch'
                else:
                    context.load_cert_chain(ownerCertFile,
                                        keyfile=ownerKeyFile,
                                        callback=lambda *ar, **kw: ownerPassphrase)
            except Exception, e:
                raise MyProxyClientConfigError("Loading certificate "
                                               "and private key for SSL "
                                               "connection [also check CA "
                                               "certificate settings]: %s" % e) 
            
            # Verify peer's certificate
            context.set_verify(SSL.verify_peer, 1) 
        
           
        # Disable for compatibility with myproxy server (er, globus)
        # globus doesn't handle this case, apparently, and instead
        # chokes in proxy delegation code
        context.set_options(m2.SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS)
        
        # connect to myproxy server
        conn = SSL.Connection(context, sock=socket.socket())
        
        # Check server host identity - if host doesn't match use explicit
        # 'serverDN' 
        # host/<hostname> one
        hostCheck = _HostCheck(host=self.hostname,
                               myProxyServerDN=self.serverDN,
                               cnHostPfx=self.serverCNPrefix)
        conn.set_post_connection_check_callback(hostCheck)
        
        return conn
    
            
    def _createCertReq(self, CN, nBitsForKey=1024, messageDigest="md5"):
        """
        Create a certificate request.
        
        @type CN: basestring
        @param CN: Common Name for certificate - effectively the same as the
        username for the MyProxy credential
        @type nBitsForKey: int
        @param nBitsForKey: number of bits for private key generation - 
        default is 1024
        @type messageDigest: basestring
        @param messageDigest: message digest type - default is MD5
        @rtype: tuple
        @return certificate request PEM text and private key PEM text
        """
        
        # Check all required certifcate request DN parameters are set                
        # Create certificate request
        req = X509.Request()
    
        # Generate keys
        key = RSA.gen_key(nBitsForKey, m2.RSA_F4)
    
        # Create public key object
        pubKey = EVP.PKey()
        pubKey.assign_rsa(key)
        
        # Add the public key to the request
        req.set_version(0)
        req.set_pubkey(pubKey)
        
        defaultReqDN = self._openSSLConfig.reqDN
             
        # Set DN
        x509Name = X509.X509_Name()
        x509Name.CN = CN
        
        if defaultReqDN:
            x509Name.OU = defaultReqDN['OU']
            x509Name.O = defaultReqDN['O']
                        
        req.set_subject_name(x509Name)
        
        req.sign(pubKey, messageDigest)

        return (req.as_der(), key.as_pem(cipher=None))


    def _deserializeResponse(self, msg, fields=False):
        """
        Deserialize a MyProxy server response
        
        @param msg: string response message from MyProxy server
        @fields: whether to return the data produced by MyProxy
        @return tuple of integer response and errorTxt string (if any)
        """
        
        lines = msg.split('\n')
        
        # get response value
        responselines = filter(lambda x: x.startswith('RESPONSE'), lines)
        responseline = responselines[0]
        respCode = int(responseline.split('=')[1])
        
        # get error text
        errorTxt = ""
        errorlines = filter(lambda x: x.startswith('ERROR'), lines)
        for e in errorlines:
            etext = e.split('=', 1)[1]
            errorTxt += os.linesep + etext

        if fields:
            fields = {}

            for fieldline in lines:
                # Nb. '1' arg to split ensures owner DN is not split up.
                if fieldline == '\0':
                    break
                field, value = fieldline.split('=', 1)
                fields[field]=value.isdigit() and int(value) or value
            if fields.has_key('RESPONSE'): del fields['RESPONSE']
            if fields.has_key('VERSION'): del fields['VERSION']
            if fields.has_key('ERROR'): del fields['ERROR']
            return respCode, errorTxt, fields
        else:
            return respCode, errorTxt
    
  
    def _deserializeCerts(self, inputDat):
        """Unpack certificates returned from a get delegation call to the
        server
        
        @param inputDat: string containing the proxy cert and private key
        and signing cert all in DER format
        
        @return list containing the equivalent to the input in PEM format"""
        pemCerts = []        
        dat = inputDat
        
        while dat:    
            # find start of cert, get length        
            ind = dat.find('\x30\x82')
            if ind < 0:
                break
                
            len = 256*ord(dat[ind+2]) + ord(dat[ind+3])
    
            # extract der-format cert, and convert to pem
            derCert = dat[ind:ind+len+4]
            
            x509 = X509.load_cert_der_string(derCert)
            pemCert = x509.as_pem()
            
            pemCerts.append(pemCert)
    
            # trim cert from data
            dat = dat[ind + len + 4:]
           
        return pemCerts
    
    
    @classmethod
    def writeProxyFile(cls,proxyCert,proxyPriKey,userX509Cert,filePath=None):
        """Write out proxy cert to file in the same way as myproxy-logon - 
        proxy cert, private key, user cert.  Nb. output from logon can be
        passed direct into this method
        
        @type proxyCert: string 
        @param proxyCert: proxy certificate
        @type proxyPriKey: string
        @param proxyPriKey: private key for proxy
        @type userX509Cert: string
        @param userX509Cert: user certificate which issued the proxy
        @type filePath: string
        @param filePath: set to override the default filePath"""
        
        if filePath is None:
            filePath = MyProxyClient.defProxyFile
            
        if filePath is None:
            MyProxyClientConfigError("Error setting proxy file path - invalid "
                                     "platform?")
        
        outStr = proxyCert + proxyPriKey + userX509Cert       
        open(MyProxyClient.defProxyFile, 'w').write(outStr)
        try:
            # Make sure permissions are set correctly
            os.chmod(MyProxyClient.defProxyFile, 0600)
        except Exception, e:
            # Don't leave the file lying around if couldn't change it's
            # permissions
            os.unlink(MyProxyClient.defProxyFile)
            
            log.error('Unable to set 0600 permissions for proxy file "%s": %s'% 
                      (MyProxyClient.defProxyFile, e))
            raise

    @classmethod
    def readProxyFile(cls, filePath=None):
        """Read proxy cert file following the format used by myproxy-logon - 
        proxy, cert, private key, user cert.
        
        @rtype: tuple
        @return: tuple containing proxy cert, private key, user cert"""
        if filePath is None:
            filePath = MyProxyClient.defProxyFile
            
        if filePath is None:
            MyProxyClientConfigError("Error setting proxy file path - invalid "
                                     "platform?")
               
        proxy = open(MyProxyClient.defProxyFile).read()
        
        # Split certs and key into separate tuple items
        return tuple(['-----BEGIN'+i for i in proxy.split('-----BEGIN')[1:]])
    

    def info(self,
             username, 
             ownerCertFile=None,
             ownerKeyFile=None,
             ownerPassphrase=None):
        """return True/False whether credentials exist on the server for a 
        given username
        
        @raise MyProxyClientGetError:
        @raise MyProxyClientRetrieveError:
        
        @type username: string
        @param username: username selected for credential
        @type ownerCertFile: string
        @param ownerCertFile: certificate used for client authentication with
        the MyProxy server SSL connection.  This ID will be set as the owner
        of the stored credentials.  Only the owner can later remove 
        credentials with myproxy-destroy or the destroy method.  If not set,
        this argument defaults to $GLOBUS_LOCATION/etc/hostcert.pem 
        @type ownerKeyFile: string 
        @param ownerKeyFile: corresponding private key file.  See explanation
        for ownerCertFile
        @type ownerPassphrase: string
        @param ownerPassphrase: passphrase for ownerKeyFile.  Omit if the
        private key is not password protected.
        """
        globusLoc = os.environ.get('GLOBUS_LOCATION')
        if not ownerCertFile:
            if globusLoc:
                ownerCertFile = os.path.join(globusLoc, 
                                            *MyProxyClient._hostCertSubDirPath)
                ownerKeyFile = os.path.join(globusLoc, 
                                            *MyProxyClient._hostKeySubDirPath)
            else:
                raise MyProxyClientError(
            "No client authentication cert. and private key file were given")

        # Set-up SSL connection
        conn = self._initConnection(ownerCertFile=ownerCertFile,
                                    ownerKeyFile=ownerKeyFile,
                                    ownerPassphrase=ownerPassphrase)
        
        conn.connect((self.hostname, self.port))
        
        # send globus compatibility stuff
        conn.write('0')
    
        # send info command - ensure conversion from unicode before writing
        cmd = MyProxyClient.infoCmd % username
        conn.write(str(cmd))
    
        # process server response
        dat = conn.recv(8192)

        # Pass in the names of fields to return in the dictionary 'field'
        respCode, errorTxt, field = self._deserializeResponse(dat, fields=True)

        return not bool(respCode), errorTxt, field


    def changePassphrase(self,
                         username, 
                         passphrase,
                         newPassphrase,
                         ownerCertFile=None,
                         ownerKeyFile=None,
                         ownerPassphrase=None):
        """change pass-phrase protecting the credentials for a given username
        
        @raise MyProxyClientGetError:
        @raise MyProxyClientRetrieveError:
        
        @param username: username of credential
        @param passphrase: existing pass-phrase for credential
        @param newPassphrase: new pass-phrase to replace the existing one.
        @param ownerCertFile: certificate used for client authentication with
        the MyProxy server SSL connection.  This ID will be set as the owner
        of the stored credentials.  Only the owner can later remove 
        credentials with myproxy-destroy or the destroy method.  If not set,
        this argument defaults to $GLOBUS_LOCATION/etc/hostcert.pem 
        @param ownerKeyFile: corresponding private key file.  See explanation
        for ownerCertFile
        @param ownerPassphrase: passphrase for ownerKeyFile.  Omit if the
        private key is not password protected.  
        @return none
        """
        globusLoc = os.environ.get('GLOBUS_LOCATION')
        if not ownerCertFile or not ownerKeyFile:
            if globusLoc:
                ownerCertFile = os.path.join(globusLoc, 
                                         *MyProxyClient._hostCertSubDirPath)
                ownerKeyFile = os.path.join(globusLoc, 
                                         *MyProxyClient._hostKeySubDirPath)
            else:
                raise MyProxyClientError(
            "No client authentication cert. and private key file were given")
        
        # Set-up SSL connection
        conn = self._initConnection(ownerCertFile=ownerCertFile,
                                    ownerKeyFile=ownerKeyFile,
                                    ownerPassphrase=ownerPassphrase)

        conn.connect((self.hostname, self.port))
        
        # send globus compatibility stuff
        conn.write('0')
    
        # send command - ensure conversion from unicode before writing
        cmd = MyProxyClient.changePassphraseCmd % (username, 
                                                   passphrase,
                                                   newPassphrase)
        conn.write(str(cmd))
    
        # process server response
        dat = conn.recv(8192)
            
        respCode, errorTxt = self._deserializeResponse(dat)
        if respCode:
            raise MyProxyClientGetError(errorTxt)


    def destroy(self,
                username,
                ownerCertFile=None,
                ownerKeyFile=None,
                ownerPassphrase=None,
                credname=None):
        """destroy credentials from the server for a given username
        
        @raise MyProxyClientGetError:
        @raise MyProxyClientRetrieveError:
        
        @param username: username selected for credential
        @param ownerCertFile: certificate used for client authentication with
        the MyProxy server SSL connection.  This ID will be set as the owner
        of the stored credentials.  Only the owner can later remove 
        credentials with myproxy-destroy or the destroy method.  If not set,
        this argument defaults to $GLOBUS_LOCATION/etc/hostcert.pem 
        @param ownerKeyFile: corresponding private key file.  See explanation
        for ownerCertFile
        @param ownerPassphrase: passphrase for ownerKeyFile.  Omit if the
        private key is not password protected.  
        @return none
        """
        globusLoc = os.environ.get('GLOBUS_LOCATION')
        if not ownerCertFile or not ownerKeyFile:
            if globusLoc:
                ownerCertFile = os.path.join(globusLoc, 
                                         *MyProxyClient._hostCertSubDirPath)
                ownerKeyFile = os.path.join(globusLoc, 
                                         *MyProxyClient._hostKeySubDirPath)
            else:
                raise MyProxyClientError(
            "No client authentication cert. and private key file were given")
        
        # Set-up SSL connection
        conn = self._initConnection(ownerCertFile=ownerCertFile,
                                    ownerKeyFile=ownerKeyFile,
                                    ownerPassphrase=ownerPassphrase)

        conn.connect((self.hostname, self.port))
        
        # send globus compatibility stuff
        conn.write('0')
    
        # send destroy command - ensure conversion from unicode before writing
        cmd = MyProxyClient.destroyCmd % username
        if credname:
            cmd += "\nCRED_NAME=%s" % credname
        conn.write(str(cmd))

        # process server response
        dat = conn.recv(8192)

        respCode, errorTxt = self._deserializeResponse(dat)
        if respCode:
            raise MyProxyClientGetError(errorTxt)



    def put(self,
            username,
            passphrase,
            certFile,
            keyFile,
            passphraseCallback,
            ownerCertFile=None,
            ownerKeyFile=None,
            ownerPassphraseCallback=None,
            lifetime=None,
            retrievers=None,
            force=True,
            credowner=None,
            credname=None,
            creddesc=None):
        """Upload credentials to the server

        @raise MyProxyClientGetError:
        @raise MyProxyClientRetrieveError:

        @type username: string
        @param username: username selected for new credential
        @type passphrase: string
        @param passphrase: pass-phrase for new credential.  This is the pass
        phrase which protects the uploaded proxy.
        @type certFile: string
        @param certFile: user's X.509 certificate in PEM format
        @type keyFile: string
        @param keyFile: equivalent private key file in PEM format
        @type ownerCertFile: string
        @param ownerCertFile: certificate used for client authentication with
        the MyProxy server SSL connection.  This ID will be set as the owner
        of the stored credentials.  Only the owner can later remove
        credentials with myproxy-destroy or the destroy method.  If not set,
        this argument defaults to $GLOBUS_LOCATION/etc/hostcert.pem or if this
        is not set, certFile
        @type ownerKeyFile: string
        @param ownerKeyFile: corresponding private key file.  See explanation
        for ownerCertFile
        @type ownerPassphraseCallback: function
        @param ownerPassphraseCallback: passphrase callback for unlocking the private
        key. e.g. lambda x: 'secret'
        @type retrievers: string
        @param retrievers: Set to '*' for anonymous
        @type Force: bool
        @param force: set to True to overwrite any existing creds with the
        same username.  If, force=False a check is made with a call to info.
        If creds already exist exit without proceeding
        """

        lifetime = lifetime or self.proxyCertMaxLifetime

        # Inputs must be string type otherwise server will reject the request
        if isinstance(username, unicode):
            username = str(username)

        if isinstance(passphrase, unicode):
            passphrase = str(passphrase)

        globusLoc = os.environ.get('GLOBUS_LOCATION')
        if not ownerCertFile or not ownerKeyFile:
            if globusLoc:
                ownerCertFile = os.path.join(globusLoc,
                                         *MyProxyClient._hostCertSubDirPath)
                ownerKeyFile = os.path.join(globusLoc,
                                         *MyProxyClient._hostKeySubDirPath)
            else:
                # Default so that the owner is the same as the ID of the
                # credentials to be uploaded.
                ownerCertFile = certFile
                ownerKeyFile = keyFile

        ownerPassphrase = passphraseCallback()


        if not force:
            # Check credentials don't already exist
            if self.info(username,
                         ownerCertFile=ownerCertFile,
                         ownerKeyFile=ownerKeyFile,
                         ownerPassphrase=ownerPassphrase)[0]:
                raise MyProxyClientError(
                        "Credentials already exist for user: %s" % username)

        # Set up SSL connection
        conn = self._initConnection(ownerCertFile=ownerCertFile,
                                    ownerKeyFile=ownerKeyFile,
                                    ownerPassphrase=ownerPassphrase)

        conn.connect((self.hostname, self.port))

        # send globus compatibility stuff
        conn.write('0')

        # send put command - ensure conversion from unicode before writing
        cmd = MyProxyClient.putCmd % (username, passphrase, lifetime)
        if retrievers:
            cmd += "\nRETRIEVER=%s" % retrievers
        if credowner:
            cmd += "\nCRED_OWNER=%s" % credowner
        if credname:
            cmd += "\nCRED_NAME=%s" % credname
        if creddesc:
            cmd += "\nCRED_DESC=%s" % creddesc
        conn.write(str(cmd))

        # process server response
        dat = conn.recv(8192)
            
        respCode, errorTxt = self._deserializeResponse(dat)
        if respCode:
            raise MyProxyClientGetError(errorTxt)

        # Certificate request from MyProxy (DER format)
        dat = conn.recv(8192)

        from arcs.gsi import CertificateRequest, Certificate, ProxyCertificate
        req = CertificateRequest(dat)
        if isinstance(certFile, Certificate):
            usercert = certFile
        else:
            usercert = Certificate(certFile, keyFile,
                                   callback=passphraseCallback)
        proxy = ProxyCertificate(usercert, proxykey=req.get_pubkey())
        proxy.sign()
        conn.send(chr(2) + proxy.as_der() + usercert.as_der())

        resp = conn.recv(8192)

        # process server response
        respCode, errorTxt = self._deserializeResponse(resp)
        if respCode:
            raise MyProxyClientRetrieveError(errorTxt)


    def store(self,
              username,
              passphrase, 
              certFile,
              keyFile,
              ownerCertFile=None,
              ownerKeyFile=None,
              ownerPassphrase=None,
              lifetime=None,
              force=True):
        """Upload credentials to the server
        
        @raise MyProxyClientGetError:
        @raise MyProxyClientRetrieveError:
        
        @type username: string
        @param username: username selected for new credential
        @type passphrase: string
        @param passphrase: pass-phrase for new credential.  This is the pass
        phrase which protects keyfile.
        @type certFile: string
        @param certFile: user's X.509 certificate in PEM format
        @type keyFile: string
        @param keyFile: equivalent private key file in PEM format
        @type ownerCertFile: string
        @param ownerCertFile: certificate used for client authentication with
        the MyProxy server SSL connection.  This ID will be set as the owner
        of the stored credentials.  Only the owner can later remove 
        credentials with myproxy-destroy or the destroy method.  If not set,
        this argument defaults to $GLOBUS_LOCATION/etc/hostcert.pem or if this
        is not set, certFile
        @type ownerKeyFile: string
        @param ownerKeyFile: corresponding private key file.  See explanation
        for ownerCertFile
        @type ownerPassphrase: string 
        @param ownerPassphrase: passphrase for ownerKeyFile.  Omit if the
        private key is not password protected.  Nb. keyFile is expected to
        be passphrase protected as this will be the passphrase used for
        logon / getDelegation.
        @type Force: bool
        @param force: set to True to overwrite any existing creds with the
        same username.  If, force=False a check is made with a call to info.
        If creds already, exist exit without proceeding
        """
        
        lifetime = lifetime or self.proxyCertMaxLifetime

        # Inputs must be string type otherwise server will reject the request
        if isinstance(username, unicode):
            username = str(username)
            
        if isinstance(passphrase, unicode):
            passphrase = str(passphrase)
        
        globusLoc = os.environ.get('GLOBUS_LOCATION')
        if not ownerCertFile or not ownerKeyFile:
            if globusLoc:
                ownerCertFile = os.path.join(globusLoc, 
                                         *MyProxyClient._hostCertSubDirPath)
                ownerKeyFile = os.path.join(globusLoc, 
                                         *MyProxyClient._hostKeySubDirPath)
            else:
                # Default so that the owner is the same as the ID of the 
                # credentials to be uploaded.
                ownerCertFile = certFile 
                ownerKeyFile = keyFile
                ownerPassphrase = passphrase
                
        if not force:
            # Check credentials don't already exist
            if self.info(username,
                         ownerCertFile=ownerCertFile,
                         ownerKeyFile=ownerKeyFile,
                         ownerPassphrase=ownerPassphrase)[0]:
                raise MyProxyClientError(
                        "Credentials already exist for user: %s" % username)

        # Set up SSL connection
        conn = self._initConnection(ownerCertFile=ownerCertFile,
                                    ownerKeyFile=ownerKeyFile,
                                    ownerPassphrase=ownerPassphrase)
        
        conn.connect((self.hostname, self.port))
        
        # send globus compatibility stuff
        conn.write('0')
    
        # send store command - ensure conversion from unicode before writing
        cmd = MyProxyClient.storeCmd % (username, lifetime)
        conn.write(str(cmd))
    
        # process server response
        dat = conn.recv(8192)
            
        respCode, errorTxt = self._deserializeResponse(dat)
        if respCode:
            raise MyProxyClientGetError(errorTxt)
        
        # Send certificate and private key
        certTxt = X509.load_cert(certFile).as_pem()
        keyTxt = open(keyFile).read()
        
        conn.send(certTxt + keyTxt)
    
    
        # process server response
        resp = conn.recv(8192)
        respCode, errorTxt = self._deserializeResponse(resp)
        if respCode:
            raise MyProxyClientRetrieveError(errorTxt)
        
         
    def logon(self, username, passphrase, lifetime=None):
        """Retrieve a proxy credential from a MyProxy server
        
        Exceptions:  MyProxyClientGetError, MyProxyClientRetrieveError
        
        @type username: basestring
        @param username: username of credential
        
        @type passphrase: basestring
        @param passphrase: pass-phrase for private key of credential held on
        server
        
        @type lifetime: int
        @param lifetime: lifetime for generated certificate
        
        @rtype: tuple
        @return credentials as strings in PEM format: the
        user certificate, it's private key and the issuing certificate.  The
        issuing certificate is only set if the user certificate is a proxy
        """
        
        lifetime = lifetime or self.proxyCertLifetime

        # Generate certificate request here - any errors will be thrown
        # prior to making the connection and so not upsetting the server
        #
        # - The client will generate a public/private key pair and send a 
        #   NULL-terminated PKCS#10 certificate request to the server.
        certReq, priKey = self._createCertReq(username)

        # Set-up SSL connection
        conn = self._initConnection()
        conn.connect((self.hostname, self.port))
        
        # send globus compatibility stuff
        conn.write('0')
    
        # send get command - ensure conversion from unicode before writing
        cmd = MyProxyClient.getCmd % (username, passphrase, lifetime)
        conn.write(str(cmd))
    
        # process server response
        dat = conn.recv(8192)
        respCode, errorTxt = self._deserializeResponse(dat)
        if respCode:
            raise MyProxyClientGetError(errorTxt)
        
        # Send certificate request
        conn.send(certReq)
    
        # process certificates
        # - 1st byte , number of certs
        dat = conn.recv(1)
        nCerts = ord(dat[0])
        
        # - n certs
        dat = conn.recv(8192)
    
        # process server response
        resp = conn.recv(8192)
        respCode, errorTxt = self._deserializeResponse(resp)
        if respCode:
            raise MyProxyClientRetrieveError(errorTxt)
    
        # deserialize certs from received cert data
        pemCerts = self._deserializeCerts(dat)
        if len(pemCerts) != nCerts:
            MyProxyClientRetrieveError("%d certs expected, %d received" % 
                                                    (nCerts, len(pemCerts)))
    
        # Return certs and private key
        # - proxy cert
        # - private key
        # - rest of cert chain
        creds = [pemCerts[0], priKey]
        creds.extend(pemCerts[1:])
        
        return tuple(creds)
        

    def getDelegation(self, *arg, **kw):
        """Retrieve proxy cert for user - same as logon"""
        return self.logon(*arg, **kw)
