#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Script for runing the recogniser of an ASR model on a set of audios

import argparse
import logging
import os

import confusions
import htk
import common
import edit_distance


def to_homophone(word, homophones_lists):
    for homophones in homophones_lists:
        if word in homophones:
            return homophones[0]

    return word


def evaluate_recognition(args):
    dictionary_filename = args.pronunciation_dictionary_filename or os.path.join(args.recognition_directory,
                                                                                 'dictionary.txt')
    recognition_filename = os.path.join(args.recognition_directory, 'recognition.mlf')

    lexicon = common.parse_dictionary(dictionary_filename)
    lexicon = dict([(k.upper(), list(v)) for k, v in lexicon.items()])

    homophones = common.get_dictionary_homophones(lexicon)

    confs = confusions.load_confusions(args.confusions_csv_filename)
    confs = dict([(u'T_{}_M'.format(conf.token_id), conf.__dict__) for conf in confs.values()])

    mlfs = htk.load_mlf(recognition_filename, utf8_normalization=common.UTF8_NORMALIZATION)

    # Take the first hypothesis for each file in the MLF
    mlfs = dict([(os.path.splitext(os.path.basename(k))[0], v) for k, v in mlfs.items()])

    total_count = 0.0
    presented_count = 0.0
    reported_count = 0.0
    unmatched_count = 0.0

    for token, recognition_data in sorted(mlfs.items()):
        if token not in confs:
            raise ValueError(u'Token {} not found in the confusions file'.format(token))

        confusion_data = confs[token]

        # Normalize homophones
        presented_words = {to_homophone(confusion_data['speech'].upper(), homophones)}
        reported_words = {to_homophone(k.upper(), homophones) for k in confusion_data['responses'].keys()}

        # Compare the 1st non-control word (normalized homophone) of the 1st hypothesis
        hypothesis_ind = 0
        word_ind = 0
        control_symbols = ['SENT-START', 'SENT-END', 'silence']
        participant_word = [to_homophone(v['symbol'], homophones)
                            for v in recognition_data[hypothesis_ind]
                            if v['symbol'] not in control_symbols][word_ind]

        # Compute the edit script from the presented word to the reported word
        for presented_word in presented_words:
            for reported_word in reported_words:
                best_match = None
                best_score = 0.0
                print u'{} {} {}'.format(presented_word, reported_word, participant_word)

                for presented_pron in lexicon[presented_word]:
                    for reported_pron in lexicon[reported_word]:
                        for participant_pron in lexicon[participant_word]:
                            ignore_symbols = [u'ˈ', u'ˌ']
                            for symbol in ignore_symbols:
                                presented_pron = presented_pron.replace(symbol, u'')
                                reported_pron = reported_pron.replace(symbol, u'')
                                participant_pron = participant_pron.replace(symbol, u'')

                            _, reported_scripts, _, _ = edit_distance.best_transforms(presented_pron.split(),
                                                                                      reported_pron.split())

                            _, participant_scripts, _, _ = edit_distance.best_transforms(presented_pron.split(),
                                                                                         participant_pron.split())

                            [s.print_colors() for s in reported_scripts]
                            [s.print_colors() for s in participant_scripts]

        total_count += 1.0
        if participant_word in presented_words:
            presented_count += 1.0

        if participant_word in reported_words:
            reported_count += 1.0

        if participant_word not in (presented_words | reported_words):
            unmatched_count += 1.0

    print u'presented: {: >6.2%}'.format(presented_count / float(total_count))
    print u'reported:  {: >6.2%}'.format(reported_count / float(total_count))

    return


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description="Evaluate the recognition against confusion data")

    parser.add_argument("-r", "--recognition-directory", dest="recognition_directory", type=str, required=True,
                        help="DIRECTORY where to write the test results", metavar="DIRECTORY")

    parser.add_argument("-c", "--confusions-csv-filename", type=str, required=True,
                        help="FILE where to find the confusions csv data", metavar="FILE")

    parser.add_argument("-l", "--log", dest="log_level", default="info",
                        help="""Log level of the python scripts.""")

    parser.add_argument("-d", "--pronunciation-dictionary-filename", dest="pronunciation_dictionary_filename", type=str,
                        default=None,
                        help="FILE containing the pronunciation of all the words in the language.", metavar="FILE")

    args = parser.parse_args()

    numeric_level = getattr(logging, args.log_level.upper(), None)
    if not isinstance(numeric_level, int):
        raise ValueError('Invalid log level: %s' % args.log_level)

    logging.basicConfig(level=numeric_level)

    evaluate_recognition(args)
