#!/usr/bin/env python
# -*- coding: utf-8 -*-

import logging
import argparse
import codecs
import tempfile
import subprocess
import os
import sys
import unicodedata

import common
import confusions
import phonetics
import inspire

def transcribe_espeak(wordset, args):
    '''Transcribe an input set of words to a lexicon.

    wordset is a set of words
    args.phonetic_format must be 'xsampa' or 'ipa'
    args.language must be supported by eSpeak
    Returns a list of tuples each containing the word and the transcription
    '''

    # Write the word list file to be used as input to the transcriptor
    temp_words_fd, temp_words_file = tempfile.mkstemp()
    words, separated = zip(*sorted(wordset))
    with codecs.open(temp_words_file, 'w', 'utf-8') as f:
        # What we feed to the transcriptor has to be in NFC
        # It does not understand NFD
        f.write('\n\n'.join([unicodedata.normalize('NFC', w) for w in separated]))

    all_languages = subprocess.check_output('espeak --voices', shell = True)
    all_language_codes = sorted(list(set([(l.split()[1], l.split()[3]) # Take the language code and voice name
                                          for l in all_languages.split('\n')[1:] # Ignore the header
                                          if len(l.split())>4]))) # Ignore lines that don't contain language data

    if args.language_code not in zip(*all_language_codes)[0]:
        raise ValueError('Language code not found.  Available languages:\n{}'.format('\n'.join(['{:>12}: {}'.format(*lang) for lang in all_language_codes])))
        
    format_args = '--ipa=3'
        
    # We run the external transcriptor
    transcriptions = subprocess.check_output('espeak -f {wordlist_filename} ' \
                                             '-v {language_code} -q --pho ' \
                                             '--stdout {format_args}'.format(wordlist_filename = temp_words_file,
                                                                             language_code = args.language_code,
                                                                             format_args = format_args),
                                             shell = True)
    os.remove(temp_words_file)
    
    # We decode what we received from the stdout
    transcriptions = transcriptions.decode(sys.stdout.encoding)
    transcriptions = transcriptions.replace(u'_', u' ').split(u'\n')

    transcriptions = map(lambda t: phonetics.convert(t, 'ipa', args.phonetic_format), transcriptions)

    # TODO: change for args.substitutions_filename
    # This should read a file with substitutions to be made and apply them
    if args.simplify:
        transcriptions = map(lambda t: phonetics.simplify(t), transcriptions)
    
    lexicon = zip([w.upper() for w in words], transcriptions)
    return lexicon

if __name__ == '__main__':
    parser = argparse.ArgumentParser(description = "Prepare a spanish dictionary using a rule-based transcriber " \
                                     "given a set of existing dictionaries, word list files and confusion files")

    parser.add_argument("-c", "--confusion-filenames", nargs='+', default = [],
                        help = """CONFUSIONS files containing the confusion information.""")

    parser.add_argument("-t", "--dataset-filenames", nargs='+', default = [],
                        help = """DATASET files containing the confusion information.""")

    parser.add_argument("-d", "--dictionary-filenames", nargs='+', default = [],
                        help = """DICTIONARIES files containing existing dictionaries, pronunciation data is discarded.""")

    parser.add_argument("-w", "--wordlist-filenames", nargs='+', default = [],
                        help = """WORDLISTS files containing a word list.""")
    
    parser.add_argument("-o", "--output-dictionary-filename", required = True,
                        help = """filename where to write the dictionary.""")

    parser.add_argument("-f", "--format", dest = "phonetic_format", default = "xsampa",
                        help = """Phonetic alphabet used: xsampa or ipa.""")

    parser.add_argument("-v", "--language-code", required = True,
                        help = """The language code should be supported by eSpeak.""")

    parser.add_argument("-e", "--simplify", action='store_true',
                        help = """Simplify the resulting transcription.""")
    
    parser.add_argument("-l", "--log", dest = "log_level", default = "info",
                        help = """Log level of the python scripts.""")
    
    args = parser.parse_args()
    
    # Set the log level
    numeric_level = getattr(logging, args.log_level.upper(), None)
    if not isinstance(numeric_level, int):
        raise ValueError('Invalid log level: %s' % args.log_level)

    logging.basicConfig(level=numeric_level)

    wordset = set()
    
    # Parse the confunsions files
    for confusion_filename in args.confusion_filenames:
        confs = confusions.load_confusions(confusion_filename)

        # Get all the words that require transcription
        for conf in confs.values():
            wordset.add(conf.speech.lower())

            for response in conf.responses:
                wordset.add(response.lower())

    # Parse the dataset files
    for dataset_filename in args.dataset_filenames:
        dataset = inspire.load_dataset(dataset_filename)
        for token in dataset['tokens'].values():
            wordset.add(token['speech'].lower())
            wordset |= {w.lower() for w in token['responses'].keys()}

    # Parse the dictionary files
    control_words = {u'silence', u'SENT-START', u'SENT-END'}
    for dictionary_filename in args.dictionary_filenames:
        dictionary = common.parse_dictionary(dictionary_filename)

        wordset = wordset | {word.lower() for word in dictionary if word not in control_words}
        
    # Parse the wordlist files
    for wordlist_filename in args.wordlist_filenames:
        wordlist = common.parse_wordlist(wordlist_filename)

        wordset = wordset | {word.lower() for word in wordlist if word not in control_words}

    words = list(wordset)
    wordset_separated = zip(words, words)
        
    lexicon = transcribe_espeak(wordset_separated, args)
    
    # We write the dictionary
    with codecs.open(args.output_dictionary_filename, 'w', 'utf-8') as f:
        f.write(u'\n'.join([u'{} {}'.format(word[0].replace(' ', '_'), word[1]) for word in lexicon]))
        f.write(u'\n')

    
