#!/usr/bin/env python

import logging
import argparse
import codecs
import re

def load_sampa_chart(sampa_chart_filename = 'data/xsampa_chart.tsv'):
    with codecs.open(sampa_chart_filename, 'r', 'utf-8') as f:
        lines = f.readlines()
        
    xsampa2ipa = {}
    ipa2xsampa = {}

    # Regular expression patterns for:
    #  XXX (or YYY)
    #  XXX or YYY
    or_pattern = re.compile(r'(?:(\S+)(?:\s*\(?or ([^\s)]+)\)?)?)?', re.UNICODE)
    
    for l in lines:
        if l.startswith('X-SAMPA') or l.strip() == '':
            continue
        
        xsampa, ipa = l.split('\t')[:2]

        xsampas = [v for v in or_pattern.search(xsampa).groups() if v is not None]
        ipas = [v for v in or_pattern.search(ipa).groups() if v is not None]
        
        for s in xsampas:
            ss = s.replace('(','').replace(')','')
            
            for i in ipas:
                if ss in xsampa2ipa:
                    logging.debug(u"Ambiguity for X-SAMPA phoneme {} that can map to two different IPA ({} and {}). Using the first one.".format(ss, xsampa2ipa[ss], i))
                    continue
            
                if i in ipa2xsampa:
                    logging.debug(u"Ambiguity for IPA phoneme {} that can map to two different X-SAMPA ({} and {}). Using the first one.".format(i, ipa2xsampa[i], ss))
                    continue
                
                xsampa2ipa[ss] = i
                ipa2xsampa[i] = ss

    return xsampa2ipa

xsampa2ipa = load_sampa_chart()

# TODO: this should go into a data file
# depends on the language and the phoneclass file
xsampa_simplifications = [(':', ''),
                          ('%', ''),
                          (' S', ' s'),
                          ('dZ', 'tS'),
                          ('eI', 'e j'),
                          ('oI', 'o j'),
                          ('aI', 'a j'),
                          ('aU', 'a w'),
                          ('eU', 'e w'),
                          ('en', 'e n'),
                          ('er', 'e r'),
                          ('r', 'rr'),
                          ('4', 'r'),
                          ('E', 'e'),
                          ('"E', 'We'),
                          ('O', 'o'),
                          ('"O', 'Wo'),
                          ('"', 'W'),
                          ('j j', 'jj'),
                          ('j\\', 'jj'),
                          ('  ', ' ')]

def simplify(string):
    for k, v in xsampa_simplifications:
        string = string.replace(k, v)

    return string

def get_compose_mapping(mapping_a, mapping_b):
    return dict([(k, mapping_b.get(v, v)) for k, v in mapping_a.items()])

def get_inverse_mapping(mapping):
    return dict([(v, k) for k, v in mapping.items()])

def get_mapping(from_alfa, to_alfa):
    if from_alfa == to_alfa:
        return {}
    
    elif from_alfa == 'xsampa' and to_alfa == 'ipa':
        return xsampa2ipa
    
    elif from_alfa == 'ipa' and to_alfa == 'xsampa':
        return get_inverse_mapping(xsampa2ipa)

    else:
        raise ValueError('Phonetic conversion from {} to {}. Not implemented.'.format(from_alfa, to_alfa))
    
def convert(string, from_alfa, to_alfa):
    mapping = get_mapping(from_alfa, to_alfa)

    #return u''.join([mapping.get(c, c) for c in string])

    pattern = re.compile('|'.join(re.escape(key) for key in mapping.keys()))
    result = pattern.sub(lambda x: mapping.get(x.group(), x.group()), string)
    return result

if __name__ == '__main__':
    print u'\n'.join([u'{:>5}  --  {}'.format(k, v) for k, v in xsampa2ipa.items()])
    assert(convert(convert('a l G o', 'xsampa', 'ipa'), 'ipa', 'xsampa') == 'a l G o')
    

    
