#!/usr/bin/env python
"""Prepare data for training.

This script create a series of files (feature files and label files) used for training of speech models.

"""

import os
import argparse
import itertools
import json
import random
import shutil
import ast
import logging
import sys
import codecs

import config
import common


def create_triphones_from_pronunciation(pronunciation):
    triphones = []

    if len(pronunciation) > 0:
        offsets = range(-1, len(pronunciation) - 1)

        for k, i in enumerate(offsets):
            context = pronunciation[max(i, 0): i + 3]

            # monophone word
            if k == 0 and k == len(offsets) - 1:
                triphones.append('{}'.format(*context))

            # left boundary
            elif k == 0:
                triphones.append('{}+{}'.format(*context))

            # right boundary
            elif k == len(offsets) - 1:
                triphones.append('{}-{}'.format(*context))

            else:
                triphones.append('{}-{}+{}'.format(*context))

    return set(triphones)


def create_phones_lists(pronunciation_dictionary_filename,
                        phones_filename,
                        phonessp_filename,
                        triphones_filename,
                        triphonessp_filename,
                        boundary_symbols={'sp', 'sil'}):
    logging.debug("Creating symbols (phones/triphones) lists")
    # TODO: Decide whether to use HDMan to do this as explained here:
    # http://www.ee.columbia.edu/ln/LabROSA/doc/HTKBook21/node33.html
    phones = set()
    with codecs.open(pronunciation_dictionary_filename, 'r', 'utf-8') as f:
        for line in f:
            v = line.strip().split(None, 1)[1]
            phones |= set(v.strip().split())

    if '[]' in phones:
        phones.remove('[]')

    phones = list(phones)

    with codecs.open(phones_filename, 'w', 'utf-8') as f:
        f.writelines('{}\n'.format(phone) for phone in phones)

    with codecs.open(phonessp_filename, 'w', 'utf-8') as f:
        f.writelines('{}\n'.format(phone) for phone in (phones + ['sp']))

    triphones = set()
    with codecs.open(pronunciation_dictionary_filename, 'r', 'utf-8') as f:
        for line in f:
            v = line.strip().split(None, 1)[1]
            pronunciation = v.strip().split()
            pronunciation = filter(lambda x: x not in boundary_symbols, pronunciation)
            if '[]' not in pronunciation:
                triphones |= create_triphones_from_pronunciation(pronunciation)

    # TODO: We should also add to the triphone set the set of possible monophones
    # so that during training the prototypes are available.
    # This would fix cases in which very little training data is available
    with codecs.open(triphones_filename, 'w', 'utf-8') as f:
        f.writelines('{}\n'.format(phone) for phone in triphones)

    with codecs.open(triphonessp_filename, 'w', 'utf-8') as f:
        f.writelines('{}\n'.format(phone) for phone in (triphones | {'sil', 'sp'}))


def create_triphone_script(monophone_filename,
                           triphone_filename,
                           script_filename):
    with codecs.open(script_filename, 'w', 'utf-8') as script_file:
        script_file.write('CL {}\n'.format(triphone_filename))

        with codecs.open(monophone_filename, 'r', 'utf-8') as monophone_file:
            for line in monophone_file:
                phone = line.strip()

                if phone != '':
                    script_file.write(
                        'TI T_{phone} {{("{phone}",'
                        '"*-{phone}+*",'
                        '"{phone}+*",'
                        '"*-{phone}").transP}}\n'.format(phone=phone))


def prepare_training_data(audio_directories,
                          speaker_train_instance_count,
                          pronunciation_dictionary_filename,
                          features_directory,
                          feature_extractor_command,
                          feature_extractor_parameters,
                          htk_trace,
                          output_directory):
    output_directory = config.path(output_directory, create=True)
    temp_directory = config.project_path('tmp', create=True)

    dictionary_filename = pronunciation_dictionary_filename

    # Select recordings for training
    trainset_filename = config.path(output_directory, 'trainset.txt')
    logging.debug('Select recording set')
    list_training = common.select_recording_set(audio_directories,
                                                speaker_train_instance_count,
                                                trainset_filename)

    # Check for missing words in the dictionary
    logging.debug(
        "Checking that all words are in dictionary and try to create an extra dictionary for those missing "
        "from TIMIT transcriptions *.wrd *.phn if they are found")
    missing_dictionary_filename = config.path(temp_directory, 'missing_dict.txt')
    common.check_words_in_dictionary(list_training,
                                     dictionary_filename,
                                     proposed_dictionary_filename=missing_dictionary_filename)

    # Create full dictionary
    full_dictionary_filename = config.path(output_directory, 'dictionary.txt')
    logging.debug('Create full dictionary (input dictionary plus missing words from training recordings)')
    common.merge_dictionaries(full_dictionary_filename,
                              [dictionary_filename, missing_dictionary_filename],
                              htk_trace)

    # Create pronunciation dictionary for training
    training_dictionary_filename = config.path(temp_directory, 'training_dict.txt')
    logging.debug('Create recordings dictionary')
    common.create_recordings_dictionary(list_training,
                                        full_dictionary_filename,
                                        training_dictionary_filename,
                                        htk_trace)

    # Create label files for training
    mlf_word_file = config.path(output_directory, 'word.mlf')
    mlf_phone_file = config.path(output_directory, 'word.phn.mlf')
    mlf_phonesp_file = config.path(output_directory, 'word.phn.sp.mlf')
    mlf_tri_file = config.path(output_directory, 'word.tri.mlf')
    mlf_trisp_file = config.path(output_directory, 'word.tri.sp.mlf')
    logging.debug('Create recordings MLFs')
    common.create_recordings_mlf(list_training,
                                 pronunciation_dictionary_filename=training_dictionary_filename,
                                 mlf_word_file=mlf_word_file,
                                 mlf_phone_file=mlf_phone_file,
                                 mlf_phonesp_file=mlf_phonesp_file,
                                 mlf_tri_file=mlf_tri_file,
                                 mlf_trisp_file=mlf_trisp_file,
                                 htk_trace=htk_trace)

    # Create a full phoneme list
    train_phones_filename = config.path(output_directory, 'train_phones_nosp')
    train_phonessp_filename = config.path(output_directory, 'train_phones')
    train_triphones_filename = config.path(output_directory, 'train_triphones_nosp')
    train_triphonessp_filename = config.path(output_directory, 'train_triphones')
    logging.debug('Create language phones lists')
    create_phones_lists(training_dictionary_filename,
                        train_phones_filename,
                        train_phonessp_filename,
                        train_triphones_filename,
                        train_triphonessp_filename)

    phones_filename = config.path(output_directory, 'phones_nosp')
    phonessp_filename = config.path(output_directory, 'phones')
    triphones_filename = config.path(output_directory, 'triphones_nosp')
    triphonessp_filename = config.path(output_directory, 'triphones')
    logging.debug('Create language phones lists')
    create_phones_lists(full_dictionary_filename,
                        phones_filename,
                        phonessp_filename,
                        triphones_filename,
                        triphonessp_filename)

    # Extract the features using the external (and replaceable) extractor
    feature_directory_path = features_directory or os.path.join(output_directory, 'train_features')
    feature_directory = config.path(feature_directory_path, create=True)
    inputoutput_feature_filename = config.path(output_directory, 'train_io_feature_file.scp')
    output_feature_filename = config.path(output_directory, 'train_feature_file.scp')
    feature_extractor = feature_extractor_command
    feature_extractor_parameters = feature_extractor_parameters
    logging.debug('Create feature files')
    common.create_feature_files(list_training,
                                feature_directory,
                                inputoutput_feature_filename,
                                output_feature_filename,
                                feature_extractor,
                                feature_extractor_parameters)

    # Save the arguments
    logging.debug('Save parameters')
    parameters_output_filename = config.path(output_directory, 'parameters.json')

    parameters = {'audio_directories': audio_directories,
                  'list_train_recordings_filename': trainset_filename,
                  'dictionary_filename': full_dictionary_filename,
                  'features_directory': features_directory,
                  'feature_extractor_command': feature_extractor_command,
                  'feature_extractor_parameters': feature_extractor_parameters,
                  'data_preparation_directory': output_directory,
                  'label_word_filename': mlf_word_file,
                  'label_phone_filename': mlf_phone_file,
                  'label_phone_shortpause_filename': mlf_phonesp_file,
                  'label_triphone_filename': mlf_tri_file,
                  'label_triphone_shortpause_filename': mlf_trisp_file,
                  'list_train_phones_filename': train_phones_filename,
                  'list_train_phones_shortpause_filename': train_phonessp_filename,
                  'list_phones_filename': phones_filename,
                  'list_phones_shortpause_filename': phonessp_filename,
                  'list_train_triphones_filename': train_triphones_filename,
                  'list_train_triphones_shortpause_filename': train_triphonessp_filename,
                  'list_triphones_filename': triphones_filename,
                  'list_triphones_shortpause_filename': triphonessp_filename,
                  'list_train_feature_files_filename': output_feature_filename,
    }

    with codecs.open(parameters_output_filename, 'w', 'utf-8') as parameters_file:
        json.dump(parameters, parameters_file, indent=2)

    return


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description="Create a baseline intelligibility model for the INSPIRE Challenge")

    parser.add_argument("-r", "--random-seed", dest="random_seed", type=int,
                        help="Seed for the random sampling of training instances.")

    parser.add_argument("-o", "--output-directory", dest="output_directory", type=str, default='models',
                        help="DIRECTORY where to write the training results (models, parameters,...)",
                        metavar="DIRECTORY")

    parser.add_argument("-f", "--features-directory", dest="features_directory", type=str, default=None,
                        help="DIRECTORY where to write the training features. "
                             "If not set they are written to directory named "
                             "train_features inside the output directory.",
                        metavar="DIRECTORY")

    parser.add_argument("-a", "--audio-dir", dest="audio_dir", nargs='+', type=str,
                        help="""DIRECTORY containing the recordings used for training.
                        All wave files are considered for training.""",
                        metavar="DIRECTORY")

    parser.add_argument("-d", "--pronunciation-dictionary-file", dest="pronunciation_dictionary_filename", type=str,
                        required=True,
                        help="FILE containing the pronunciation of all the words in the language.", metavar="FILE")

    parser.add_argument("-n", "--speaker-train-instance-count", dest="speaker_train_instance_count",
                        type=ast.literal_eval, default=None,
                        help="""PYTHON_LITERAL is a dictionary in JSON format specifying
                        the amount of training instances for each speaker.
                        Speakers not appearing in the dictionary will not be used for training.
                        e.g. "{1: 1000, 2: 500}" will use 1000 instances of speaker s1,
                        500 of speaker s2 and none of other speakers.
                        If the argument is not set then all the instances of all speakers are used for training.""",
                        metavar="PYTHON_LITERAL")

    parser.add_argument("-x", "--feature-extractor-command", dest="feature_extractor_command",
                        default=config.project_path("extract_features_htkmfcc.py"),
                        help="""CMD command to be called with a single space
                        separated inputOutputFile argument in order
                        to extract features from input wav files into output feature files.""", metavar="CMD")

    parser.add_argument("-xp", "--feature-extractor-parameters", dest="feature_extractor_parameters", default="",
                        help="""STR commandline argument string to be passed to the feature extractor command.""",
                        metavar="STR")

    parser.add_argument("-T", "--htk-trace", action='store', type=int, default=0,
                        help="""Verbose output and trace from HTK commands.""")

    parser.add_argument("-l", "--log", dest="log_level", default="info",
                        help="""Log level of the python scripts.""")

    args = parser.parse_args()

    # Set the log level
    numeric_level = getattr(logging, args.log_level.upper(), None)
    if not isinstance(numeric_level, int):
        raise ValueError('Invalid log level: %s' % args.log_level)

    logging.basicConfig(level=numeric_level)

    # Set the random seed
    if args.random_seed is not None:
        random.seed(args.random_seed)

    # Get the absolute path of the feature extractor executable
    feature_extractor_command = config.which(args.feature_extractor_command)
    if not feature_extractor_command:
        logging.error('Invalid parameter for --feature-extractor-command={}. '
                      'Not valid command or executable.'.format(args.feature_extractor_command))
        parser.print_help()
        sys.exit(1)

    feature_extractor_command = os.path.abspath(feature_extractor_command)

    prepare_training_data(args.audio_dir,
                          args.speaker_train_instance_count,
                          args.pronunciation_dictionary_filename,
                          args.features_directory,
                          feature_extractor_command,
                          args.feature_extractor_parameters,
                          args.htk_trace,
                          args.output_directory)

