#!/usr/bin/env python

# Script for reestimating a model (useful when training with custom features)


import os
import argparse
import json
import logging
import codecs
import itertools
import shutil
import shlex
import numpy as np

import config
import common
import htk


def _expand_feature_files(old_feature_filenames,
                          new_feature_filenames,
                          feature_directory,
                          old_dims,
                          new_dims,
                          old_new_feature_filename):

    next_dims = max(old_dims, new_dims)

    with codecs.open(old_new_feature_filename, 'w', 'utf-8') as old_new_feature_file:
        with codecs.open(new_feature_filenames, 'r', 'utf-8') as new_feature_file:
            with codecs.open(old_feature_filenames, 'r', 'utf-8') as old_feature_file:
                for old_line, new_line in itertools.izip(old_feature_file, new_feature_file):
                    new_feature_filename = common.shlex_split(new_line)[0]
                    old_feature_filename = common.shlex_split(old_line)[0]

                    meta_new = htk.load_parameter(new_feature_filename, only_header=True)
                    meta_old = htk.load_parameter(old_feature_filename, only_header=True)

                    if meta_new.samp_period != meta_old.samp_period:
                        raise ValueError(u'Different feature vector hop sizes'
                                         u' in {} ({}) and {} ({}). '
                                         u'They must have same hop size.'.format(old_feature_filename,
                                                                                 meta_old.samp_period,
                                                                                 new_feature_filename,
                                                                                 meta_new.samp_period))

                    if meta_new.n_samples != meta_old.n_samples:
                        logging.warning(u'Different number of feature vectors'
                                        u' in {} ({}) and {} ({}). '
                                        u'Keeping the smallest.'.format(old_feature_filename,
                                                                        meta_old.n_samples,
                                                                        new_feature_filename,
                                                                        meta_new.n_samples))

                    next_sample_count = min(meta_new.n_samples, meta_old.n_samples)

                    next_old_feature_filename = _expand_feature_file(old_feature_filename,
                                                                     feature_directory,
                                                                     old_dims,
                                                                     next_dims,
                                                                     next_sample_count)
                    next_new_feature_filename = _expand_feature_file(new_feature_filename,
                                                                     feature_directory,
                                                                     new_dims,
                                                                     next_dims,
                                                                     next_sample_count)

                    old_new_feature_file.write(u'"{}" "{}"\n'.format(next_old_feature_filename,
                                                                     next_new_feature_filename))

    return


def _expand_feature_file(prev_parameter_file,
                         next_directory,
                         prev_dims,
                         next_dims,
                         next_sample_count):
    if prev_dims >= next_dims:
        return prev_parameter_file

    next_feature_filename = config.path(next_directory,
                                        os.path.basename(prev_parameter_file))

    param_new = htk.load_parameter(prev_parameter_file)

    param_new = _expand_parameter(param_new, next_dims, next_sample_count)

    htk.save_parameter(param_new, next_feature_filename)

    return next_feature_filename


def _expand_parameter(param, next_dims, next_sample_count):
    param = param._replace(meta=param.meta._replace(samp_size=next_dims*4))

    new_samples = np.zeros((next_sample_count, next_dims))
    new_samples[:, :param.samples.shape[1]] = param.samples[:next_sample_count, :]
    param = param._replace(samples=new_samples)

    return param


def single_pass_reestimate(input_directory,
                           old_feature_filenames,
                           new_feature_filenames,
                           label_filename,
                           output_directory,
                           htk_trace):
    models_directory = config.path('tmp/models/', create=True)
    feature_directory = config.path(models_directory, 'retrain_features', create=True)
    output_directory = config.path(output_directory, create=True)

    # Check if we need to expand the models
    with open(old_feature_filenames) as old_feature_file:
        example_old_feature_filename = shlex.split(old_feature_file.readline())[0]

    with open(new_feature_filenames) as new_feature_file:
        example_new_feature_filename = shlex.split(new_feature_file.readline())[0]

    meta_old = htk.load_parameter(example_old_feature_filename, only_header=True)
    meta_new = htk.load_parameter(example_new_feature_filename, only_header=True)

    old_dims = int(meta_old.samp_size / 4)
    new_dims = int(meta_new.samp_size / 4)

    old_new_feature_filename = config.path(output_directory, 'old_new_feature_file.scp')
    _expand_feature_files(old_feature_filenames,
                          new_feature_filenames,
                          feature_directory,
                          old_dims,
                          new_dims,
                          old_new_feature_filename)

    prev_model_directory = config.path(input_directory)

    if old_dims < new_dims:
        # Expand the old model
        next_model_directory = config.path(models_directory, 'spr_expanded', create=True)

        script_filename = config.path(models_directory, 'spr_expansion.hed')
        with codecs.open(script_filename, 'w', 'utf-8') as f:
            f.write(u'SW 1 {}\n'.format(new_dims))

        config.htk_command('HHed'
                           ' -T {}'
                           ' -H {}'
                           ' -H {}'
                           ' -M {}'
                           ' {}'
                           ' {}'.format(htk_trace,
                                        os.path.join(prev_model_directory, 'macros'),
                                        os.path.join(prev_model_directory, 'hmmdefs'),
                                        next_model_directory,
                                        script_filename,
                                        os.path.join(prev_model_directory, 'symbollist')))

        shutil.copy(os.path.join(prev_model_directory, 'symbollist'),
                    os.path.join(next_model_directory, 'symbollist'))

        prev_model_directory = next_model_directory

    next_model_directory = config.path(models_directory, 'spr', create=True)

    # Perform the reestimation
    cmd = 'HERest -r' \
          ' -T {}' \
          ' -S {}' \
          ' -I {}' \
          ' -H {}' \
          ' -H {}' \
          ' -M {}' \
          ' {}'.format(htk_trace,
                       old_new_feature_filename,
                       label_filename,
                       os.path.join(prev_model_directory, 'macros'),
                       os.path.join(prev_model_directory, 'hmmdefs'),
                       next_model_directory,
                       os.path.join(prev_model_directory, 'symbollist'))
    logging.info('HTK command:\n{}'.format(cmd))
    config.htk_command(cmd)

    shutil.copy(os.path.join(prev_model_directory, 'symbollist'),
                os.path.join(next_model_directory, 'symbollist'))

    prev_model_directory = next_model_directory

    if old_dims > new_dims:
        # Contract the new model
        next_model_directory = config.path(models_directory, 'spr_contracted', create=True)

        script_filename = config.path(models_directory, 'spr_contraction.hed')
        with codecs.open(script_filename, 'w', 'utf-8') as f:
            f.write(u'SW 1 {}\n'.format(new_dims))

        config.htk_command('HHed'
                           ' -T {}'
                           ' -H {}'
                           ' -H {}'
                           ' -M {}'
                           ' {}'
                           ' {}'.format(htk_trace,
                                        os.path.join(prev_model_directory, 'macros'),
                                        os.path.join(prev_model_directory, 'hmmdefs'),
                                        next_model_directory,
                                        script_filename,
                                        os.path.join(prev_model_directory, 'symbollist')))

        shutil.copy(os.path.join(prev_model_directory, 'symbollist'),
                    os.path.join(next_model_directory, 'symbollist'))

        prev_model_directory = next_model_directory

    shutil.copy(os.path.join(next_model_directory, 'macros'),
                os.path.join(output_directory, 'macros'))
    shutil.copy(os.path.join(next_model_directory, 'hmmdefs'),
                os.path.join(output_directory, 'hmmdefs'))
    shutil.copy(os.path.join(prev_model_directory, 'symbollist'),
                os.path.join(output_directory, 'symbollist'))


def train_reestimate(input_directory,
                     model_directory,
                     feature_extractor_command,
                     feature_extractor_parameters,
                     htk_trace):
    output_directory = config.path(model_directory, create=True)

    parameters_filename = os.path.join(input_directory, 'parameters.json')
    with open(parameters_filename, 'r') as parameters_file:
        parameters = json.load(parameters_file)

    list_training_filename = parameters['list_train_recordings_filename']
    with codecs.open(list_training_filename, 'r', 'utf-8') as list_training_file:
        list_training = [line.strip() for line in list_training_file]

    # Extract the features using the external (and replaceable) extractor
    feature_directory_path = os.path.join(output_directory, 'train_features')
    feature_directory = config.path(feature_directory_path, create=True)
    inputoutput_feature_filename = config.path(output_directory, 'train_io_feature_file.scp')
    new_feature_filenames = config.path(output_directory, 'train_feature_file.scp')
    logging.debug('Create feature files')
    common.create_feature_files(list_training,
                                feature_directory,
                                inputoutput_feature_filename,
                                new_feature_filenames,
                                feature_extractor_command,
                                feature_extractor_parameters)

    old_feature_filenames = parameters['list_train_feature_files_filename']

    parameters['features_directory'] = feature_directory
    parameters['feature_extractor_command'] = feature_extractor_command
    parameters['feature_extractor_parameters'] = feature_extractor_parameters
    parameters['list_train_feature_files_filename'] = new_feature_filenames

    single_pass_reestimate(input_directory,
                           old_feature_filenames,
                           new_feature_filenames,
                           parameters['label_triphone_shortpause_filename'],
                           output_directory,
                           htk_trace)

    shutil.copy(os.path.join(input_directory, 'tri.stats'),
                os.path.join(output_directory, 'tri.stats'))

    # Reestimate the speech prior
    if 'label_speech_prior_filename' in parameters:
        label_speech_prior_filename = parameters['label_speech_prior_filename']
        single_pass_reestimate(config.path(input_directory, 'speech_prior'),
                               old_feature_filenames,
                               new_feature_filenames,
                               label_speech_prior_filename,
                               config.path(output_directory, 'speech_prior'),
                               htk_trace)

    # Save the arguments
    logging.debug('Save parameters')
    parameters_output_filename = config.path(output_directory, 'parameters.json')

    with codecs.open(parameters_output_filename, 'w', 'utf-8') as parameters_file:
        json.dump(parameters, parameters_file, indent=2)

    return


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description="Reestimate a model")

    parser.add_argument("-i", "--input-model", dest="input_directory", type=str,
                        help="DIRECTORY where to find the prepared training data "
                             "(parameters, feature, label and symbol files)",
                        metavar="DIRECTORY")

    parser.add_argument("-x", "--feature-extractor-command", dest="feature_extractor_command",
                        default=config.project_path("extract_features_htkmfcc.py"),
                        help="""CMD command to be called with a single space
                        separated inputOutputFile argument in order
                        to extract features from input wav files into output feature files.""", metavar="CMD")

    parser.add_argument("-xp", "--feature-extractor-parameters", dest="feature_extractor_parameters", default="",
                        help="""STR commandline argument string to be passed to the feature extractor command.""",
                        metavar="STR")

    parser.add_argument("-o", "--model-directory", dest="model_directory", type=str, default='model',
                        help="DIRECTORY where to write the training results (parameters, models)",
                        metavar="DIRECTORY")

    parser.add_argument("-T", "--htk-trace", action='store', type=int, default=0,
                        help="""Verbose output and trace from HTK commands.""")

    parser.add_argument("-l", "--log", dest="log_level", default="info",
                        help="""Log level of the python scripts.""")

    args = parser.parse_args()

    # Set the log level
    numeric_level = getattr(logging, args.log_level.upper(), None)
    if not isinstance(numeric_level, int):
        raise ValueError('Invalid log level: %s' % args.log_level)

    logging.basicConfig(level=numeric_level)

    train_reestimate(args.input_directory,
                     args.model_directory,
                     args.feature_extractor_command,
                     args.feature_extractor_parameters,
                     args.htk_trace)

