#!/usr/bin/env python

# Script for creating an ASR model

import os
import argparse
import ast
import random
import logging
import shutil
import codecs

import config
import common


def generate_cmllr_transforms(model_directory,
                              output_feature_filename,
                              mlf_trisp_file,
                              symbollist_filename,
                              tristats_filename,
                              regressiontree_filename,
                              output_directory,
                              htk_trace=0):
    # Create regressiontree script
    regressiontree_template = """LS "{tristats_filename}"
RC 64 "rtree"
    """
    with open(regressiontree_filename, 'w') as f:
        f.write(regressiontree_template.format(tristats_filename=tristats_filename))

    classes_directory = config.path(output_directory, 'classes', create=True)
    transforms_directory = config.path(output_directory, 'xforms', create=True)

    # Copy the global classes file
    global_class_filename = config.project_path('etc', 'classes', 'global')
    shutil.copy(global_class_filename, classes_directory)

    config_filename = config.path('etc', 'tri.conf')
    configglobal_filename = config.path('etc', 'config.global.cmllr')
    configrc_filename = config.path('etc', 'config.rc.cmllr')

    # Build a regression class tree and store it along with a set of baseclasses
    logging.info("Build a regression class tree and store it along with a set of baseclasses")
    config.htk_command("HHEd -C {} -B -H {} "
                       "-H {} -M {} {} {}".format(config.project_path('etc', 'config_empty'),
                                                  os.path.join(model_directory, 'macros'),
                                                  os.path.join(model_directory, 'hmmdefs'),
                                                  classes_directory,
                                                  regressiontree_filename,
                                                  symbollist_filename))

    # Estimate transforms
    # Step1: global adaptation
    logging.info("Estimate transforms, step 1: global adaptation")
    config.htk_command(
        "HERest -C {} -T {} -C {} -C {} -S {} -I {} -H {} "
        "-u a -H {} -K {} mllr1 -J {} -h '*/*_s%.mfc' {}".format(
            config.project_path('etc', 'config_empty'),
            htk_trace,
            config_filename,
            configglobal_filename,
            output_feature_filename,
            mlf_trisp_file,
            os.path.join(model_directory, 'macros'),
            os.path.join(model_directory, 'hmmdefs'),
            transforms_directory,
            classes_directory,
            symbollist_filename))

    # Step2: model set transformation
    logging.info("Estimate transforms, step 2: model set adaptation")
    config.htk_command(
        "HERest -C {} -a -A -D -T {} -C {} -C {} -I {} -S {} -H {} "
        "-u a -H {} -J {} mllr1 -K {} mllr2 -J {} -h '*/*_s%.mfc' {}".format(
            config.project_path('etc', 'config_empty'),
            htk_trace,
            config_filename,
            configrc_filename,
            mlf_trisp_file,
            output_feature_filename,
            os.path.join(model_directory, 'macros'),
            os.path.join(model_directory, 'hmmdefs'),
            transforms_directory,
            transforms_directory,
            classes_directory,
            symbollist_filename))

    return


def adapt_cmllr_models(args):
    input_directory = args.models_directory
    temp_directory = config.project_path('tmp', create=True)

    output_directory = config.path(args.models_directory,
                                   args.model_name, 'cmllr',
                                   args.transform_name, create=True)

    train_arguments_filename = config.path(input_directory, 'train_arguments.json')
    args = common.load_train_arguments(train_arguments_filename, args)

    list_words = common.parse_wordlist(args.list_words_filename)

    # Select recordings for adaptation
    trainset_filename = config.path(input_directory, 'trainset.txt')
    list_training = []
    with codecs.open(trainset_filename, 'r', 'utf-8') as f:
        list_training = [l.strip() for l in f.readlines()]

    adaptset_filename = config.path(output_directory, 'adaptset.txt')
    list_adaptation = common.select_recording_set(args.audio_dir,
                                                  args.speaker_adapt_instance_count,
                                                  adaptset_filename,
                                                  possible_words=list_words,
                                                  unaccepted_recordings=list_training
                                                  if args.filter_training_recordings else [])

    pronunciation_adaptation_dictionary_filename = config.path(temp_directory, 'adapt_dict.txt')
    logging.info("Create recordings dictionary")
    common.create_recordings_dictionary(list_adaptation,
                                        args.pronunciation_dictionary_filename,
                                        pronunciation_adaptation_dictionary_filename,
                                        args)


    # Extract the features using the external (and replaceable) extractor
    logging.info("Extract features")
    feature_directory = config.path(temp_directory, 'adapt_features', create=True)
    inputoutput_feature_filename = config.path(temp_directory, 'adapt_io_feature_file.scp')
    output_feature_filename = config.path(temp_directory, 'adapt_feature_file.scp')
    feature_extractor = args.feature_extractor_command
    feature_extractor_parameters = args.feature_extractor_parameters
    logging.debug('Create feature files')
    common.create_feature_files(list_adaptation,
                                feature_directory,
                                inputoutput_feature_filename,
                                output_feature_filename,
                                feature_extractor,
                                feature_extractor_parameters)

    # Create label files for adaptation
    symbollist_filename = config.path(args.models_directory, 'tiedlist')
    model_directory = config.path(args.models_directory, args.model_name)

    mlf_word_file = config.path(temp_directory, 'adapt_word.mlf')
    mlf_phone_file = config.path(temp_directory, 'adapt_word.phn.mlf')
    mlf_phonesp_file = config.path(temp_directory, 'adapt_word.phn.sp.mlf')
    mlf_tri_file = config.path(temp_directory, 'adapt_word.tri.mlf')
    mlf_trisp_file = config.path(temp_directory, 'adapt_word.tri.sp.mlf')
    phone_script_filename = config.path('etc', 'mkphones1_ad.led')
    phonesp_script_filename = config.path('etc', 'mkphones1sp_ad.led')
    logging.info("Create recordings Master Label Files (mlf)")
    common.create_recordings_mlf(list_adaptation,
                                 pronunciation_dictionary_filename=pronunciation_adaptation_dictionary_filename,
                                 force_align_model_directory=model_directory,
                                 force_align_output_feature_filename=output_feature_filename,
                                 force_align_symbollist_filename=symbollist_filename,
                                 mlf_word_file=mlf_word_file,
                                 mlf_phone_file=mlf_phone_file,
                                 mlf_phonesp_file=mlf_phonesp_file,
                                 mlf_tri_file=mlf_tri_file,
                                 mlf_trisp_file=mlf_trisp_file,
                                 phone_script_filename=phone_script_filename,
                                 phonesp_script_filename=phonesp_script_filename,
                                 htk_trace=args.htk_trace)

    # Create the adaptation transforms using the source model, the forced alignment
    regressiontree_filename = config.path(temp_directory, 'regtree.hed')
    tristats_filename = config.path(input_directory, 'tri.stats')
    logging.info("Generate CMLLR transforms")
    generate_cmllr_transforms(model_directory,
                              output_feature_filename,
                              mlf_trisp_file,
                              symbollist_filename,
                              tristats_filename,
                              regressiontree_filename,
                              output_directory,
                              htk_trace=args.htk_trace)

    return


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description="Create a baseline intelligibility model for the INSPIRE Challenge")

    parser.add_argument("-r", "--random-seed", dest="random_seed", type=int,
                        help="Seed for the random sampling of training instances.")

    parser.add_argument("-i", "--models-directory", dest="models_directory", type=str, default='models',
                        help="DIRECTORY from where to read the training results (models, parameters,...)",
                        metavar="DIRECTORY")

    parser.add_argument("-m", "--model-name", dest="model_name", type=str, default='tstri.5mix.4',
                        help="MODELNAME is the name of model to adapt", metavar="MODELNAME")

    parser.add_argument("-t", "--transform-name", dest="transform_name", type=str, default='adaptation',
                        help="ADAPTATIONNAME is the name of model to adapt", metavar="ADAPTATIONNAME")

    parser.add_argument("-o", "--results-directory", dest="results_directory", type=str, default='results',
                        help="DIRECTORY where to write the test results", metavar="DIRECTORY")

    parser.add_argument("-w", "--list-words-filename", dest="list_words_filename", type=str,
                        help="FILE containing list of all possible language words", metavar="FILE")

    parser.add_argument("-a", "--audio-dir", dest="audio_dir", nargs='*',
                        help="""DIRECTORY containing the recordings used for adaptation.
                        Wav filenames must end in X_sY.wav where X is the word
                        and Y is the numeric ID of the spearker.""",
                        metavar="DIRECTORY")

    parser.add_argument("-d", "--pronunciation-dictionary-filename", dest="pronunciation_dictionary_filename", type=str,
                        help="FILE containing the pronunciation of all the words in the language.", metavar="FILE")

    parser.add_argument("--filter-training-recordings", dest="filter_training_recordings", default=False,
                        action='store_true',
                        help="Set this flag to False to avoid using training recordings for testing.")

    parser.add_argument("-n", "--speaker-adapt-instance-count", dest="speaker_adapt_instance_count",
                        type=ast.literal_eval, default=None,
                        help="""PYTHON_LITERAL is a dictionary in Python literal format specifying
                        the amount of adaptation instances for each speaker.
                        Speakers not appearing in the dictionary will not be used for adaptation.
                        e.g. "{1: 1000, 2: 500}" will use 1000 instances of speaker s1, 500 of speaker s2
                        and none of other speakers.
                        If the argument is not set then all the instances of all speakers are used for adaptation
                        (see argument --filter-training-recordings to control the exclusion of
                        recordings used for training).""",
                        metavar="PYTHON_LITERAL")

    parser.add_argument("-x", "--feature-extractor-command", dest="feature_extractor_command", default=None,
                        help="""CMD command to be called with a single space separated inputOutputFile argument in order
                        to extract features from input wav files into output feature files.""", metavar="CMD")

    parser.add_argument("-xp", "--feature-extractor-parameters", dest="feature_extractor_parameters", default="",
                        help="""STR commandline argument string to be passed to the feature extractor command.""",
                        metavar="STR")

    parser.add_argument("-T", "--htk-trace", action='store', type=int, default=0,
                        help="""Verbose output and trace from HTK commands.""")

    parser.add_argument("-l", "--log", dest="log_level", default="info",
                        help="""Log level of the python scripts.""")

    args = parser.parse_args()

    numeric_level = getattr(logging, args.log_level.upper(), None)
    if not isinstance(numeric_level, int):
        raise ValueError('Invalid log level: {}'.format(args.log_level))

    logging.basicConfig(level=numeric_level)

    if args.random_seed is not None:
        random.seed(args.random_seed)

    adapt_cmllr_models(args)
