#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""evaluation.py
Perform the evaluations of the INSPIRE Challenge.

Usage:
    evaluation.py <dataset_file> <participant_file> [options]
    evaluation.py --help

Options:
    <dataset_file>      JSON file containing the stimuli and confusions information
    <participant_file>  JSON file containing a participant's result
    -d <dictionary_file> --dictionary=<dictionary_file>   dictionary file containing the pronunciations [default: data/spanish_dictionary.txt].
    -h, --help          print this help screen
"""

import docopt
import json

import common
import edit_distance

def evaluate_recognition(args):
    dictionary_filename = args.pronunciation_dictionary_filename or os.path.join(args.recognition_directory,
                                                                                 'dictionary.txt')
    recognition_filename = os.path.join(args.recognition_directory, 'recognition.mlf')

    lexicon = common.parse_dictionary(dictionary_filename)
    lexicon = dict([(k.upper(), list(v)) for k, v in lexicon.items()])

    homophones = common.get_dictionary_homophones(lexicon)

    confs = confusions.load_confusions(args.confusions_csv_filename)
    confs = dict([(u'T_{}_M'.format(conf.token_id), conf.__dict__) for conf in confs.values()])

    mlfs = htk.load_mlf(recognition_filename, utf8_normalization=common.UTF8_NORMALIZATION)

    # Take the first hypothesis for each file in the MLF
    mlfs = dict([(os.path.splitext(os.path.basename(k))[0], v) for k, v in mlfs.items()])

    total_count = 0.0
    presented_count = 0.0
    reported_count = 0.0
    unmatched_count = 0.0

    for token, recognition_data in sorted(mlfs.items()):
        if token not in confs:
            raise ValueError(u'Token {} not found in the confusions file'.format(token))

        confusion_data = confs[token]

        # Normalize homophones
        presented_words = {to_homophone(confusion_data['speech'].upper(), homophones)}
        reported_words = {to_homophone(k.upper(), homophones) for k in confusion_data['responses'].keys()}

        # Compare the 1st non-control word (normalized homophone) of the 1st hypothesis
        hypothesis_ind = 0
        word_ind = 0
        control_symbols = ['SENT-START', 'SENT-END', 'silence']
        participant_word = [to_homophone(v['symbol'], homophones)
                            for v in recognition_data[hypothesis_ind]
                            if v['symbol'] not in control_symbols][word_ind]

        # Compute the edit script from the presented word to the reported word
        for presented_word in presented_words:
            for reported_word in reported_words:
                best_match = None
                best_score = 0.0
                print u'{} {} {}'.format(presented_word, reported_word, participant_word)

                for presented_pron in lexicon[presented_word]:
                    for reported_pron in lexicon[reported_word]:
                        for participant_pron in lexicon[participant_word]:
                            ignore_symbols = [u'ˈ', u'ˌ']
                            for symbol in ignore_symbols:
                                presented_pron = presented_pron.replace(symbol, u'')
                                reported_pron = reported_pron.replace(symbol, u'')
                                participant_pron = participant_pron.replace(symbol, u'')

                            _, reported_scripts, _, _ = edit_distance.best_transforms(presented_pron.split(),
                                                                                      reported_pron.split())

                            _, participant_scripts, _, _ = edit_distance.best_transforms(presented_pron.split(),
                                                                                         participant_pron.split())

                            [s.print_colors() for s in reported_scripts]
                            [s.print_colors() for s in participant_scripts]

        total_count += 1.0
        if participant_word in presented_words:
            presented_count += 1.0

        if participant_word in reported_words:
            reported_count += 1.0

        if participant_word not in (presented_words | reported_words):
            unmatched_count += 1.0

    print u'presented: {: >6.2%}'.format(presented_count / float(total_count))
    print u'reported:  {: >6.2%}'.format(reported_count / float(total_count))

    return


def evaluate(participant, dataset, lexicon):
    for token in dataset:
        results = filter(lambda x: x['token_id'] == token['token_id'], participant)

        if len(results) != 1:
            raise ValueError('The participant produced an amount of results ({})'
                             ' different from 1 to token {}'.format(len(results),
                                                                    token['token_id']))




def main():
    arguments = docopt.docopt(__doc__)

    dataset = json.load(open(arguments['<dataset_file>']))
    participant = json.load(open(arguments['<participant_file>']))
    lexicon = common.parse_dictionary(arguments['--dictionary'])

    evaluate(participant, dataset, lexicon)

if __name__ == '__main__':
    main()