#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""inspire.py
Collection of functions useful for the INSPIRE challenge.

"""
from __future__ import absolute_import, division, print_function, unicode_literals

import json
import itertools
import logging
import pprint as pp
import jsonschema
import numpy as np
import scipy.misc
import collections

import common
import edit_distance

UTF8_NORMALIZATION = 'NFD'


class MyPrettyPrinter(pp.PrettyPrinter):
    _escape = dict((q, dict(((c, unicode(repr(chr(c)))[1:-1])
                             for c in range(32) + [ord('\\')] +
                             range(128, 161)),
                            **{ord(q): u'\\' + q}))
                   for q in ["'", '"'])

    def format(self, object, context, maxlevels, level):
        if type(object) is unicode:
            q = "'" if "'" not in object or '"' in object \
                else '"'
            return ("u" + q + object.translate(self._escape[q]) +
                    q, True, False)
        return pp.PrettyPrinter.format(
            self, object, context, maxlevels, level)


def pprint(obj, **kwargs):
    printer = MyPrettyPrinter(**kwargs)
    printer.pprint(obj)


class Multinomial(object):
    def __init__(self, params, exact=False):
        self._exact = exact
        self._params = params

    def pmf(self, counts):
        return np.prod(np.power(self._params, counts)) * np.exp(self._log_multinomial_coeff(counts))

    def log_pmf(self, counts):
        olderr = np.seterr(divide='ignore', invalid='ignore')
        result = np.dot(counts, np.nan_to_num(np.log(self._params))) + self._log_multinomial_coeff(counts)
        np.seterr(**olderr)
        return result

    def _log_multinomial_coeff(self, counts):
        return self._log_factorial(np.sum(counts)) - np.nansum(self._log_factorial(counts))

    def _log_factorial(self, num):
        if self._exact and not np.isscalar(num):
            result = np.zeros_like(num, dtype=np.float)
            for i, v in enumerate(num):
                v = int(v)
                for j in xrange(1, v + 1):
                    result[i] += np.log(j)

            return result

        else:
            return np.log(scipy.misc.factorial(np.array(num)))


class Dataset(dict):
    def __init__(self, *args, **kwargs):
        self.update(*args, **kwargs)

    def __repr__(self):
        return json.dumps(self, indent=2)


class Lexicon(dict):
    def __init__(self, *args, **kwargs):
        self.update(*args, **kwargs)

    def __getitem__(self, key):
        return dict.__getitem__(self, key.upper())

    def __setitem__(self, key, value):
        return dict.__setitem__(self, key.upper(), value)

    def __contains__(self, key):
        return dict.__contains__(self, key.upper())

    def __repr__(self):
        return json.dumps(self, indent=2)


class Submission(dict):
    def __init__(self, authors=[], email='', challenge_edition='', lexicon=None):
        self.update({'metadata': {'authors': authors,
                                  'email': email,
                                  'challenge_edition': challenge_edition},
                     'lexicon': lexicon,
                     'tokens': {}})

        with open('etc/submission_schema.json') as f:
            self.schema = json.load(f)

        self.validate()

    def where_task(self, token_id, confusion_probability):
        self['tokens'].setdefault(token_id, {})['where'] = {'confusion_probability': list(confusion_probability)}

    def what_task(self, token_id, index, phonemes, phonemes_probability):
        self['tokens'].setdefault(token_id, {}) \
            .setdefault('what', {}) \
            .setdefault('per_index_phonemes_probability', {}) \
            .setdefault(str(index), {})[phonemes] = phonemes_probability

    def full_task(self, token_id, pronunciation, pronunciation_probability):
        self['tokens'].setdefault(token_id, {}) \
            .setdefault('full', {}) \
            .setdefault('pronunciations_probability', {})[pronunciation] = pronunciation_probability

    def validate(self):
        jsonschema.validate(self, self.schema)

    def __repr__(self):
        return json.dumps(self, indent=2)


def load_wordlist(wordlist_filename):
    return set(common.parse_wordlist(wordlist_filename))


def load_lexicon(lexicon_filename):
    return Lexicon({k: [pron.split(' ') for pron in v]
                    for k, v in common.parse_dictionary(lexicon_filename).items()})


def load_dataset(dataset_filename):
    with open(dataset_filename) as dataset_file:
        return Dataset(json.load(dataset_file))

    return {}


def save_submission(submission, submission_filename):
    with open(submission_filename, 'w') as f:
        json.dump(submission, f, indent=2)

    return


def _combine_dicts(*args):
    all = []
    for arg in args:
        all += list(arg.items())

    return dict(all)


def get_edit_scripts(pron_a, pron_b):
    """Get the edit scripts to transform between two given pronunciations.

    :param pron_a: Source pronunciation as list of strings, each string corresponding to a phoneme
    :param pron_b: Target pronunciation as list of strings, each string corresponding to a phoneme
    :return: List of edit scripts.  Each edit script is represented as a list of operations,
                where each operation is a dictionary.
    """
    distance, scripts, costs, ops = edit_distance.best_transforms(pron_a, pron_b)
    return [script.to_primitive() for script in scripts]


def print_edit_script(edit_script):
    """Print an edit script to the terminal.

    :param edit_script: The edit script as a list of operations, where each operation is a dictionary.
    """
    print('{}\n{}'.format(*edit_script_to_strings(edit_script)))


def edit_script_to_strings(edit_script, use_colors=True):
    """Convert an edit script to a pair of strings representing the operation in a human readable way.

    :param edit_script: The edit script as a list of operations, where each operation is a dictionary.
    :param use_colors: Boolean indicating whether to use terminal color codes to color the output.
    :return: Tuple with text corresponding to the first pronunciation and the text of the second one.
    """
    colors = collections.defaultdict(str)

    if use_colors:
        colors['red'] = '\x1b[31m'
        colors['normal'] = '\x1b[m'
        colors['green'] = '\x1b[32m'
        colors['on_red'] = '\x1b[41m'

    src_txt = ''
    dst_txt = ''
    for op in edit_script:
        if op['op_code'] == 'match':
            width = max(len(op['from_symbol']), len(op['to_symbol']))
            if op['from_symbol'] == op['to_symbol']:
                src_txt += '{green}{from_symbol: ^{width}}{normal}'.format(**_combine_dicts(colors,
                                                                                            op,
                                                                                            {'width': width}))
                dst_txt += '{green}{to_symbol: ^{width}}{normal}'.format(**_combine_dicts(colors,
                                                                                          op,
                                                                                          {'width': width}))
            else:
                src_txt += '{red}{from_symbol: ^{width}}{normal}'.format(**_combine_dicts(colors,
                                                                                          op,
                                                                                          {'width': width}))
                dst_txt += '{red}{to_symbol: ^{width}}{normal}'.format(**_combine_dicts(colors,
                                                                                        op,
                                                                                        {'width': width}))

        elif op['op_code'] == 'insert':
            space = ' '*len(op['to_symbol'])
            src_txt += '{on_red}{space}{normal}'.format(space=space, **_combine_dicts(colors,  op))
            dst_txt += '{red}{to_symbol}{normal}'.format(**_combine_dicts(colors, op))

        elif op['op_code'] == 'delete':
            space = ' '*len(op['from_symbol'])
            src_txt += '{red}{from_symbol}{normal}'.format(**_combine_dicts(colors, op))
            dst_txt += '{on_red}{space}{normal}'.format(space=space, **_combine_dicts(colors, op))

        src_txt += ' '
        dst_txt += ' '

    return src_txt, dst_txt


def where_compact_locations(confusion_probability):
    """Compact the location representation from the phoneme+interphoneme representation to only the phoneme indices.

    :return: An array of contributions to each phoneme.
    """
    compact_confusion_probability = np.zeros((confusion_probability.size - 1) / 2)
    for index, prob in enumerate(confusion_probability):
        if index == 0:
            compact_confusion_probability[0] += (prob / 2.0)

        elif index == compact_confusion_probability.size * 2:
            compact_confusion_probability[-1] += (prob / 2.0)

        elif index % 2 == 0:
            phoneme_index = index / 2
            compact_confusion_probability[phoneme_index:phoneme_index+2] += (prob / 2.0)

        else:
            phoneme_index = index / 2
            compact_confusion_probability[phoneme_index] += prob

    return compact_confusion_probability


class WhatTrial(dict):
    def __getitem__(self, key):
        return self.setdefault(key, 0)

    def __hash__(self):
        return hash(frozenset(self.items()))

    def __eq__(self, other):
        return frozenset(self.items()) == frozenset(other.items())

    def __add__(self, other):
        if not isinstance(other, WhatTrial):
            raise TypeError('Can only add a WhatTrial to another WhatTrial')

        new = WhatTrial()
        new.update(self)
        for k, v in other.items():
            new[k] += v

        return new

    def __mul__(self, other):
        new = WhatTrial()
        new.update(self)
        for k in new:
            new[k] *= other

        return new

    def to_primitive(self):
        return dict(self)


class WhereTrial(np.ndarray):
    def __new__(cls, input_array):
        # Input array is an already formed ndarray instance
        # We first cast to be our class type
        obj = np.asarray(input_array).view(cls)
        obj.flags.writeable = False
        return obj

    def __hash__(self):
        return hash(self.data)

    def __eq__(self, other):
        return np.ndarray.__eq__(self, other).all()

    def __mul__(self, other):
        return WhereTrial(np.ndarray.__mul__(self, other))

    def __add__(self, other):
        return WhereTrial(np.ndarray.__add__(self, other))

    def to_primitive(self):
        return tuple(self)


def _dynamic_programming_response_trials(response_counts, response_trials_possible):
    """Check all the combinations of scripts (also called trials) per response for a set of responses
    (for a given index).

    This implementation uses dynamic programming to prune out already visited partial solutions.

    :param response_counts: a list of response counts
    :param response_trials_possible: a list of sets of acceptable what confusions per response
    :return: return a list of all possible accumulated trials for the set of responses
    (each accumulated trial is a dict from the reported phoneme to the number of times it was reported)
    """
    # Create an empty lattice
    lattice = [[set() for _ in response_trials]
               for response_trials in response_trials_possible]

    # Initialize the first state of the trellis
    for j, response_trial in enumerate(response_trials_possible[0]):
        lattice[0][j].add(response_trial * response_counts[0])

    # Forward pass
    # Because i=0 is for the response of index 1 (second response)
    for i, (response_count, response_trials) in enumerate(zip(response_counts[1:], response_trials_possible[1:])):
        for j, response_trial in enumerate(response_trials):
            new_trial = response_trial * int(response_count)

            # the previous trials are at i
            for previous_trials in lattice[i]:
                for previous_trial in previous_trials:
                    # and the current trial is at i+1
                    lattice[i + 1][j].add(previous_trial + new_trial)

    # Reduce the trials of the last response tested
    return [v.to_primitive() for v in reduce(lambda x, y: x | y, lattice[-1])]


def _unique_rows(x):
    """Return a new array with the duplicate rows of x removed.

    :param x:
    :return:
    """
    _, idx = np.unique(np.ascontiguousarray(x).view(np.dtype((np.void, x.dtype.itemsize * x.shape[1]))),
                       return_index=True)

    return x[idx]


def _script_to_indices(script):
    return list({op["index"] for op in script if not op["from_symbol"] == op["to_symbol"]})


def _script_to_phonemes_indices(script):
    new_script = collections.defaultdict(list)

    for op in script:
        new_script[str(op['index'])].append(op['to_symbol'] or '')

    return {k: ' '.join(v) if v is not None else None for k, v in new_script.items()}


def _what_naive_response_trials(response_counts, response_trials_possible):
    """Check all the combinations of scripts (also called trials) per response for a set of responses
    (for a given index).

    This implementation simply tests all combinations, therefore it may sometimes be computationally intractable.

    :param response_counts: a list of response counts
    :param response_trials_possible: a list of sets of acceptable what confusions per response
    :return: return a list of all possible accumulated trials for the set of responses
    (each accumulated trial is a dict from the reported phoneme to the number of times it was reported)
    """
    combination_count = np.prod([len(x) for x in response_trials_possible])
    total_trials_non_unique = [collections.defaultdict(lambda: 0) for _ in xrange(combination_count)]
    for i, response_trials in enumerate(itertools.product(*response_trials_possible)):
        for response_count, response_trial in zip(response_counts, response_trials):
            total_trials_non_unique[i][response_trial] += response_count

    # Make the what confusions hashable by converting them (currently a dict) to a frozenset
    # Make a set of the what confusions, to only output unique what confusions
    # Convert them back to a dict, which is more user-friendly
    return [dict(trials_hashable) for trials_hashable in
            {frozenset(trials.items()) for trials in total_trials_non_unique}]


def _where_naive_response_trials(response_counts, response_trials_possible):
    """Check all the combinations of scripts (also called trials) per response for a set of responses.

    This implementation simply tests all combinations, therefore it may sometimes be computationally intractable.

    :param response_counts: a list of response counts
    :param response_trials_possible: a list of sets of acceptable confusion trial per response
    :return: return a set of all possible accumulated trials for the set of responses
    """
    combination_count = np.prod([len(x) for x in response_trials_possible])
    total_trials_non_unique = np.zeros((combination_count, response_trials_possible[0][0].size))
    for i, response_trials in enumerate(itertools.product(*response_trials_possible)):
        total_trials_non_unique[i, :] += np.dot(response_counts, response_trials)

    return {tuple(x) for x in _unique_rows(np.array(total_trials_non_unique))}


def full_expected_predictions(token, lexicon):
    responses = token['responses']

    response_items = sorted(responses.items(), key=lambda x: -x[1])

    response_counts = []
    response_trials_possible = []
    for i, (response, response_count) in enumerate(response_items):
        unique_trials = set()

        # Collect scripts of all pronunciations of a response
        for pronunciation in lexicon[response.upper()]:
            trial = ' '.join(pronunciation)
            unique_trials.add(trial)

        response_trials_possible.append({WhatTrial({trial: 1}) for trial in unique_trials})
        response_counts.append(response_count)

    total_trials_possible = _dynamic_programming_response_trials(response_counts,
                                                                 response_trials_possible)

    total_count = sum(response_counts)

    return total_trials_possible, total_count


def what_expected_predictions(token, lexicon):
    speech = token['speech']
    responses = token['responses']

    # TODO: Make a better pronunciation choice for the produced speech (manual annotation?)
    speech_pron = lexicon[speech.upper()][0]
    index_count = len(speech_pron) * 2 + 1

    response_items = sorted(responses.items(), key=lambda x: -x[1])

    response_counts = []
    response_trials_possible = collections.defaultdict(list)
    for i, (response, response_count) in enumerate(response_items):
        all_scripts = []

        # Collect scripts of all pronunciations of a response
        for response_pron in lexicon[response.upper()]:
            scripts = get_edit_scripts(speech_pron, response_pron)
            all_scripts += scripts

        unique_per_index = collections.defaultdict(set)
        # Consider all possible scripts of all pronunciations
        for j, script in enumerate(all_scripts):
            index_to_symbol = _script_to_phonemes_indices(script)

            for index, symbol in index_to_symbol.items():
                unique_per_index[index].add(symbol)

            # Some indices may have not had appeared in the edit script
            for index in range(index_count):
                unique_per_index.setdefault(str(index), {''})

        for index in unique_per_index:
            response_trials_possible[index].append({WhatTrial({symbol: 1})
                                                    for symbol in unique_per_index[index]})

        response_counts.append(response_count)

    total_trials_possible = {}
    for index in response_trials_possible:
        total_trials_possible[index] = _dynamic_programming_response_trials(response_counts,
                                                                            response_trials_possible[index])

    total_count = sum(response_counts)

    return total_trials_possible, total_count


def where_expected_predictions(token, lexicon):
    speech = token['speech']
    responses = token['responses']

    # TODO: Make a better pronunciation choice for the produced speech (manual annotation?)
    speech_pron = lexicon[speech.upper()][0]
    index_count = len(speech_pron) * 2 + 1
    gt_confusion_probability = np.zeros(index_count)

    response_trials_possible = []
    response_items = sorted(responses.items(), key=lambda x: -x[1])

    response_counts = np.zeros(len(response_items))
    response_trials_counts = np.zeros(len(response_items), dtype=np.float32)

    # Collect all the possible edit scripts of each response
    for i, (response, response_count) in enumerate(response_items):
        all_scripts = []

        # Collect scripts of all pronunciations of a response
        for response_pron in lexicon[response.upper()]:
            scripts = get_edit_scripts(speech_pron, response_pron)
            all_scripts += scripts

        # Consider all possible scripts of all pronunciations
        non_unique = np.zeros((len(all_scripts), gt_confusion_probability.size), dtype=np.int)
        for j, script in enumerate(all_scripts):
            indices = _script_to_indices(script)

            non_unique[j, indices] = 1

        unique = _unique_rows(non_unique)
        unique_count = unique.shape[0]

        response_trials_possible.append({WhereTrial(trial) for trial in unique})

        response_counts[i] = int(response_count)
        response_trials_counts[i] = unique_count

    total_trials_possible = _dynamic_programming_response_trials(response_counts,
                                                                 response_trials_possible)
    total_count = np.sum(response_counts)

    return total_trials_possible, total_count


def where_evaluation(accepted_trials, trial_count, where_confusion_probability, epsilon=1e-2):
    """Compute the phone-level and token-level log-likelihood for the Where task.

    We assume a Binomial distribution for the event of observing a confusion at each position
    (phonemes and inter-phoneme indices $i \in N_i$).

    The result is the likelihood of the observations (set of listeners responses) given the probability
    of a confusion at each position supplied by the participant.

    :param accepted_trials: The accepted observations as (set of integer arrays). Arrays contain the
    amount of confusions at each position: $X[i] for i \in N_i$ where $N_i$ is the possible positions
    :param trial_count: The total amount of trials: $n$
    :param where_confusion_probability: The probability of observing a confusion at each position: $p[i] for i \in N_i$
    :param epsilon: Probabilities are clipped in the interval [epsilon, 1-epsilon]. This is useful to avoid potential
    -inf log-likelihoods for submissions of the Where task where probabilities have been set to 0 or to 1.
    :return: a dict with several information of the evaluation
    """
    import scipy.stats

    accepted_trials = np.array(list(accepted_trials))

    # Clip the probabilities between [epsilon 1-epsilon]
    # (otherwise it could lead to -inf log likelihoods)
    where_confusion_probability = np.array(where_confusion_probability).clip(epsilon, 1. - epsilon)

    accepted_phoneme_loglikelihoods = scipy.stats.binom.logpmf(accepted_trials,
                                                               trial_count,
                                                               where_confusion_probability)

    accepted_token_loglikelihoods = np.nanmean(accepted_phoneme_loglikelihoods, axis=1)
    best = np.argmax(accepted_token_loglikelihoods)

    evaluation = {'trials': accepted_trials[best],
                  'trial_count': trial_count,
                  'phoneme': {'loglikelihood': accepted_phoneme_loglikelihoods[best]},
                  'token': {'loglikelihood': accepted_token_loglikelihoods[best]}}

    return evaluation


def _multinomial_evaluation(trial,
                            probability,
                            min_unaccounted_probability=1e-2,
                            unaccounted_probability_optimistic=True):
    all_symbols = set(trial.keys()) | set(probability.keys())
    unaccounted_symbols = all_symbols - set(probability.keys())
    accounted_probability = sum(probability.values())
    unaccounted_probability = 1. - accounted_probability

    unaccounted_symbol_probability = 0.
    if len(unaccounted_symbols) > 0:
        unaccounted_symbol_probability = max(unaccounted_probability,
                                             min_unaccounted_probability) / len(unaccounted_symbols)

        if not unaccounted_probability_optimistic:
            unaccounted_symbol_probability = min(unaccounted_symbol_probability, min_unaccounted_probability)

    symbol_probabilities = [probability[symbol]
                            / min(unaccounted_probability / min_unaccounted_probability, 1.)
                            if symbol in probability
                            else unaccounted_symbol_probability
                            for symbol in all_symbols]

    symbol_trial_count = [trial.get(symbol, 0)
                          for symbol in all_symbols]

    multinom = Multinomial(symbol_probabilities + [1.0 - sum(symbol_probabilities)])

    return multinom.log_pmf(symbol_trial_count + [0])

def what_evaluation(per_index_accepted_trials, trial_count, per_index_phoneme_probability,
                    min_unaccounted_probability=1e-2, unaccounted_probability_optimistic=False):
    """Compute the phone-level and token-level log-likelihood for the What task.

    We assume a Multinomial distribution for the event of observing a given phoneme (or phoneme sequence in
    the case of an insertion) at certain position (phonemes and inter-phoneme indices $i \in N_i$).

    The result is the likelihood of the observations (set of listeners responses) given the probability
    of each possible phoneme (or phoneme sequence) at each position supplied by the participant.

    :param per_index_accepted_trials: The accepted observations as dictionary mapping from index (in string) to
    a dictionary (string -> float) of phonemes (or phoneme sequences) successes.
    amount of confusions at each position: $X[i, k] for i \in N_i, k \in N_k$ where $N_i$ is the possible positions
     and $N_k$ are the possible phonemes
    :param trial_count: The total amount of trials: $n$
    :param per_index_phoneme_probability: The probability of observing symbol k at at each position: $p[i, k] for
    i \in N_i, k \in N_k$.
    :param min_unaccounted_probability: Minimum probability assigned to unaccounted phonemes (or phoneme sequences)
    (phonemes that may have been reported by listeners but for which the participant has not submitted any probability).
    :param unaccounted_probability_optimistic: When set to True the probability of an unaccounted possibility
    is uniformly distributed among all unaccounted reported possibilities. If set to False then the probability
    of an unaccounted reported possibility is set to :param min_unaccounted_probability.
    :return: a dict with several information of the evaluation
    """
    per_index_phoneme_accepted_loglikelihoods = {}
    for index in per_index_phoneme_probability:
        accepted_trials = per_index_accepted_trials[index]
        phoneme_probability = per_index_phoneme_probability[index]

        per_index_phoneme_accepted_loglikelihoods[index] = np.full(len(accepted_trials), np.nan)
        for i, accepted_trial in enumerate(accepted_trials):
            result = _multinomial_evaluation(accepted_trial,
                                             phoneme_probability,
                                             min_unaccounted_probability=min_unaccounted_probability,
                                             unaccounted_probability_optimistic=unaccounted_probability_optimistic)

            per_index_phoneme_accepted_loglikelihoods[index][i] = result

    per_index_best_accepted = {index: np.argmax(accepted_loglikelihoods)
                               for index, accepted_loglikelihoods
                               in per_index_phoneme_accepted_loglikelihoods.items()}

    # Create a dense vector with loglikelihoods for the indices for which it has been computed
    # and NaN for those indices for which we do not have phoneme probabilities
    index_count = max([int(v) for v in per_index_phoneme_accepted_loglikelihoods]) + 1
    phoneme_loglikelihoods = np.full(index_count, np.nan)
    for index, best in per_index_best_accepted.items():
        phoneme_loglikelihoods[int(index)] = per_index_phoneme_accepted_loglikelihoods[index][best]

    token_loglikelihood = np.nanmean(phoneme_loglikelihoods)

    evaluation = {'trials': {index: per_index_accepted_trials[index][best]
                             for index, best
                             in per_index_best_accepted.items()},
                  'trial_count': trial_count,
                  'phoneme': {'loglikelihood': phoneme_loglikelihoods},
                  'token': {'loglikelihood': token_loglikelihood}}

    return evaluation


def full_evaluation(accepted_trials, trial_count, pronunciations_probability,
                    min_unaccounted_probability=1e-2, unaccounted_probability_optimistic=False):
    """Compute the token-level log-likelihood for the Full task.

    We assume a Multinomial distribution for the event of reporting a pronunciation.

    The result is the likelihood of the observations (set of listeners responses) given the probability
    of a given pronunciation being reported.

    :param accepted_trials: The accepted observations as (set of integer arrays). Arrays contain the
    amount of confusions at each position: $k[i] for i \in N_i$
    :param trial_count: The total amount of trials: $n$
    :param pronunciations_probability: The probability of observing symbol k at at each position: $p[k, i] for
    i \in N_i, k \in N_k$ where $N_i$ is the possible positions and $N_k$ are the possible phonemes.
    :param min_unaccounted_probability: Minimum probability assigned to unaccounted symbols (symbols that may have
    happened but for which the participant has not submitted any probability).
    :param unaccounted_probability_optimistic: When set to True the probability of an unaccounted possibility
    is uniformly distributed among all unaccounted reported possibilities. If set to False then the probability
    of an unaccounted reported possibility is set to :param min_unaccounted_probability.
    :return: a dict with several information of the evaluation
    """
    pronunciation_accepted_loglikelihoods = np.full(len(accepted_trials), np.nan)
    for i, accepted_trial in enumerate(accepted_trials):
        result = _multinomial_evaluation(accepted_trial,
                                         pronunciations_probability,
                                         min_unaccounted_probability=min_unaccounted_probability,
                                         unaccounted_probability_optimistic=unaccounted_probability_optimistic)

        pronunciation_accepted_loglikelihoods[i] = result

    best_accepted = np.argmax(pronunciation_accepted_loglikelihoods)

    evaluation = {'trials': accepted_trials[best_accepted],
                  'trial_count': trial_count,
                  'token': {'loglikelihood': pronunciation_accepted_loglikelihoods[best_accepted]}}

    return evaluation


def evaluate_token(prediction, token, task, lexicon):
    if task == 'where':
        accepted_trials, trial_count = where_expected_predictions(token, lexicon)
        confusion_probability = prediction[task]['confusion_probability']

        return where_evaluation(accepted_trials, trial_count, confusion_probability)

    elif task == 'what':
        per_index_accepted_trials, trial_count = what_expected_predictions(token, lexicon)
        per_index_phoneme_probability = prediction[task]['per_index_phonemes_probability']

        return what_evaluation(per_index_accepted_trials, trial_count, per_index_phoneme_probability)

    elif task == 'full':
        accepted_trials, trial_count = full_expected_predictions(token, lexicon)
        pronunciations_probability = prediction[task]['pronunciations_probability']

        return full_evaluation(accepted_trials, trial_count, pronunciations_probability)

    else:
        raise NotImplementedError('The evaluation for task {} has not been implement.'.format(task))

    return None


def evaluate_submission(submission, dataset, lexicon):
    evaluations = {}

    logs = []
    warnings = []

    ds_tokens = set(dataset['tokens'].keys())

    tasks = ['where', 'what', 'full']

    for task in tasks:
        su_tokens = {key for key in submission['tokens'].keys() if task in submission['tokens'][key]}
        if not su_tokens:
            logs.append('No predictions found for the {} task'.format(task))

        predicted_tokens = ds_tokens & su_tokens
        unpredicted_tokens = ds_tokens - su_tokens
        nonexistant_tokens = su_tokens - ds_tokens

        if nonexistant_tokens:
            warnings.append('The following tokens '
                            'could not be found in '
                            'the dataset: {}'.format(' '.join(nonexistant_tokens)))

        if unpredicted_tokens:
            warnings.append('The following tokens '
                            'could not be found in '
                            'the submission: {}'.format(' '.join(unpredicted_tokens)))

        evaluations[task] = {'tokens': {}}
        for token_id in sorted(list(predicted_tokens)):
            prediction = submission['tokens'][token_id]
            token = dataset['tokens'][token_id]

            evaluation = evaluate_token(prediction, token, task, lexicon)
            evaluations[task]['tokens'][token_id] = evaluation

    return evaluations


def score_evaluation(evaluations):
    scores = {}

    tasks = ['where', 'what', 'full']

    for task in tasks:
        scores[task] = {}
        scores[task]['tokens'] = {}

        for token_id, evaluation in evaluations[task]['tokens'].items():
            scores[task]['tokens'][token_id] = {}

            if 'phoneme' in evaluation:
                scores[task]['tokens'][token_id]['phoneme'] = evaluation['phoneme']

            scores[task]['tokens'][token_id]['token'] = evaluation['token']

        token_averaged_loglikelihood = np.nanmean([v['token']['loglikelihood']
                                                   for v in scores[task]['tokens'].values()])

        scores[task]['token_averaged'] = {'loglikelihood': token_averaged_loglikelihood}

    return scores