#!/usr/bin/env python

# Script for runing the recogniser of an ASR model on a set of audios

import argparse
import random
import logging
import os
import glob
import shutil

import config
import common


def recognise_models(args):
    output_directory = config.path(args.results_directory, create=True)
    temp_directory = config.project_path('tmp', create=True)

    # Gather information about the models to be tested
    model_directories = [args.model_directory]
    models = {}

    for model_directory in model_directories:
        model_name = model_directory
        models[model_name] = {}
        models[model_name]['model_directory'] = model_directory
        models[model_name]['output_directory'] = config.path(output_directory, create=True)
        models[model_name]['cmllr_directory'] = None

        if args.use_cmllradapted_models:
            cmllr_directories = glob.glob(os.path.join(model_directory, 'cmllr', '*'))
            for cmllr_directory in cmllr_directories:
                model_name = cmllr_directory
                models[model_name] = {}
                models[model_name]['model_directory'] = model_directory
                models[model_name]['output_directory'] = config.path(output_directory,
                                                                     os.path.relpath(cmllr_directory, model_directory),
                                                                     create=True)
                models[model_name]['cmllr_directory'] = cmllr_directory

    # Try to load the default parameters
    default_arguments_filename = config.path(model_directory, '..',
                                             'train_arguments.json') if args.default_arguments_filename is None else ''
    if os.path.isfile(default_arguments_filename):
        args = common.load_train_arguments(default_arguments_filename, args)

    # Select recordings
    list_recordings = common.glob_recordings(args.audio_dir, args.audio_pattern)

    # Prepare dictionary
    dictionary_filename = config.path(output_directory, 'dictionary.txt')
    logging.debug('Create full dictionary (adding SENT-START, SENT-END and silence)')
    common.merge_dictionaries(dictionary_filename,
                              [args.pronunciation_dictionary_filename],
                              args)

    # Extract the features using the external (and replaceable) extractor
    feature_directory = config.path(temp_directory, 'test_features', create=True)
    inputoutput_feature_filename = config.path(temp_directory, 'test_io_feature_file.scp')
    output_feature_filename = config.path(temp_directory, 'test_feature_file.scp')
    feature_extractor = args.feature_extractor_command
    feature_extractor_parameters = args.feature_extractor_parameters
    common.create_feature_files(list_recordings,
                                feature_directory,
                                inputoutput_feature_filename,
                                output_feature_filename,
                                feature_extractor,
                                feature_extractor_parameters)

    for model_name, model in models.items():
        logging.info('Recognising with model {}'.format(model_name))
        results_directory = config.path(model['output_directory'], create=True)
        recognition_filename = config.path(results_directory, 'recognition.mlf')

        # Copy recognition required files to the results directory
        symbollist_filename = config.path(model_directory, 'symbollist')
        output_symbollist_filename = config.path(results_directory, 'symbollist')
        shutil.copyfile(symbollist_filename, output_symbollist_filename)

        common.recognise_model(output_feature_filename,
                               output_symbollist_filename,
                               model['model_directory'],
                               recognition_filename,
                               dictionary_filename,
                               list_words_filename=args.list_words_filename,
                               cmllr_directory=model['cmllr_directory'],
                               hypotheses_count=args.hypotheses_count,
                               tokens_count=args.tokens_count,
                               htk_trace=args.htk_trace)

    return


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description="Test the recognition rate of a model")

    parser.add_argument("-r", "--random-seed", dest="random_seed", type=int,
                        help="Seed for the random sampling of training instances.")

    parser.add_argument("-e", "--default-arguments-filename", type=str, default=None,
                        help="FILE in JSON format containing the default parameters. "
                             "If left unset it will try to use MODEL_DIRECTORY/../train_arguments.json", metavar="FILE")

    parser.add_argument("-i", "--model-directory", dest="model_directory", type=str, required=True,
                        help="MODEL_DIRECTORY of the model to be tested. "
                             "Should contain hmmdefs, macros and symbollist files.",
                        metavar="MODEL_DIRECTORY")

    parser.add_argument("-o", "--results-directory", dest="results_directory", type=str, default='results',
                        help="DIRECTORY where to write the test results", metavar="DIRECTORY")

    parser.add_argument("-w", "--list-words-filename", dest="list_words_filename", type=str, default='',
                        help="""FILE containing list of all possible language words.
                        If empty then all the words of the dictionary are used as language words.""", metavar="FILE")

    parser.add_argument("-a", "--audio-dir", dest="audio_dir", nargs='+', required=True,
                        help="""DIRECTORY containing the recordings used for testing.
                        Wav filenames will be selected matching the pattern given using the -p argument.""",
                        metavar="DIRECTORY")

    parser.add_argument("-p", "--audio-pattern", dest="audio_pattern", nargs='+', default=['*.wav'],
                        help="""PATTERN the filenames must match to be selected for recognition.""",
                        metavar="PATTERN")

    parser.add_argument("-nh", "--hypotheses-count", dest="hypotheses_count", default=1, type=int,
                        help="""Number of output hypotheses to generate.
                        Often referred to N-Best recognition when higher than 1.""")

    parser.add_argument("-nt", "--tokens-count", dest="tokens_count", default=None, type=int,
                        help="""Number of tokens to maintain in each lattice node.
                        If not provided it is set to tokens_count = ceil(hypotheses_count/5).""")

    parser.add_argument("-d", "--pronunciation-dictionary-filename", dest="pronunciation_dictionary_filename", type=str,
                        help="FILE containing the pronunciation of all the words in the language."
                             "If not provided the training dictionary is used.", metavar="FILE")

    parser.add_argument("-x", "--feature-extractor-command", dest="feature_extractor_command", default=None,
                        help="""CMD command to be called with a single space separated inputOutputFile argument in order
                        to extract features from input wav files into output feature files.""", metavar="CMD")

    parser.add_argument("-xp", "--feature-extractor-parameters", dest="feature_extractor_parameters", default=None,
                        help="""STR commandline argument string to be passed to the feature extractor command.""",
                        metavar="STR")

    parser.add_argument("-T", "--htk-trace", action='store', type=int, default=0,
                        help="""Verbose output and trace from HTK commands.""")

    parser.add_argument("-c", "--use-cmllradapted-models", action='store_true',
                        help="""Set this flag to also test the adapted models.""")

    parser.add_argument("-l", "--log", dest="log_level", default="info",
                        help="""Log level of the python scripts.""")

    args = parser.parse_args()

    numeric_level = getattr(logging, args.log_level.upper(), None)
    if not isinstance(numeric_level, int):
        raise ValueError('Invalid log level: %s' % args.log_level)

    logging.basicConfig(level=numeric_level)

    if args.random_seed is not None:
        random.seed(args.random_seed)

    recognise_models(args)