#!/usr/bin/env python

# Script for creating an ASR model


import argparse
import ast
import random
import logging
import os
import glob
import shutil
import codecs
import json

import config
import common


def test_models(input_directory,
                output_directory,
                speaker_test_instance_count,
                list_words_filename='',
                filter_training_recordings=True,
                filter_adaptation_recordings=True,
                use_cmllradapted_models=True,
                audio_directories=[],
                htk_trace=0):
    output_directory = config.path(output_directory, create=True)
    temp_directory = config.project_path('tmp', create=True)

    parameters_filename = os.path.join(input_directory, 'parameters.json')
    with open(parameters_filename, 'r') as parameters_file:
        parameters = json.load(parameters_file)

    # Gather information about the models to be tested
    model_directories = [input_directory]
    models = {}
    list_unaccepted = []

    for model_directory in model_directories:
        model_name = model_directory
        models[model_name] = {}
        models[model_name]['model_directory'] = model_directory
        models[model_name]['output_directory'] = config.path(output_directory, create=True)
        models[model_name]['cmllr_directory'] = None

        if filter_training_recordings:
            # Select unaccepted recordings from training
            trainset_filename = parameters['list_train_recordings_filename']
            with codecs.open(trainset_filename, 'r', 'utf-8') as f:
                list_unaccepted += [l.strip() for l in f.readlines()]

        if use_cmllradapted_models:
            cmllr_directories = glob.glob(os.path.join(model_directory, 'cmllr', '*'))
            for cmllr_directory in cmllr_directories:
                model_name = cmllr_directory
                models[model_name] = {}
                models[model_name]['model_directory'] = model_directory
                models[model_name]['output_directory'] = config.path(output_directory,
                                                                     os.path.relpath(cmllr_directory, model_directory),
                                                                     create=True)
                models[model_name]['cmllr_directory'] = cmllr_directory

                if filter_adaptation_recordings:
                    # Select unaccepted recordings from adaptation
                    adaptset_filename = config.path(cmllr_directory, 'adaptset.txt')
                    with open(adaptset_filename) as f:
                        list_unaccepted += [l.strip() for l in f.readlines()]

    # Create language word list
    if list_words_filename:
        list_words = common.parse_wordlist(list_words_filename)
    else:
        list_words = sorted(common.parse_dictionary(parameters['dictionary_filename']).keys())


    # Select recordings for testing
    testset_filename = config.path(output_directory, 'testset.txt')
    list_recordings = common.select_recording_set(parameters['audio_directories'] + audio_directories,
                                                  speaker_test_instance_count,
                                                  testset_filename,
                                                  possible_words=list_words,
                                                  unaccepted_recordings=list_unaccepted)

    # Prepare dictionary
    dictionary_filename = config.path(output_directory, 'dictionary.txt')
    logging.debug('Create full dictionary (adding SENT-START, SENT-END and silence)')
    common.merge_dictionaries(dictionary_filename,
                              [parameters['dictionary_filename']],
                              htk_trace=htk_trace)

    # Extract the features using the external (and replaceable) extractor
    feature_directory = config.path(temp_directory, 'test_features', create=True)
    inputoutput_feature_filename = config.path(temp_directory, 'test_io_feature_file.scp')
    output_feature_filename = config.path(temp_directory, 'test_feature_file.scp')
    feature_extractor = parameters['feature_extractor_command']
    feature_extractor_parameters = parameters['feature_extractor_parameters']
    common.create_feature_files(list_recordings,
                                feature_directory,
                                inputoutput_feature_filename,
                                output_feature_filename,
                                feature_extractor,
                                feature_extractor_parameters)

    reference_filename = config.path(temp_directory, 'test_word.mlf')
    testphones_filename = config.path(temp_directory, 'test_phones')
    common.create_recordings_mlf(list_recordings,
                                 pronunciation_dictionary_filename=dictionary_filename,
                                 phones_filename=testphones_filename,
                                 mlf_word_file=reference_filename,
                                 htk_trace=htk_trace)

    for model_name, model in models.items():
        logging.info('Evaluating model {}'.format(model_name))
        output_directory = config.path(model['output_directory'], create=True)
        output_recognition_filename = config.path(output_directory, 'recognition.mlf')

        # Copy recognition required files to the results directory
        symbollist_filename = config.path(model_directory, 'symbollist')
        output_symbollist_filename = config.path(output_directory, 'symbollist')
        shutil.copyfile(symbollist_filename, output_symbollist_filename)

        common.recognise_model(output_feature_filename,
                               output_symbollist_filename,
                               model['model_directory'],
                               output_recognition_filename,
                               dictionary_filename,
                               list_words_filename=list_words_filename,
                               cmllr_directory=model['cmllr_directory'],
                               htk_trace=htk_trace)

        # Copy evaluation required files to the results directory
        output_reference_filename = config.path(output_directory, os.path.basename(reference_filename))
        shutil.copyfile(reference_filename, output_reference_filename)

        common.evaluate_model(output_reference_filename,
                              output_symbollist_filename,
                              output_recognition_filename,
                              dictionary_filename=dictionary_filename,
                              htk_trace=htk_trace)

    # Save the arguments
    logging.debug('Save parameters')
    parameters_output_filename = config.path(output_directory, 'parameters.json')

    with codecs.open(parameters_output_filename, 'w', 'utf-8') as parameters_file:
        json.dump(parameters, parameters_file, indent=2)

    return


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description="Test the recognition rate of a model")

    parser.add_argument("-r", "--random-seed", dest="random_seed", type=int,
                        help="Seed for the random sampling of training instances.")

    parser.add_argument("-e", "--default-arguments-filename", type=str, default=None,
                        help="FILE in JSON format containing the default parameters. "
                             "If left unset it will try to use MODEL_DIRECTORY/../train_arguments.json",
                        metavar="FILE")

    parser.add_argument("-i", "--model-directory", dest="model_directory", type=str, required=True,
                        help="MODEL_DIRECTORY of the model to be tested. "
                             "Should contain hmmdefs, macros and symbollist files.",
                        metavar="MODEL_DIRECTORY")

    parser.add_argument("-o", "--results-directory", dest="results_directory", type=str, default='results',
                        help="DIRECTORY where to write the test results", metavar="DIRECTORY")

    parser.add_argument("-w", "--list-words-filename", dest="list_words_filename", type=str,
                        help="FILE containing list of all possible language words."
                             "If not supplied then the same word list as for training is used.", metavar="FILE")

    parser.add_argument("-a", "--audio-dir", dest="audio_dir", nargs='+', default=[],
                        help="""DIRECTORY containing the recordings used for testing.
                        Wav filenames must end in X_sY.wav where X is the word
                        and Y is the numeric ID of the spearker.""",
                        metavar="DIRECTORY")

    parser.add_argument("-d", "--pronunciation-dictionary-filename", dest="pronunciation_dictionary_filename", type=str,
                        help="FILE containing the pronunciation of all the words in the language.", metavar="FILE")

    parser.add_argument("--no-filter-training-recordings", dest="filter_training_recordings", default=True,
                        action='store_false',
                        help="Set this flag to allow the reuse training recordings for testing.")

    parser.add_argument("--no-filter-adaptation-recordings", dest="filter_adaptation_recordings", default=True,
                        action='store_false',
                        help="Set this flag to allow the reuse adaptation recordings for testing.")

    parser.add_argument("-n", "--speaker-test-instance-count", dest="speaker_test_instance_count",
                        type=ast.literal_eval, default=None,
                        help="""PYTHON_LITERAL is a dictionary in Python literal format
                        specifying the amount of testing instances for each speaker.
                        Speakers not appearing in the dictionary will not be used for testing.
                        e.g. "{1: 1000, 2: 500}" will use 1000 instances of speaker s1, 500 of speaker s2
                         and none of other speakers.
                        If the argument is not set then all the instances of all speakers are used for testing
                        (see argument --no-filter-training-recordings
                        to control the exclusion of recordings used for training).""",
                        metavar="PYTHON_LITERAL")

    parser.add_argument("-T", "--htk-trace", action='store', type=int, default=0,
                        help="""Verbose output and trace from HTK commands.""")

    parser.add_argument("-c", "--use-cmllradapted-models", action='store_true',
                        help="""Set this flag to also test the adapted models.""")

    parser.add_argument("-l", "--log", dest="log_level", default="info",
                        help="""Log level of the python scripts.""")

    args = parser.parse_args()

    numeric_level = getattr(logging, args.log_level.upper(), None)
    if not isinstance(numeric_level, int):
        raise ValueError('Invalid log level: {}'.format(args.log_level))

    logging.basicConfig(level=numeric_level)

    if args.random_seed is not None:
        random.seed(args.random_seed)

    test_models(args.model_directory,
                args.results_directory,
                args.speaker_test_instance_count,
                list_words_filename=args.list_words_filename,
                filter_training_recordings=args.filter_training_recordings,
                filter_adaptation_recordings=args.filter_adaptation_recordings,
                use_cmllradapted_models=args.use_cmllradapted_models,
                audio_directories=args.audio_dir,
                htk_trace=args.htk_trace)