#!/usr/bin/env python

# Script for creating an ASR model


import os
import argparse
import itertools
import json
import shutil
import logging
import codecs

import config
import common


def create_hmm_models(output_feature_filename,
                      pronunciation_dictionary_filename,
                      prototype_filename,
                      phones_filename,
                      phonessp_filename,
                      triphones_filename,
                      triphonessp_filename,
                      train_phones_filename,
                      train_phonessp_filename,
                      train_triphones_filename,
                      train_triphonessp_filename,
                      output_directory,
                      mlf_word_file,
                      mlf_phone_file,
                      mlf_phonesp_file,
                      mlf_tri_file,
                      mlf_trisp_file,
                      mktri_script_filename,
                      phoneme_class_filename,
                      htk_trace):
    models_directory = config.path('tmp/models', create=True)

    # Train an initial set of monophone models using a flat start
    model_directory = config.path(models_directory, 'mono.0', create=True)
    config.htk_command("HCompV -A -D -T {} -f 0.01 "
                       "-S {} -m -o m3states -M {} {}".format(htk_trace,
                                                              output_feature_filename,
                                                              model_directory,
                                                              prototype_filename))

    # Parse the statistics of the speech features
    vfloors_filename = os.path.join(model_directory, 'vFloors')
    m3states_filename = os.path.join(model_directory, 'm3states')

    with codecs.open(m3states_filename, 'r', 'utf-8') as m3states_file:
        m3states_lines = m3states_file.readlines()
        m3states_first = m3states_lines[:3]
        m3states_second = m3states_lines[3:]

    # Synthesize the monophones initial model
    with codecs.open(os.path.join(model_directory, 'macros'), 'w', 'utf-8') as macros_file:
        macros_file.writelines(m3states_first)

        with codecs.open(vfloors_filename, 'r', 'utf-8') as vFloors_file:
            macros_file.writelines(vFloors_file.readlines())

    os.remove(vfloors_filename)

    with codecs.open(os.path.join(model_directory, 'hmmdefs'), 'w', 'utf-8') as hmmdefs_file:
        with codecs.open(phonessp_filename, 'r', 'utf-8') as phones_file:
            phones = [phone.strip() for phone in phones_file.readlines()]

            for phone in phones:
                hmmdefs_file.write(''.join(m3states_second).replace('m3states', phone))

    # Reestimate parameters for monophone models (5 iterations)
    iterations = [1, 2, 3, 4, 5]
    for iteration in iterations:
        previous_model_directory = model_directory
        model_directory = config.path(models_directory, 'mono.{}'.format(iteration), create=True)

        config.htk_command("HERest -A -D -T {} -S {} -d {} "
                           "-H {} -H {} -I {} -M {} {}".format(htk_trace,
                                                               output_feature_filename,
                                                               previous_model_directory,
                                                               os.path.join(
                                                                   previous_model_directory,
                                                                   'macros'),
                                                               os.path.join(
                                                                   previous_model_directory,
                                                                   'hmmdefs'),
                                                               mlf_phone_file,
                                                               model_directory,
                                                               phones_filename))

        # Copy the necessary to the model directory
        shutil.copy(phones_filename, os.path.join(model_directory, 'symbollist'))

    # Fix the silence model
    previous_model_directory = model_directory
    model_directory = config.path(models_directory, 'mono.6', create=True)

    with codecs.open(os.path.join(model_directory, 'hmmdefs'), 'w', 'utf-8') as hmmdefs_file:
        with codecs.open(os.path.join(previous_model_directory, 'hmmdefs'), 'r', 'utf-8') as previous_hmmdefs_file:
            previous_hmmdefs_lines = previous_hmmdefs_file.readlines()
            previous_hmmdefs_header = previous_hmmdefs_lines[-18:][:5]

            hmmdefs_file.writelines(previous_hmmdefs_lines)
            hmmdefs_file.write("""~h "sp"
<BEGINHMM> 
<NUMSTATES> 3
<STATE> 2
""")
            hmmdefs_file.writelines(previous_hmmdefs_header)
            hmmdefs_file.write("""<TRANSP> 3
 0.0 1.0 0.0
 0.0 0.9 0.1
 0.0 0.0 0.0
<ENDHMM>
""")

    shutil.copyfile(os.path.join(previous_model_directory, 'macros'), os.path.join(model_directory, 'macros'))

    # Copy the necessary to the model directory
    shutil.copy(phones_filename, os.path.join(model_directory, 'symbollist'))

    previous_model_directory = model_directory
    model_directory = config.path(models_directory, 'mono.7', create=True)
    config.htk_command("HHEd -A -H {} -H {} -M {} {} {}".format(os.path.join(previous_model_directory, 'macros'),
                                                                os.path.join(previous_model_directory, 'hmmdefs'),
                                                                model_directory,
                                                                config.project_path('etc/sil.hed'),
                                                                phonessp_filename))

    # Copy the necessary to the model directory
    shutil.copy(phonessp_filename, os.path.join(model_directory, 'symbollist'))

    # TODO: we should realign to cope with multiple pronunciations of words
    # http://www.ee.columbia.edu/ln/LabROSA/doc/HTKBook21/node30.html

    # Reestimate parameters for monophone models with fixed silence (2 iterations)
    iterations = [8, 9]
    for iteration in iterations:
        previous_model_directory = model_directory
        model_directory = config.path(models_directory, 'mono.{}'.format(iteration), create=True)

        config.htk_command("HERest -A -D -T {} -S {} "
                           "-H {} -H {} -I {} -M {} {}".format(htk_trace,
                                                               output_feature_filename,
                                                               os.path.join(
                                                                   previous_model_directory,
                                                                   'macros'),
                                                               os.path.join(
                                                                   previous_model_directory,
                                                                   'hmmdefs'),
                                                               mlf_phonesp_file,
                                                               model_directory,
                                                               phonessp_filename))

        # Copy the necessary to the model directory
        shutil.copy(phonessp_filename, os.path.join(model_directory, 'symbollist'))

    # Upgrade monophone models to triphone models
    previous_model_directory = model_directory
    model_directory = config.path(models_directory, 'tri.0', create=True)

    config.htk_command("HHEd -A -D -T {} "
                       "-H {} -H {} -M {} {} {}".format(htk_trace,
                                                        os.path.join(previous_model_directory,
                                                                     'macros'),
                                                        os.path.join(previous_model_directory,
                                                                     'hmmdefs'),
                                                        model_directory,
                                                        mktri_script_filename,
                                                        phonessp_filename))

    # Copy the necessary to the model directory
    shutil.copy(phonessp_filename, os.path.join(model_directory, 'symbollist'))

    # Reestimating parameters for triphone models
    tristats_filename = config.path(models_directory, 'tri.stats')

    iterations = [1, 2, 3]
    for iteration in iterations:
        previous_model_directory = model_directory
        model_directory = config.path(models_directory, 'tri.{}'.format(iteration), create=True)

        config.htk_command("HERest -A -D -T {} -S {} "
                           "-H {} -H {} -I {} -M {} -s {} {}".format(htk_trace,
                                                                     output_feature_filename,
                                                                     os.path.join(
                                                                         previous_model_directory,
                                                                         'macros'),
                                                                     os.path.join(
                                                                         previous_model_directory,
                                                                         'hmmdefs'),
                                                                     mlf_trisp_file,
                                                                     model_directory,
                                                                     tristats_filename,
                                                                     train_triphonessp_filename))

        # Copy the necessary to the model directory
        shutil.copy(train_triphonessp_filename, os.path.join(model_directory, 'symbollist'))

    # Upgrade triphone models to tied-state triphones
    previous_model_directory = model_directory
    model_directory = config.path(models_directory, 'tstri.1mix.0', create=True)

    log_filename = config.project_path('tmp', 'log')
    triphonehtk_filename = config.project_path('tmp', 'triphones_htk')
    pronunciation_triphone_dictionary_filename = config.project_path('tmp', 'full_dict.tri.txt')

    # this step should go to the create dictionaries function
    config.htk_command("HDMan -A -D -T {} -b sp "
                       "-n {} -g {} -l {} {} {}".format(htk_trace,
                                                        triphonehtk_filename,
                                                        config.project_path('etc',
                                                                            'global.tri.ded'),
                                                        log_filename,
                                                        pronunciation_triphone_dictionary_filename,
                                                        pronunciation_dictionary_filename))

    # triphonessp_filename = triphonehtk_filename

    tiedlist_filename = config.path(models_directory, 'tiedlist')
    tree_filename = config.path(models_directory, 'tree')
    script_filename = config.project_path('tmp', 'tree.hed')
    create_triphone_tied_script(phonessp_filename,
                                phoneme_class_filename,
                                tristats_filename,
                                tiedlist_filename,
                                tree_filename,
                                script_filename,
                                language_triphones_filename=triphonessp_filename)

    config.htk_command("HHEd -A -D -T {} -H {} -H {} -M {} {} {}".format(htk_trace,
                                                                         os.path.join(previous_model_directory,
                                                                                      'macros'),
                                                                         os.path.join(previous_model_directory,
                                                                                      'hmmdefs'),
                                                                         model_directory,
                                                                         script_filename,
                                                                         train_triphonessp_filename))

    # Copy the necessary to the model directory
    shutil.copy(triphonessp_filename, os.path.join(model_directory, 'symbollist'))

    # Upgrade triphone models to tied-state triphones
    iterations = [1, 2, 3]
    for iteration in iterations:
        previous_model_directory = model_directory
        model_directory = config.path(models_directory, 'tstri.1mix.{}'.format(iteration), create=True)

        config.htk_command("HERest -A -D -T {} -S {} "
                           "-H {} -H {} -I {} -M {} -s {} {}".format(htk_trace,
                                                                     output_feature_filename,
                                                                     os.path.join(
                                                                         previous_model_directory,
                                                                         'macros'),
                                                                     os.path.join(
                                                                         previous_model_directory,
                                                                         'hmmdefs'),
                                                                     mlf_trisp_file,
                                                                     model_directory,
                                                                     tristats_filename,
                                                                     tiedlist_filename))

        # Copy the necessary to the model directory
        shutil.copy(tiedlist_filename, os.path.join(model_directory, 'symbollist'))
        shutil.copy(tristats_filename, os.path.join(model_directory, 'tri.stats'))

    # Increase the number of state mixtures
    mixtures = [2, 3, 4, 5]
    for mixture in mixtures:
        previous_model_directory = model_directory
        model_directory = config.path(models_directory, 'tstri.{}mix.0'.format(mixture), create=True)

        script_filename = config.project_path('tmp/mix{}.hed'.format(mixture))
        with codecs.open(script_filename, 'w', 'utf-8') as f:
            f.write('MU {} {{*.state[2-20].mix}}\n'.format(mixture))

        config.htk_command('HHEd -A -D -T {} -d {} '
                           '-H {} -H {} -M {} {} {}'.format(htk_trace,
                                                            previous_model_directory,
                                                            os.path.join(
                                                                previous_model_directory,
                                                                'macros'),
                                                            os.path.join(
                                                                previous_model_directory,
                                                                'hmmdefs'),
                                                            model_directory,
                                                            script_filename,
                                                            tiedlist_filename))

        # Copy the necessary to the model directory
        shutil.copy(tiedlist_filename, os.path.join(model_directory, 'symbollist'))
        shutil.copy(tristats_filename, os.path.join(model_directory, 'tri.stats'))

        iterations = [1, 2, 3, 4]
        for iteration in iterations:
            previous_model_directory = model_directory
            model_directory = config.path(models_directory, 'tstri.{}mix.{}'.format(mixture, iteration), create=True)

            config.htk_command("HERest -A -D -T {} -S {} -d {} "
                               "-H {} -H {} -I {} -M {} -s {} {}".format(htk_trace,
                                                                         output_feature_filename,
                                                                         previous_model_directory,
                                                                         os.path.join(
                                                                             previous_model_directory,
                                                                             'macros'),
                                                                         os.path.join(
                                                                             previous_model_directory,
                                                                             'hmmdefs'),
                                                                         mlf_trisp_file,
                                                                         model_directory,
                                                                         tristats_filename,
                                                                         tiedlist_filename))

            # Copy the necessary to the model directory
            shutil.copy(tiedlist_filename, os.path.join(model_directory, 'symbollist'))
            shutil.copy(tristats_filename, os.path.join(model_directory, 'tri.stats'))

    shutil.copy(os.path.join(model_directory, 'macros'), os.path.join(output_directory, 'macros'))
    shutil.copy(os.path.join(model_directory, 'hmmdefs'), os.path.join(output_directory, 'hmmdefs'))
    shutil.copy(os.path.join(model_directory, 'symbollist'), os.path.join(output_directory, 'symbollist'))
    shutil.copy(os.path.join(model_directory, 'tri.stats'), os.path.join(output_directory, 'tri.stats'))

    return output_directory


def generate_stateclustering_triphones_script(command,
                                              threshold,
                                              phones_filename,
                                              emitting_state_count=3):
    script = []

    phones = []
    with codecs.open(phones_filename, 'r', 'utf-8') as f:
        phones += [l.strip() for l in f.readlines() if l.strip() != '']

    for state in range(2, 2 + emitting_state_count):
        for phone in phones:
            line = '{command} {threshold} "ST_{phone}_{state}_" ' \
                   '{{("{phone}","*-{phone}+*",' \
                   '"{phone}+*",' \
                   '"*-{phone}").state[{state}]}}\n'.format(command=command,
                                                            threshold=threshold,
                                                            state=state,
                                                            phone=phone)
            script.append(line)

    return script


def create_triphone_tied_script(phones_filename,
                                phoneme_class_filename,
                                stats_filename,
                                tiedlist_filename,
                                tree_filename,
                                script_filename,
                                language_triphones_filename=None):
    script_template_filename = phoneme_class_filename

    if language_triphones_filename is None:
        triphones = []
        with open(phones_filename) as phones_file:
            phones = [l.strip() for l in phones_file.readlines()]
            phones = filter(lambda x: x not in ['sp', 'sil'], phones)

            # Generate all possible triphones given the monophone list
            triphone_tuples = sorted(list(itertools.product(phones, repeat=3)), key=lambda x: x[1])
            triphones += ['{}-{}+{}'.format(*triphone_tuple) for triphone_tuple in triphone_tuples]

            # Generate all possible biphones (left and right boundary triphones) given the monophone list
            triphone_bound_tuples = list(itertools.product(phones, repeat=2))
            triphones += ['{}+{}'.format(*triphone_tuple) for triphone_tuple in
                          sorted(triphone_bound_tuples, key=lambda x: x[0])]
            triphones += ['{}-{}'.format(*triphone_tuple) for triphone_tuple in
                          sorted(triphone_bound_tuples, key=lambda x: x[1])]

        triphones += ['sil', 'sp']

        triphones_filename = config.project_path('tmp/triphones_all')

        with codecs.open(triphones_filename, 'w', 'utf-8') as f:
            f.writelines(['{}\n'.format(triphone) for triphone in triphones])

    with codecs.open(script_filename, 'w', 'utf-8') as script:
        script.write('RO 100 "{}"\n'.format(stats_filename))

        script.write('\n')

        with codecs.open(script_template_filename, 'r', 'utf-8') as script_template:
            script.writelines(script_template.readlines())

        lines = generate_stateclustering_triphones_script('TB',
                                                          350,
                                                          phones_filename)
        script.writelines(lines)

        script.write('\n')

        text = '''TR 1

AU "{}"
CO "{}"

ST "{}"
       '''.format(language_triphones_filename,
                  tiedlist_filename,
                  tree_filename)

        script.write(text)


def create_triphones_from_pronunciation(pronunciation):
    triphones = []

    if len(pronunciation) > 0:
        offsets = range(-1, len(pronunciation) - 1)

        for k, i in enumerate(offsets):
            context = pronunciation[max(i, 0): i + 3]

            # monophone word
            if k == 0 and k == len(offsets) - 1:
                triphones.append('{}'.format(*context))

            # left boundary
            elif k == 0:
                triphones.append('{}+{}'.format(*context))

            # right boundary
            elif k == len(offsets) - 1:
                triphones.append('{}-{}'.format(*context))

            else:
                triphones.append('{}-{}+{}'.format(*context))

    return set(triphones)


def create_phones_lists(pronunciation_dictionary_filename,
                        phones_filename,
                        phonessp_filename,
                        triphones_filename,
                        triphonessp_filename,
                        boundary_symbols={'sp', 'sil'}):
    logging.debug("Creating symbols (phones/triphones) lists")
    # TODO: Decide whether to use HDMan to do this as explained here:
    # http://www.ee.columbia.edu/ln/LabROSA/doc/HTKBook21/node33.html
    phones = set()
    with codecs.open(pronunciation_dictionary_filename, 'r', 'utf-8') as f:
        for line in f:
            v = line.strip().split(None, 1)[1]
            phones |= set(v.strip().split())

    if '[]' in phones:
        phones.remove('[]')

    phones = list(phones)

    with codecs.open(phones_filename, 'w', 'utf-8') as f:
        f.writelines('{}\n'.format(phone) for phone in phones)

    with codecs.open(phonessp_filename, 'w', 'utf-8') as f:
        f.writelines('{}\n'.format(phone) for phone in (phones + ['sp']))

    triphones = set()
    with codecs.open(pronunciation_dictionary_filename, 'r', 'utf-8') as f:
        for line in f:
            v = line.strip().split(None, 1)[1]
            pronunciation = v.strip().split()
            pronunciation = filter(lambda x: x not in boundary_symbols, pronunciation)
            if '[]' not in pronunciation:
                triphones |= create_triphones_from_pronunciation(pronunciation)

    # TODO: We should also add to the triphone set the set of possible monophones
    # so that during training the prototypes are available.
    # This would fix cases in which very little training data is available
    with codecs.open(triphones_filename, 'w', 'utf-8') as f:
        f.writelines('{}\n'.format(phone) for phone in triphones)

    with codecs.open(triphonessp_filename, 'w', 'utf-8') as f:
        f.writelines('{}\n'.format(phone) for phone in (triphones | {'sil', 'sp'}))


def create_triphone_script(monophone_filename,
                           triphone_filename,
                           script_filename):
    with codecs.open(script_filename, 'w', 'utf-8') as script_file:
        script_file.write('CL {}\n'.format(triphone_filename))

        with codecs.open(monophone_filename, 'r', 'utf-8') as monophone_file:
            for line in monophone_file:
                phone = line.strip()

                if phone != '':
                    script_file.write(
                        'TI T_{phone} {{("{phone}",'
                        '"*-{phone}+*",'
                        '"{phone}+*",'
                        '"*-{phone}").transP}}\n'.format(phone=phone))


def train_models(input_directory,
                 models_directory,
                 prior_component_count,
                 phoneme_class_filename,
                 htk_trace):
    output_directory = config.path(models_directory, create=True)
    temp_directory = config.project_path('tmp', create=True)

    parameters_filename = os.path.join(input_directory, 'parameters.json')
    with open(parameters_filename, 'r') as parameters_file:
        parameters = json.load(parameters_file)

    # Create the make triphone script
    mktri_script_filename = config.path(temp_directory, 'mktri.hed')
    logging.debug('Create triphone script')
    create_triphone_script(parameters['list_phones_shortpause_filename'],
                           parameters['list_train_triphones_shortpause_filename'],
                           mktri_script_filename)

    # Create prototype model based on the first features (e.g proto.3states.mfc)
    prototype_filename = config.path(temp_directory, 'proto.3states.mfc')
    logging.debug('Create prototype model')
    common.create_prototype_model(parameters['list_train_feature_files_filename'],
                                  prototype_filename,
                                  state_stay_probabilities=[0.6, 0.6, 0.7])

    # Create the HMM models by estimating the parameters given the recording data
    logging.debug('Create HMM models')
    model_directory = create_hmm_models(parameters['list_train_feature_files_filename'],
                                        parameters['dictionary_filename'],
                                        prototype_filename,
                                        parameters['list_phones_filename'],
                                        parameters['list_phones_shortpause_filename'],
                                        parameters['list_triphones_filename'],
                                        parameters['list_triphones_shortpause_filename'],
                                        parameters['list_train_phones_filename'],
                                        parameters['list_train_phones_shortpause_filename'],
                                        parameters['list_train_triphones_filename'],
                                        parameters['list_train_triphones_shortpause_filename'],
                                        output_directory,
                                        parameters['label_word_filename'],
                                        parameters['label_phone_filename'],
                                        parameters['label_phone_shortpause_filename'],
                                        parameters['label_triphone_filename'],
                                        parameters['label_triphone_shortpause_filename'],
                                        mktri_script_filename,
                                        phoneme_class_filename,
                                        htk_trace)

    with codecs.open(parameters['list_train_recordings_filename'], 'r', 'utf-8') as f:
        list_training = [l.strip() for l in f.readlines()]

    # Create the prior model if requested
    if prior_component_count > 0:
        output_prior_directory = config.path(output_directory, 'speech_prior', create=True)
        speech_prior_mlf_filename = config.project_path(output_directory, 'speech_prior.mlf')
        common.train_speech_prior(model_directory,
                                  prior_component_count,
                                  output_prior_directory,
                                  parameters['dictionary_filename'],
                                  list_training,
                                  parameters['list_train_feature_files_filename'],
                                  speech_prior_mlf_filename,
                                  htk_trace)

        parameters['label_speech_prior_filename'] = speech_prior_mlf_filename

    # Save the arguments
    logging.debug('Save parameters')
    parameters_output_filename = config.path(output_directory, 'parameters.json')

    with codecs.open(parameters_output_filename, 'w', 'utf-8') as parameters_file:
        json.dump(parameters, parameters_file, indent=2)

    return


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description="Create a baseline intelligibility model for the INSPIRE Challenge")

    parser.add_argument("-i", "--input-directory", dest="input_directory", type=str,
                        help="DIRECTORY where to find the prepared training data "
                             "(parameters, feature, label and symbol files)",
                        metavar="DIRECTORY")

    parser.add_argument("-o", "--models-directory", dest="models_directory", type=str, default='models',
                        help="DIRECTORY where to write the training results (parameters, models)",
                        metavar="DIRECTORY")

    parser.add_argument("-p", "--phoneme-class-file", dest="phoneme_class_filename", type=str, required=True,
                        help="FILE containing the HTK questions used in the triphone decision tree clustering. "
                             "These are language dependent.",
                        metavar="FILE")

    parser.add_argument("--prior-component-count", type=int, default=0,
                        help="Train the speech prior with a GMM of N components. "
                             "If 0 the prior is not trained.",
                        metavar="PRIOR_COMPONENTS")

    parser.add_argument("-T", "--htk-trace", action='store', type=int, default=0,
                        help="""Verbose output and trace from HTK commands.""")

    parser.add_argument("-l", "--log", dest="log_level", default="info",
                        help="""Log level of the python scripts.""")

    args = parser.parse_args()

    # Set the log level
    numeric_level = getattr(logging, args.log_level.upper(), None)
    if not isinstance(numeric_level, int):
        raise ValueError('Invalid log level: %s' % args.log_level)

    logging.basicConfig(level=numeric_level)

    train_models(args.input_directory,
                 args.models_directory,
                 args.prior_component_count,
                 args.phoneme_class_filename,
                 args.htk_trace)

