# Copyright (c) 2012 Leif Johnson <leif@leifjohnson.net>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

import itertools
import logging
import sqlite3
import time

ITERATION = '__ITERATION__'


class Weights(object):
    '''Weights is an interface for storing and retrieving feature weights.'''

    def __iter__(self):
        '''Iterate over feature, weight pairs in this object.'''
        raise NotImplementedError

    def sum(self, features):
        '''Sum the weights for the given features.'''
        raise NotImplementedError

    def update(self, deltas, repeat=-1):
        '''Increment feature weights according to the given delta map.'''
        raise NotImplementedError

    def average(self, edge):
        '''Combine the given edge weights into our weight averages.'''
        raise NotImplementedError


class Sqlite(Weights):
    '''Store feature weights in a sqlite3 database.'''

    def __init__(self, filename, table='weights', encoding='UTF-8', wal=True, page_size=65536):
        self.filename = filename
        self.table = table

        self._db = None

        self.db.execute('PRAGMA encoding = "%s"' % encoding)
        self.db.execute('PRAGMA page_size = %d' % page_size)
        if wal:
            self.db.execute('PRAGMA journal = WAL')

        # try to create our table for storing weights.
        try:
            self.db.execute('CREATE TABLE %s ('
                            'feature VARCHAR PRIMARY KEY, '
                            'weight REAL)' % self.table)
        except:
            pass

    @property
    def db(self):
        if self._db is None:
            self._db = sqlite3.connect(self.filename)
        return self._db

    def __iter__(self):
        sql = 'SELECT feature, weight FROM %s' % self.table
        return iter(self.db.execute(sql))

    def sum(self, features, repeat=-1):
        args = tuple(features)
        if not args:
            return 0.
        sql = 'SELECT SUM(weight) FROM %s WHERE feature IN (%s)' % (
            self.table, ','.join('?' * len(args)))
        while repeat != 0:
            try:
                return self.db.execute(sql, args).fetchone()[0]
            except sqlite3.OperationalError:
                logging.debug('sum operation failed, retrying ...')
                repeat -= 1

    def update(self, deltas, repeat=-1):
        insert = 'INSERT OR IGNORE INTO %s (weight, feature) VALUES (?, ?)' % self.table
        update = 'UPDATE %s SET weight = weight + ? WHERE feature = ?' % self.table
        while repeat != 0:
            try:
                with self.db as db:
                    for feature, delta in deltas:
                        db.execute(insert, (delta, feature))
                        db.execute(update, (delta, feature))
                break
            except sqlite3.OperationalError:
                logging.debug('update operation failed, retrying ...')
                repeat -= 1

    def average(self, edge, repeat=-1):
        start = time.time()

        select = 'SELECT weight FROM %s WHERE feature = ?' % self.table
        i, = self.db.execute(select, (ITERATION, )).fetchone()
        self.update([(ITERATION, 1)])

        current = dict(self)

        count = 0
        targets = []
        for count, (feature, delta) in enumerate(edge):
            weight = current.get(feature)
            if weight is not None:
                delta += i * weight
            targets.append((feature, delta / (i + 1.)))

        update = 'REPLACE INTO %s SELECT ? AS weight, ? AS feature' % self.table
        union = ' UNION SELECT ?, ?'
        updates = []
        while targets:
            active, targets = targets[:490], targets[490:]
            updates.append((update + union * (len(active) - 1),
                            itertools.chain.from_iterable(active)))

        while repeat != 0:
            try:
                with self.db as db:
                    for sql, active in updates:
                        db.execute(sql, tuple(active))
                break
            except sqlite3.OperationalError:
                logging.exception('average operation failed, retrying ...')
                repeat -= 1

        logging.debug('averaged %d weights in %dms',
                      count, 1000 * (time.time() - start))
