#! /usr/bin/env python
# -*- encoding: utf-8 -*-
# vim:fenc=utf-8:

import hashlib

from mole.helper import AttrDict
from mole.helper import timeformat

class EventError(Exception):
    """Generic error related with events."""

class EventTimeError(EventError):
    """A problem happened when try to convert a time field into
    a understable time format."""


class Event(object):
    """Events are logic lines generated by plotter and parsing by parser.

    :param `ev_dict`: a initial dictionary to build the event, if
        missing empty dict is used.
    """

    def __init__(self, ev_dict={}):
        self.items = AttrDict(ev_dict)
        self._time = None
        self.hash = hashlib.md5(str(ev_dict)).hexdigest()[0:5]

    @property
    def time(self):
        """Get the time object related with event time. This property uses
        internal _time field for the event and convert it to an understable
        time object.
        """
        if self._time:
            return self._time
        try:
            if isinstance(self.items["_time"], set):
                self._time = timeformat.TimeFormat(map(lambda x:x, self.items["_time"])[0])
            else:
                self._time = timeformat.TimeFormat(self.items["_time"])
            return self._time
        except KeyError:
            raise EventTimeError(repr(self))
            #raise EventTimeError("Unable to convert time for event %s." % self.hash)

    def fields(self):
        """Return the keyname of all fields present in the event."""
        return self.items.keys()

    def has_any(self, items):
        """Return True iff any of the items passed as argument is present in
        the event.

        :param `items`: an iterable object which contains the field names to
            search for. The fields name must be strings (or unicode).
        """
        for x in items:
            if x in self.items:
                return True
        return False

    def has_all(self, items):
        """Return True iff all items passed as argument are present in the
        event.

        :param `items`: an iterable object which contains the field names to
            search for. The fields name must be strings (or unicode).
        """
        for x in items:
            if x not in self.items:
                return False
        return True

    def __iter__(self):
        return self.items.iteritems()

    def __getitem__(self, item, default=None):
        return self.items.get(item, default)

    def __setitem__(self, item, value):
        self.items[item] = value

    def __contains__(self, item):
        return item in self.items

    def __delitem__(self, item):
        del self.items[item]

    def __repr__(self):
        return repr(self.items)

    def __dict__(self):
        return self.items

    @classmethod
    def merge(cls, events, timeline=0):
        """Create a new :class:`Event` which is the aggregation of all
        events passed as argument.

        :param `events`: an iterable which contains a sequence of
            :class:`Event`.
        :param `timeline`: the order key to identify an event in events list
            passed as argumetn from where the time used in new event will be
            taken of.
        """
        d = {}
        for ev in events:
            for (k,v) in ev.items.iteritems():
                if k != "_time":
                    if d.get(k,None):
                        d[k].add(v)
                    else:
                        d[k] = set([v])
        d["_time"] = events[timeline]["_time"]
        return Event(d)


