'''
------------------------------------------
common.parsing_routines.generic_classes.py
------------------------------------------

This module contains generic classes used in the other parsing routines. 

.. moduleauthor:: Nick Schurch <nschurch@dundee.ac.uk>

:module_version: 1.0
:created_on: 2013-11-26
'''
import re, os, gzip, numpy
from Bio.Seq import Seq
from Bio.Alphabet import generic_dna, generic_rna, generic_protein

def openfile(fname):
    
    """ open and read a gzipped or uncompressed file """
    
    # try and cast string to a string if it isn't already!
    if type(fname) is not str:
        raise TypeError("The filename passed to openfile is not a " \
                        "string. Value: %s, type: %s" \
                        "" % (str(fname), type(fname)))
    
    # resolve relative paths to a full path
    full_path = os.path.abspath(fname)
    
    # check that the file exists
    if not os.path.isfile(full_path):
        msg = "The file specified (%s) does not exist.\nPlease specify a " \
              "valid file." % full_path
        raise IOError(msg)
    
    # read the file
    filedata=None
    try:
        filehandle = gzip.open(full_path, "r")
        filedata = filehandle.readlines()
        filehandle.close()
    except IOError:
        filehandle = open(full_path, "r")
        filedata = filehandle.readlines()
        filehandle.close()
    
    return(filedata)


class region:
    
    """A simple class for representing a a chromosome region"""
    
    def __init__(self, region_name, chrid, start, stop,
                 strand=None, seq=None, seqtype=None, desc=None):

        """ class contrustor: this requires name and chromosome id as strings, 
        start and stop and ints, and optionally strand, sequence and 
        description as strings"""
                
        self.name=None
        self.chrid=None
        self.start=None
        self.stop=None
        self.strand=None
        self.sequence=None
        self.desc=None

        if type(region_name) is str and region_name!="":
            self.name=region_name
        else:
            raise TypeError("Region name must be a non-empty string.")
        
        if type(chrid) is str and chrid!="":
            self.chrid=chrid
        else:
            raise TypeError("Chromosome ID must be a non-empty string.")
        
        if type(start) is int or type(start) is numpy.int64 or start is None:
            self.start=start
        else:
            raise TypeError("Region start must be an integer (or None).")
        
        if type(stop) is int or type(stop) is numpy.int64 or stop is None:
            self.stop=stop
        else:
            raise TypeError("Region stop must be an integer (or None).")
        
        if strand is not None and type(strand) is str:
            if strand=="+" or strand=="-":
                self.strand=strand
            elif strand=="fwd" or strand=="forward":
                self.strand="+"
            elif strand=="rev" or strand=="reverse":
                self.strand="-"
            else:
                raise ValueError("Strand specification should be one of " \
                                 "+/-/fwd/rev/forward/reverse.")
        elif strand is not None and type(strand) is not str:
            raise TypeError("Region stand must be a string.")

        if seq is not None:
            if type(seq) is Seq:
                self.sequence=seq
            elif type(seq) is str and seqtype is None:
                self.sequence=Seq(seq)
            elif type(seq) is str and type(seqtype) is str:
                if seqtype=="dna" or seqtype=="DNA":
                    self.sequence=Seq(seq, generic_dna)
                elif seqtype=="rna" or seqtype=="RNA":
                    self.sequence=Seq(seq, generic_rna)
                elif seqtype=="prot" or seqtype=="protein":
                    self.sequence=Seq(seq, generic_protein)
                else:
                    raise ValueError("type of sequence must be one of " \
                                     "dna/DNA/rna/RNA/prot/protein.") 
            elif type(seq) is str and type(seqtype) is not str and seqtype is not None:
                raise TypeError("Region sequence 'type' must be a string or " \
                                "None.")            
            elif type(seq) is not str and type(seq) is not Seq:
                raise TypeError("Region sequence must be a string or a Seq " \
                                "record from Biopython or None.")
        
        if desc is not None and (type(desc) is str or type(desc) is dict):
            self.desc=desc
        elif desc is not None:
            raise TypeError("Region desc must be a string or a dictionary.")

def generic_set_region(region_rep, start=None, stop=None, strand=None):
    
    ''' Returns the input region, or builds one from the input details
    
    This is used in several of the parsers including the anotation, and wigData
    classes
    '''

    general_msg = "Invalid region specification. Regions should be either a " \
                  "valid 'region' instance, or be specified as a string of " \
                  "format 'chrid' (which results in selecting the entire " \
                  "chromosome) or 'chrid:start-stop' or with three arguments " \
                  "as set_region('chrid', start_int, stop_int)."
        
    if region_rep.__class__.__name__ is 'region':
        return(region_rep)
    elif type(region_rep) is str and region_rep!="":
        if start is not None and stop is None:
            raise ValueError("Please specify a stop position")
        if stop is not None and start is None:
            raise ValueError("Please specify a start position")
        if (start is not None and type(start) is not int and 
            type(start) is not numpy.int64):
            raise TypeError("start position should be an int")
        if (stop is not None and type(stop) is not int and 
            type(stop) is not numpy.int64):
            raise TypeError("stop position should be an int")
        
        # parse string for region details if necessary
        if start is None and stop is None:
            try:
                split_1 = re.split(":",region_rep.strip())
                chrid = split_1[0]
            except IndexError:                
                raise ValueError(general_msg)
            
            if len(split_1)>1:
                try:
                    split_2 = re.split("-",split_1[1])
                    start = int(split_2[0])
                    stop = int(split_2[1])
                except IndexError:
                    raise ValueError(general_msg)            
        else:
            if type(start) is not int or type(stop) is not int:
                raise ValueError(general_msg)
            chrid = region_rep
        
        if start is not None and stop is not None:
            # sanitize stupid region start/stop values
            if start<1:
                msg="Attempt to set start position less-than 1."
                raise ValueError(msg)
            if stop<1:
                msg="Attempt to set stop position less-than 1."
                raise ValueError(msg)
            if start==stop:
                msg="Attempt to set zero-length region."
                raise ValueError(msg)
            if start>stop:
                msg="Attempt to set stop position < start position."
                raise ValueError(msg)
        
        return(region("set_region", chrid, start, stop, strand))
    elif region_rep=="":
        raise ValueError(general_msg)
    else:
        raise TypeError(general_msg)

