#!/usr/bin/python
'''
-------------------
tests.general_classes.py
-------------------

A module containing tests for the general_classes module I'm 
using in :ref:`general_classes.py <general_classes_autodocs>`. These tests use 
the standard `unittest <http://docs.python.org/2/library/unittest.html>`_ 
package and extend the :py:class:`unittest.TestCase` class.

.. moduleauthor:: Nick Schurch <nschurch@dundee.ac.uk>

:module_version: 1.0
:created_on: 2013-11-26

'''

__version__ = "1.0"

import unittest, os, tempfile, shutil, argparse, warnings, numpy, gzip
import parsing_routines.general_classes_and_functions as gc
from Bio.Seq import Seq
from Bio.Alphabet import generic_dna

class TestGeneralClasses(unittest.TestCase):

    ''' Test the classes defined in :ref:`general_classes.py 
    <general_classes_autodocs>`.
    
    Each function is tested with existing and non-existent instances of the 
    variable type that it is designed for. We also test that providing non-
    string basic python objects produces string representations and doesn't 
    case a failure.
    
    '''
    
    def setUp(self):
        
        ''' Initialize the framework for testing. 
        
        Define and create a new system temporary directory with the
        `tempfile <http://docs.python.org/2/library/tempfile.html>`_ package
        for use as an existing directory. A range of wig files with different
        parameters are created in this directory to test the routines with. 
        '''
        
        # setup a new temp directory
        self.existing_path = tempfile.mkdtemp()
        
        file_content = "some stuff\nhere\nto\nread\n"
        
        # a simple text file to read
        self.simple = "%s/textfile.txt" % self.existing_path
        temp_file = open(self.simple, "w")
        temp_file.write(file_content)
        temp_file.close()
        
        # a compressed text file to read
        self.gzipped = "%s/textfile.txt.gz" % self.existing_path
        temp_file = gzip.open(self.gzipped, "wb")
        temp_file.write(file_content)
        temp_file.close()        
        
        # make up a mythical path and file based on the existing dir
        self.nonexisting_file = self.existing_path+"/thisdoesntexist.txt"
    
    def test_region(self):
        
        """ test the region class contructor raises sensible errors
        
        ================================    ===================================
        Test object                         Expectation
        ================================    ===================================
        first and second args not string    raise TypeError
        third and fouth args not int        raise TypeError
        strand and desc not string          raise TypeError
        seq not string or Seq               raise TypeError
        seqType not string                  raise TypeError
        
        """
        
        # test types of args
        self.assertRaises(TypeError, gc.region,"","","","")
        self.assertRaises(TypeError, gc.region,"test","","","")
        self.assertRaises(TypeError, gc.region,"test","chr1","","")
        self.assertRaises(TypeError, gc.region,"test","chr1", 1,"")
        self.assertRaises(TypeError, gc.region,"test","chr1", 1,10000,0,0,0,0)
        self.assertRaises(TypeError, gc.region,"test","chr1", 1,10000,"+",0,
                                                0,0)
        self.assertRaises(TypeError, gc.region,"test","chr1", 1,10000,"+",
                                                "AGC",0,0)
        self.assertRaises(TypeError, gc.region,"test","chr1", 1,10000,"+",
                                                "AGC","dna",0)
        
        # test values of args
        self.assertRaises(ValueError, gc.region, "test","chr1", 1,10000,"bob",
                                                 "AGC","dna","testdesc")        
        self.assertRaises(ValueError, gc.region, "test","chr1", 1,10000,"+",
                                                 "AGC","bob","testdesc")
        
        try:
            x=gc.region("test","chr1", 1, 10000)
            x=gc.region("test","chr1", 1, 10000, "+", "AGC","dna","testdesc")
            
            x=gc.region("test","chr1", 1, 10000,
                        strand="+", seq="AGC", seqtype="dna", desc="testdesc")
            x=gc.region("test","chr1", 1, 10000,
                        strand="+", seq="AGC", desc="testdesc")
            test_seq = Seq("AGNNNGTAGT", generic_dna)
            x=gc.region("test","chr1", 1, 10000,
                        strand="+", seq=test_seq, desc="testdesc")
            x=gc.region("test","chr1", 1, 10000,
                        strand="+", seq=test_seq, desc={"some":"other data"})            
        except:
            self.fail("region failing even with correct arguments")
        
    def test_openfile(self):
        
        """ tests the openfile function raises sensible errors/data
        
        ===================================   =================================
        Test object                           Expectation
        ===================================   =================================
        filename is not a string              TypeError
        file doesn't exist                    IOError
        correctly reads test files            Correct file-data
        """
        
        # test types of args
        self.assertRaises(TypeError, gc.openfile, 1)
        self.assertRaises(IOError, gc.openfile, self.nonexisting_file)
        
        try:
            fd = gc.openfile(self.simple)
            fd = gc.openfile(self.gzipped)
        except:
            self.fail("Function failing with correct arguments")

    def test_generic_set_region(self):
        
        ''' test region setter generic_set_region
        
        ============================   =========================================
        Test object                    Expectation
        ============================   =========================================
        no args/first arg not string   raise TypeError
        poorly formatted string        raise ValueError
        second & third args not int    raise TypeError
        just chr id                    sets selector key but not start/stop
                                       i.e., select whole chromosome
        valid string representation    correct selector setting
        valid triple arg               correct selector setting
        '''
        
        self.assertRaises(TypeError, gc.generic_set_region)
        self.assertRaises(TypeError, gc.generic_set_region, 1, 50)
        self.assertRaises(ValueError, gc.generic_set_region, "", "1", "50")
        self.assertRaises(TypeError, gc.generic_set_region, "chr1", "1", 50)
        self.assertRaises(TypeError, gc.generic_set_region, "chr1", 1, "50")
        self.assertRaises(ValueError, gc.generic_set_region, "chr1", 1)
        self.assertRaises(ValueError, gc.generic_set_region, "", 1, 50)
        self.assertRaises(ValueError, gc.generic_set_region, "chr1", -1, 1)
        self.assertRaises(ValueError, gc.generic_set_region, "chr1", 1, -1)
        self.assertRaises(ValueError, gc.generic_set_region, "chr1", 1, 1)
        self.assertRaises(ValueError, gc.generic_set_region, "chr1", 10, 5)

        test_region = gc.region("set_region", "chr1", 1, 50)
        x = gc.generic_set_region(test_region)
        self.assertEqual(test_region, x)
        
        x = gc.generic_set_region("chr1")
        self.assertEqual("chr1", x.chrid)
        x = gc.generic_set_region("chr1", 1, 50)
        self.assertEqual("chr1", x.chrid)
        self.assertEqual(1, x.start)
        self.assertEqual(50, x.stop)
        x = gc.generic_set_region("chr1:1-50")
        self.assertEqual("chr1", x.chrid)
        self.assertEqual(1, x.start)
        self.assertEqual(50, x.stop)

if __name__ == '__main__':
    unittest.main(verbosity=2)