'''
----------------------------------------
common.parsing_routines.gff_gtf_tools.py
----------------------------------------

This module contains routines for reading, parsing and summarizing data in 
gft and gff files. There are at least three versions of the gff format (so this
will be fun!) and then gtf (v2.2). This is a good page for information about the
versions: http://www.broadinstitute.org/igv/GFF

.. moduleauthor:: Nick Schurch <nschurch@dundee.ac.uk>

:module_version: 1.0
:created_on: 2013-11-27
'''

import numpy, tempfile, time
from parsing_routines.general_classes_and_functions import openfile, region, generic_set_region
from Bio import SeqIO

def parse_gff_v3(fname, skip_fasta=True, force_type=None, 
                 progress_report=False):
    
    """ read a gff_v3 file and convert it into a set of 'regions' 
    
    skip_fasta controls what to do with fasta sequences storred at the end of 
    gff files, force_type controls whether to autodetect the type from the gff
    file or whether to assume a known filetype.
    
    """
        
    filedata = openfile(fname)
    
    current_filetype=None
    fasta_temp=None
    fasta_isOpen=False
    headerlines=[]
    type_region_dic={}
    type_region_key_dic={}
    
    if force_type is not None:
        current_filetype=force_type
    
    progress_counter=0
    progress_mult=10000
    progress_done=0
    progress_time=time.time()
    progress_inner_time=time.time()
    
    line_count=0
    for line in filedata:
        line = line.strip()
        if line!="":
            
            if line.startswith("##gff-version 3"):
                current_filetype="gff3"
            
            if line.startswith("##FASTA"):
                current_filetype="fasta"
            
            if line.startswith("#"):
                headerlines.append(line)
            elif current_filetype=="gff3":
                linedata = line.split("\t")
                if len(linedata)!=9:
                    msg = "Line %i does not contain 9 fields as per gff v3 " \
                          "spec." % line_count 
                    raise IndexError(msg)
                else:
                    region_name = str(line_count)
                                        
                    # munge attributes
                    attrib_dic={}
                    if linedata[8]!=".":
                        other_attributes = linedata[8].split(";")
                        for entry in other_attributes:
                            if entry!="":
                                attrib_data = entry.strip().split("=")
                                if len(attrib_data)!=2:
                                    msg="Error at line %i in the file: " \
                                        "Attribute %s does not have the " \
                                        "correct format: xx=xx,yy" \
                                        "" % (line_count, entry)
                                    raise IndexError(msg)
                                else:
                                    attrib_vals = attrib_data[1].split(",")
                                    if len(attrib_vals)==1:
                                        attrib_vals = attrib_vals[0]
                                attrib_dic[attrib_data[0].lower()]=attrib_vals
                    
                    # add some columns as new attributes
                    if linedata[1]!=".":
                        attrib_dic["source"]=linedata[1]
                    if linedata[2]!=".":
                        attrib_dic["type"]=linedata[2]
                    if linedata[5]!=".":
                        attrib_dic["score"]=float(linedata[5])
                    else:
                        attrib_dic["score"]=None
                    if linedata[7]!=".":
                        attrib_dic["phase"]=int(linedata[7])
                    else:
                        attrib_dic["phase"]=None
                    
                    # make region objects
                    if "name" in attrib_dic:
                        region_name = attrib_dic["name"]
                    
                    chrid=None
                    if linedata[0]!=".":
                        chrid=linedata[0]
                    
                    start=None
                    if linedata[3]!=".":
                        start=int(linedata[3])

                    stop=None
                    if linedata[4]!=".":
                        stop=int(linedata[4])

                    strand=None
                    if linedata[6]!=".":
                        strand=linedata[6]

                    this_region = region(region_name, chrid, start, stop, 
                                         strand=strand, desc=attrib_dic)
                    
                    # add to data dictionaries
                    type_region_key = line_count
                    if linedata[2]==".":
                        type_region_dic[type_region_key] = this_region
                    else:
                        type_region_key = "%ss" % linedata[2]
                        if type_region_key not in type_region_dic.keys():
                            type_region_dic[type_region_key] = {}
                            type_region_key_dic[type_region_key] = 0
                        else:
                            type_region_key_dic[type_region_key]+=1
                                                    
                        type_region_dic[type_region_key][type_region_key_dic[type_region_key]]=this_region                    
                    
            elif current_filetype=="fasta":
                if not skip_fasta:
                    if not fasta_isOpen:
                        tmpdir = tempfile.mkdtemp()
                        fasta_temp = open("%s/thistempfile.fasta" % tmpdir,"w")
                        fasta_isOpen=True
                    fasta_temp.write("%s\n" % line)
                
        line_count+=1
        progress_counter+=1
        if progress_report and progress_counter==progress_mult:
            progress_done+=1
            print "processed %i new lines in %.4fs, (%i total lines in %.2fs)" \
                  "" % (progress_mult, time.time()-progress_inner_time,
                        progress_mult*progress_done, time.time()-progress_time)
            progress_inner_time=time.time()
            progress_counter=0
        
    if fasta_isOpen:
        fasta_temp.close()
        fasta_temp = open("%s/thistempfile.fasta" % tmpdir, "rU")
        for record in SeqIO.parse(fasta_temp, "fasta") :
            for entry in type_region_dic["chromosomes"].keys():
                if type_region_dic["chromosomes"][entry].name==record.id:
                    type_region_dic["chromosomes"][entry].sequence = record
                    break
    
    return(type_region_dic, headerlines)

class annotation:
    
    """A simple class for representing annotations from a gff or gtf file"""
    
    __sort_order=["chr","start","stop","strand"]
    __ordering_array_dtype=[("type","|S255"), ("id","int"), ("chr","|S255"),
                            ("start","int"),("stop","int"),("strand","|S10"),
                            ("name","|S255")]
    
    __known_types=["gff3"]
    
    def __init__(self, fname, skip_fasta=True, filetype=None, verbose=False):
        
        """ class constructor: this takes the filename of the gff or gtf file
        and reads the data from it into the class attributes. Filename should
        be a string and point to a file that exists. skip_fasta is passed to 
        the parsing options where appropriate and filetype controls wether to
        assume a particular filetype or to try and autodetect it from the file.
        """
        
        self._full_annotation=None
        self.file_header = None
        self.current_chr=None
        self._featurelist=[]
        self._chrid_dic={}
        self._ordering_array=None
        self.current_feature=None
        self._feature_index=None
        self.current_region=None
        self._region_index=None
        self.current_name=None
        self._name_index=None
        self.current_attributes=None
        
        # parse the annotation data
        if filetype in self.__known_types:
            if filetype=="gff3":
                annotation_data, headerlines = parse_gff_v3(fname, 
                                                            skip_fasta=skip_fasta,
                                                            force_type=filetype,
                                                            progress_report=verbose)

            if len(annotation_data)==0:
                raise ValueError("Something went wrong with parsing the file " \
                                 "using filetype %s. Try specifying a " \
                                 "different one of the following with the " \
                                 "argument 'filetype='xxx': %s" \
                                 "" % (filetype, ", ".join(self.__known_types)))
                
        else:
            annotation_data, headerlines = parse_gff_v3(fname, 
                                                        skip_fasta=skip_fasta,
                                                        progress_report=verbose)
            
            if len(annotation_data)==0:
                raise ValueError("Something went wrong with parsing the file " \
                                 "using autodetection for the filetype. Try " \
                                 "specifying one of the following with the " \
                                 "argument 'filetype='xxx': %s" \
                                 "" % ", ".join(self.__known_types))
        
        self._full_annotation=annotation_data
        self.file_header = headerlines
        tuplelist=[]
        for region_type in annotation_data:
            self._featurelist.append(region_type)
            setattr(self, str(region_type), annotation_data[region_type])
            
            try:            
                for each_region in annotation_data[region_type]:
                    this_region = annotation_data[region_type][each_region]                    
                    tuplelist.append(self.__region_to_tuple(this_region,
                                                            region_type,
                                                            each_region))
            except TypeError:
                    this_region = annotation_data[region_type]                    
                    tuplelist.append(self.__region_to_tuple(this_region,
                                                            region_type,
                                                            region_type))
        
        iarray = numpy.zeros(len(tuplelist),
                             dtype=self.__ordering_array_dtype)
        i=0
        while i<len(tuplelist):
            iarray[i]=tuplelist[i]
            i+=1
        
        self._ordering_array = numpy.sort(iarray, order=self.__sort_order)
        
        try:
            for entry in self.chromosomes:
                self._chrid_dic[self.chromosomes[entry].chrid] = entry
        except:
            for entry in numpy.unique(self._ordering_array["chr"]):
                self._chrid_dic[entry] = None        
    
    def __region_to_tuple(self, thisregion, region_type, region_id):
        
        """ build a tuple appropriate for the ordering array from a region 
        
        This should contain 7 elements, see the __ordering_arrray_dtype"""
        
        name=""
        if "name" in thisregion.desc.keys():
            name = thisregion.desc["name"].lower()
        
        this_tuple = (region_type, region_id, thisregion.chrid,
                      thisregion.start, thisregion.stop, 
                      thisregion.strand, name)
        
        return(this_tuple)
    
    def set_feature(self, feature_str):
        
        """ set the feature type you're interested in """
        
        if type(feature_str) is not str:
            msg = "Feature type should be a string. Available " \
                  "features are: %s" % ", ".join(self._featurelist)
            raise TypeError(msg)
                
        if feature_str not in self._featurelist:
            msg = "Feature type %s not found in annotation. Available " \
                  "features are: %s" % (feature_str,
                                        ", ".join(self._featurelist))
            raise ValueError(msg)
        
        self.current_feature = feature_str
        self._feature_index = numpy.where(self._ordering_array["type"]==self.current_feature)[0]
    
    def clear_feature_selection(self):
        
        """ clears the variables defining the current feature selection """
        
        self.current_feature=None
        self._feature_index=None
    
    def set_region(self, region_rep, start=None, stop=None, strand=None):
        
        """ set the feature region you're interested in. 
        
        Input can either be a 'region' instance or a string that will be parsed 
        into a 'region' instance. See 
        parsing_routines.general_classes_and_functions.py for details on the 
        input for the 'region' constructor.
        """
        
        this_region = generic_set_region(region_rep, start, stop, strand)
                
        # check selected chromosome is in the wigData tracks
        if this_region.chrid not in self._chrid_dic.keys():
            msg = "Invalid region specification, chromosome id not found in " \
                  "annotation. Regions should be specified either as a " \
                  "string 'chrid:start-stop' or with three arguments as " \
                  "set_region('chrid', [start_int], [stop_int])."
            raise ValueError(msg)
        
        # sanitize stupid region start/stop values
        if this_region.start is None:
            this_region.start = 1
        if this_region.stop is None:
            
            # if there is a chromosome that matches get the max val from there, 
            # other wise get the max val from the max stop of an annotation 
            # on that chr.
            if self._chrid_dic[this_region.chrid] is not None:
                this_region.stop = self.chromosomes[self._chrid_dic[this_region.chrid]].stop
            else:            
                chr_id_ind = numpy.where(self._ordering_array["chr"]==this_region.chrid)[0]
                feature_ends = self._ordering_array["stop"][chr_id_ind]
                this_region.stop = feature_ends.max()
        
        self.current_region = this_region
        
        chr_ind = numpy.where(self._ordering_array["chr"]==self.current_region.chrid)[0]
        
        # we want to include features where the start and/or stop occur partway 
        # through the feature. Hence why in indexes are stop>blah and start<blah 
        start_ind = numpy.where(self._ordering_array["stop"]>=self.current_region.start)[0]
        stop_ind = numpy.where(self._ordering_array["start"]<self.current_region.stop)[0]
        
        # if the strandedness is set in the region then use that as an index too
        strand_ind = numpy.arange(len(self._ordering_array["start"]))
        if self.current_region.strand is not None:
            strand_ind = numpy.where(self._ordering_array["strand"]==self.current_region.strand)[0]        
        
        the1_ind = numpy.intersect1d(chr_ind, strand_ind, assume_unique=True)
        the1_ind = numpy.intersect1d(the1_ind, start_ind, assume_unique=True)
        the1_ind = numpy.intersect1d(the1_ind, stop_ind, assume_unique=True)
        self._region_index = the1_ind

    def clear_region_selection(self):
        
        """ clears the variables defining the current feature selection """
        
        self.current_region=None
        self._region_index=None

    def set_name_selection(self, name_str):
        
        """ set the name of the feature your are interested in """

        if type(name_str) is not str:
            msg = "Feature name should be a string."
            raise TypeError(msg)
        
        self.current_name = name_str.lower()
        self._name_index = numpy.where(self._ordering_array["name"]==self.current_name)[0]
    
    def clear_name_selection(self):
        
        """ clears the variables defining the current name selection """
        
        self.current_name=None
        self._name_index=None
        
    def set_attribute_filters(self, this_dict):
        
        """ filters regions for only those with a given attribute value"""
        
        if type(this_dict) is not dict:
            msg = "Attribute filters should be a dictionary of the form: " \
                  "{atribute_name:value, ... }."
            raise TypeError(msg)
        self.current_attributes=this_dict
        
    def clear_attribute_filter(self):
        
        """ clears the variables defining the current feature selection """
        
        self.current_attributes=None
    
    def clear_all(self):
        
        """ clears all the selectors and filter options """
        
        self.current_feature=None
        self._feature_index=None
        self.current_region=None
        self._region_index=None
        self.current_name=None
        self._name_index=None
        self.current_attributes=None
    
    def __build_final_index(self):
        
        """ return a combined index based on feature & region indexes """
        
        indexes = [numpy.arange(len(self._ordering_array["start"]))]
        if self._region_index is not None:
            indexes.append(self._region_index)
        if self._feature_index is not None:
            indexes.append(self._feature_index)
        if self._name_index is not None:
            indexes.append(self._name_index)
        
        if len(indexes)==1 and self.current_attributes is None:
            raise ValueError("No selections specified. Returning all regions!")
        
        final_ind = indexes[0]
        i=1
        while i<len(indexes):
            final_ind = numpy.intersect1d(final_ind, indexes[i],
                                          assume_unique=True)
            i+=1
        return(final_ind)
        
    def get_selection(self, match_case=False, return_indexes=False):
        
        """ return the selected annotations based on feature & region indexes """
        
        final_ind = self.__build_final_index()
        selection = self._ordering_array[final_ind]
        selected_regions=[]
        i=0
        for selected in selection:   
            try:
                this_region = self._full_annotation[selected[0]][selected[1]]
            except IndexError:
                this_region = self._full_annotation[selected[0]]
            
            include_this_region=True
            if self.current_attributes is not None:
                include_this_region=False
                for key in self.current_attributes:
                    if key in this_region.desc.keys():
                        if match_case and self.current_attributes[key]==this_region.desc[key]:
                            include_this_region=True
                        elif not match_case and self.current_attributes[key].lower()==this_region.desc[key].lower():
                            include_this_region=True
            
            if include_this_region:
                if return_indexes:
                    selected_regions.append(final_ind[i])
                else:
                    selected_regions.append(this_region)
            i+=1
        
        if len(selected_regions)==0:
            raise ValueError("No regions selected. Nothing to get!")
        
        return(selected_regions)
    
    def get_selection_starts(self, index=None):

        """ return the start positions of the selected annotations 
        
        The keyword arg 'index' allows a specific index to be supplied. for the
        start position selection. This should be used mostly internally ;)
        """
        
        if index is None:
            final_ind = self.get_selection(return_indexes=True)
        else:
            final_ind = index
        
        starts = self._ordering_array["start"][final_ind]
        return(starts)

    def get_selection_stops(self, index=None):

        """ return the stop positions of the selected annotations

        The keyword arg 'index' allows a specific index to be supplied. for the
        stop position selection. This should be used mostly internally ;)
        """
        
        
        if index is None:
            final_ind = self.get_selection(return_indexes=True)
        else:
            final_ind = index

        stops = self._ordering_array["stop"][final_ind]
        return(stops)
    
    def get_selection_lengths(self):

        """ return the lengths of the selected annotations """
        
        final_ind = self.get_selection(return_indexes=True)
        
        return(self.get_selection_stops(final_ind)-self.get_selection_starts(final_ind))
    
    def get_selection_sum_length(self):
        
        """ return the sum of the lengths of the selected annotations """
        
        return(self.get_selection_lengths().sum())

    def get_selection_mean_length(self):
        
        """ return the mean of the lengths of the selected annotations """
        
        return(self.get_selection_lengths().mean())
    
    def get_selection_median_length(self):
    
        """ return the median of the lengths of the selected annotations """
        
        return(numpy.median(self.get_selection_lengths()))

    def get_selection_stddev_length(self):
    
        """ return the unbiassed standard deviation of the lengths of the 
        selected annotations. 
        
        Here we correct numpys typical standard deviation routine to make it an
        unbiased estimator, according to GURLAND, J. & TRIPATHI, R. C. 1971. 
        "Simple Approximation for Unbiased Estimation of Standard Deviation." 
        American Statistician, 25, 30. Note that this correction will propogate 
        through to the standard error on the mean. 

        """
        
        return(self.get_selection_lengths().std(ddof=1)*
               (1.0+(1.0/(4.0*(len(self.get_selection_lengths())-1.0)))))

    def get_selection_stderr_length(self):

        """ return the unbiassed standard error on the mean of the lengths of 
        the selected annotations."""
                
        return(self.get_selection_stddev_length()/len(self.get_selection_lengths()))
        
    def get_selection_lengthstats(self):

        """ return the lengths of the selected annotations """

        return(self.get_selection_mean_lengths(),
               self.get_selection_median_lengths(),
               self.get_selection_stddev_lengths(),
               self.get_selection_stderr_lengths())
    

            
            
        
        
