#!/usr/bin/python
'''
-------------------
tests.gff_gtf_tools.py
-------------------

A module containing tests for the gff_gtf_tools module I'm 
using in :ref:`gff_gtf_tools.py <gff_gtf_tools_autodocs>`. These tests use 
the standard `unittest <http://docs.python.org/2/library/unittest.html>`_ 
package and extend the :py:class:`unittest.TestCase` class.

.. moduleauthor:: Nick Schurch <nschurch@dundee.ac.uk>

:module_version: 1.0
:created_on: 2013-11-27

'''

__version__ = "1.0"

import unittest, os, tempfile, shutil, argparse, warnings, numpy
from numpy.testing import *
from parsing_routines.gff_gtf_tools import parse_gff_v3, annotation
from parsing_routines.general_classes_and_functions import region, generic_set_region

class TestGFFGTFTools(unittest.TestCase):

    ''' Test the class and functions defined in :ref:`gff_gtf_tools.py 
    <gff_gtf_tools_autodocs>`.
    
    Each function is tested with existing and non-existent instances of the 
    file or path type that it is designed for. We also test that providing non-
    string basic python objects produces string representations and doesn't 
    case a failure.
    
    '''

    def setUp(self):
        
        ''' Initialize the framework for testing. 
        
        Define and create a new system temporary directory with the
        `tempfile <http://docs.python.org/2/library/tempfile.html>`_ package
        for use as an existing directory. A range of gff and gtf files with 
        different parameters are created in this directory to test the routines 
        with. 
        '''
        
        # setup a new temp directory
        self.existing_path = tempfile.mkdtemp()
        
        # make a very simple gff v3 file
        self.simple = "%s/simple_gff_v3.gff" % self.existing_path
        temp_file = open(self.simple, "w")
        temp_file.write("##gff-version 3\n# These are \n# header\n# lines\n")
        temp_file.write("chrI\t.\tchromosome\t1\t10000\t.\t.\t.\t.\n")
        temp_file.write("chrI\t.\texon\t1000\t2000\t.\t-\t.\t.\n")
        temp_file.write("chrI\t.\tgene\t900\t2200\t.\t-\t.\t.\n")
        temp_file.write("chr2\t.\tchromosome\t1\t20000\t.\t.\t.\t.\n")
        temp_file.write("chr2\t.\texon\t2000\t4000\t.\t+\t.\t.\n")
        temp_file.write("chr2\t.\tgene\t1800\t4400\t.\t+\t.\t.\n")
        temp_file.close()

        # make a very simple but wrong gff v3 file
        self.simple_wrong = "%s/simple_wrong_gff_v3.gff" % self.existing_path
        temp_file = open(self.simple_wrong, "w")
        temp_file.write("##gff-version 3\n# These are \n# header\n# lines\n")
        temp_file.write("chrI\t.\tchromosome\t1\t10000\t.\t.\t.\n")
        temp_file.write("chrI\t.\texon\t1000\t2000\t.\t-\t.\t.\n")
        temp_file.write("chrI\t.\tgene\t900\t2200\t.\t-\t.\t.\n")
        temp_file.write("chr2\t.\tchromosome\t1\t20000\t.\t.\t.\t.\n")
        temp_file.write("chr2\t.\texon\t2000\t4000\t.\t+\t.\t.\n")
        temp_file.write("chr2\t.\tgene\t1800\t4400\t.\t+\t.\t.\n")
        temp_file.close()

        # make a very simple gff v3 file with no types
        self.simple_notype = "%s/simple_notype_gff_v3.gff" % self.existing_path
        temp_file = open(self.simple_notype, "w")
        temp_file.write("##gff-version 3\n# These are \n# header\n# lines\n")
        temp_file.write("chrI\t.\t.\t1\t20000\t.\t.\t.\t.\n")
        temp_file.write("chrI\t.\t.\t1000\t2000\t.\t-\t.\t.\n")
        temp_file.write("chrI\t.\t.\t900\t2200\t.\t-\t.\t.\n")
        temp_file.write("chr2\t.\t.\t1\t20000\t.\t.\t.\t.\n")
        temp_file.write("chr2\t.\t.\t2000\t4000\t.\t+\t.\t.\n")
        temp_file.write("chr2\t.\t.\t1800\t4400\t.\t+\t.\t.\n")
        temp_file.close()

        # make a complex gff v3 file
        self.complex = "%s/complex_gff_v3.gff" % self.existing_path
        temp_file = open(self.complex, "w")
        temp_file.write("##gff-version 3\n# These are \n# header\n# lines\n")
        temp_file.write("chrI\ttest\tchromosome\t1\t10000\t.\t.\t.\t.\n")
        temp_file.write("chrI\ttest\texon\t1000\t2000\t.\t-\t.\t" \
                        "ID=TESTGENE_EXON1;NAME=BOB;PARENT=FRANK\n")
        temp_file.write("chrI\ttest\tgene\t900\t2200\t.\t-\t.\t" \
                        "ID=TESTGENE;NAME=FRANK\n")
        temp_file.write("chrI\ttest\ttelomere\t1\t5000\t.\t.\t.\t.\n")
        temp_file.write("chrI\ttest\tnucleotide_match\t2048\t3100\t200\t+\t" \
                        "0\t.\n")        
        temp_file.write("chr2\ttest\tchromosome\t1\t20000\t.\t.\t.\t.\n")
        temp_file.write("chr2\ttest\texon\t2000\t4000\t.\t+\t.\t.\n")
        temp_file.write("chr2\ttest\tgene\t1800\t4400\t.\t+\t.\t.\n")
        temp_file.close()

        # make a more detailed complex gff v3 file
        self.detailed_complex = "%s/detailed_complex_gff_v3.gff" % self.existing_path
        temp_file = open(self.detailed_complex, "w")
        temp_file.write("##gff-version 3\n# These are \n# header\n# lines\n")
        temp_file.write("chrI\ttest\tchromosome\t1\t10000\t.\t.\t.\t.\n")
        temp_file.write("chrI\ttest\texon\t2000\t3000\t.\t-\t.\t" \
                        "ID=TESTGENE_EXON1;NAME=BOB;PARENT=FRANK\n")
        temp_file.write("chrI\ttest\tgene\t1800\t3200\t.\t-\t.\t" \
                        "ID=TESTGENE;NAME=FRANK\n")
        temp_file.write("chrI\ttest\texon\t2500\t3500\t.\t+\t.\t" \
                        "ID=TESTGENE2_EXON1;NAME=BOBFISH;PARENT=FRANKFISH\n")
        temp_file.write("chrI\ttest\tgene\t2300\t3700\t.\t+\t.\t" \
                        "ID=TESTGENE2;NAME=FRANKFISH\n")
        temp_file.write("chrI\ttest\texon\t7000\t7500\t.\t-\t.\t" \
                        "ID=TESTGENE3_EXON1;NAME=BOBCOW;PARENT=FRANKBIRD\n")
        temp_file.write("chrI\ttest\texon\t7600\t7700\t.\t-\t.\t" \
                        "ID=TESTGENE3_EXON2;NAME=BOBPIG;PARENT=FRANKBIRD\n")
        temp_file.write("chrI\ttest\tgene\t6800\t8200\t.\t-\t.\t" \
                        "ID=TESTGENE3;NAME=FRANKBIRD\n")
        temp_file.write("chrI\ttest\ttelomere\t1\t1000\t.\t.\t.\t.\n")
        temp_file.write("chrI\ttest\tnucleotide_match\t2048\t3100\t200\t+\t" \
                        "0\t.\n")
        temp_file.write("chr2\ttest\tchromosome\t1\t20000\t.\t.\t.\t.\n")
        temp_file.write("chr2\ttest\texon\t2000\t4000\t.\t+\t.\t.\n")
        temp_file.write("chr2\ttest\tgene\t1800\t4400\t.\t+\t.\t.\n")
        temp_file.close()

        # use a real gff v3 from yeast
        self.real = "test_fixtures/test_gff_v3.gff"
        
    def test_parse_gff_v3(self):
        
        """ Test the parsing routine for v3 gff files 
        
        =========================    ==========================================
        Test object                  Expectation
        =========================    ==========================================
        simple gff file              correct wigData object with correct 
                                     wigTrack objects
        gff w. incorrect data        IndexError
        """
        
        x, headerlines = parse_gff_v3(self.simple)
        self.assertTrue(type(x) is dict)
        self.assertEqual(len(x.keys()),3)
        
        types = ["chromosomes", "exons", "genes"]
        type_keys=[0,1]
        
        for y in types:
            self.assertIn(y, x.keys())
            for z in type_keys:
                self.assertIn(z, x[y].keys())
        
        # test fail with gff with incorrect number of cols
        self.assertRaises(IndexError, parse_gff_v3, self.simple_wrong)
        
        # test behaviour if no type designation in gff
        x, headerlines = parse_gff_v3(self.simple_notype)        
        self.assertTrue(type(x) is dict)
        self.assertEqual(len(x.keys()),6)
        line_numbers=[4,5,6,7,8,9]
        for y in line_numbers:
            self.assertIn(y, x.keys())
            self.assertTrue(x[y].start>0)
            self.assertTrue(x[y].stop>0)
            self.assertTrue(type(x[y].chrid) is str)
                
        x, headerlines = parse_gff_v3(self.complex)
        self.assertEqual(len(x.keys()),5)
        
        types = ["chromosomes", "exons", "genes", "telomeres",
                 "nucleotide_matchs"]
        
        for y in types:
            self.assertIn(y, x.keys())
        
        self.assertTrue(type(x["exons"][0].desc) is dict)
        attributes = ["ID", "NAME", "PARENT", "source", "score", 
                      "type", "phase"]
        for y in attributes:
            self.assertIn(y.lower(), x["exons"][0].desc.keys())
        
        values=["TESTGENE_EXON1","BOB","FRANK"]
        for y in values:
            self.assertIn(y, x["exons"][0].desc.values())
        
        values=["TESTGENE", "FRANK"]
        for y in values:
            self.assertIn(y, x["genes"][0].desc.values())
        
        self.assertAlmostEqual(x["nucleotide_matchs"][0].desc["score"], 200.0)
        self.assertEqual(x["nucleotide_matchs"][0].desc["phase"], 0)
        
        x, headerlines=parse_gff_v3(self.real, progress_report=True)
        self.assertEqual(len(x.keys()),26)
        self.assertEqual(len(x["chromosomes"].keys()),17)
        
        x, headerlines=parse_gff_v3(self.real, skip_fasta=False,
                                    progress_report=True)
        for entry in x["chromosomes"].keys():
            self.assertTrue(x["chromosomes"][entry].sequence is not None)

    def test_annotation(self):
        
        """ tests for the annotations class 
        
        ==============================   =======================================
        Test object                      Expectation
        ==============================   =======================================
        small complex gff file           correct annotation object 
        set_feature arg not string       TypeError
        set_feature arg not in feature   ValueError 
            types   
        set_feature correct args         sorting index for just that feature
        """
        
        x=annotation(self.complex)
        self.assertRaises(TypeError, x.set_feature, 1.0)
        self.assertRaises(ValueError, x.set_feature, "bobfish")
         
        # check that the feature can be set correctly
        x.set_feature("genes")
        self.assertEqual(len(x._feature_index), 2)
        self.assertEqual(numpy.unique(x._ordering_array["type"][x._feature_index])[0],
                         "genes")
         
        # check that stop is set correctly for whole chromosome selecions
        x=annotation(self.simple_notype)
        thisreg = generic_set_region("chrI")
        x.set_region(thisreg)
        self.assertEqual(x.current_region.stop, 20000)
         
        # check that the region selection returns the crrect selection indecies
        x=annotation(self.detailed_complex)
        x.set_region("chrI")
        self.assertEqual(len(x._region_index),10)
        x.set_region("chrI:1-1000")
        self.assertEqual(len(x._region_index),2)
        self.assertEqual(x._region_index[0],3)
        self.assertEqual(x._region_index[1],4)
        x.set_region("chrI:6500-9000")
        self.assertEqual(len(x._region_index),4)
        self.assertEqual(x._region_index[0],4)
        self.assertEqual(x._region_index[1],10)
        self.assertEqual(x._region_index[2],11)
        self.assertEqual(x._region_index[3],12)
        x.set_region("chrI:7200-9000")
        self.assertEqual(len(x._region_index),4)
        self.assertEqual(x._region_index[0],4)
        self.assertEqual(x._region_index[1],10)
        self.assertEqual(x._region_index[2],11)
        self.assertEqual(x._region_index[3],12)
        x.set_region("chrI:6500-7650")
        self.assertEqual(len(x._region_index),4)
        self.assertEqual(x._region_index[0],4)
        self.assertEqual(x._region_index[1],10)
        self.assertEqual(x._region_index[2],11)
        self.assertEqual(x._region_index[3],12)
        x.set_region("chrI", start=500, stop=5000, strand="+")
        self.assertEqual(len(x._region_index),3)
        self.assertEqual(x._region_index[0],7)
        self.assertEqual(x._region_index[1],8)
        self.assertEqual(x._region_index[2],9)
         
        # check that nothing bad happens if the selected region is empty
        x.set_region("chrI", start=6500, stop=9000, strand="+")
        self.assertEqual(len(x._region_index),0)
         
        # check that the right regions are returned with get_selection, 
        # including feature, regions and the combo selection
        x.set_region("chrI", start=6500, stop=9000, strand="-")
        regs = x.get_selection()
        self.assertEqual(len(regs),3)
        for reg in regs:
            self.assertTrue(reg.stop>6500)
            self.assertTrue(reg.start<9000)
            self.assertTrue(reg.chrid=="chrI")
            self.assertTrue(reg.strand=="-")
         
        x.set_feature("genes")
        regs = x.get_selection()
        self.assertEqual(len(regs),1)
        reg=regs[0]
        self.assertTrue(reg.stop==8200)
        self.assertTrue(reg.start==6800)
        self.assertTrue(reg.chrid=="chrI")
        self.assertTrue(reg.strand=="-")
         
        x.set_feature("exons")
        regs = x.get_selection()
        self.assertEqual(len(regs),2)
        
        x=annotation(self.detailed_complex)
        x.set_feature("exons")
        x.set_region("chrI", start=6500, stop=9000, strand="+")     
        self.assertRaises(ValueError, x.get_selection)

        x.set_region("chrI", start=6500, stop=9000, strand="-")
        x.clear_feature_selection()
        
        vals = [7000,7600,6800]
        for val in vals:
            self.assertIn(val, x.get_selection_starts())
        
        vals = [7500,7700,8200]
        for val in vals:
            self.assertIn(val, x.get_selection_stops())

        vals = [500,100,1400]
        for val in vals:
            self.assertIn(val, x.get_selection_lengths())

        self.assertAlmostEqual(x.get_selection_mean_length(), 666.66666666666)
        self.assertAlmostEqual(x.get_selection_median_length(), 500.0)
        self.assertAlmostEqual(x.get_selection_stddev_length(), 749.06191332893)
        self.assertAlmostEqual(x.get_selection_stderr_length(), 249.68730444297)
        
        x.set_feature("genes")
        x.clear_region_selection()
        
        vals = [1800,2300,6800,1800]
        for val in vals:
            self.assertIn(val, x.get_selection_starts())
        
        vals = [3200,3700,8200,4400]
        for val in vals:
            self.assertIn(val, x.get_selection_stops())

        vals = [1400, 1400, 1400, 2600]
        for val in vals:
            self.assertIn(val, x.get_selection_lengths())

        self.assertAlmostEqual(x.get_selection_mean_length(), 1700.0)
        self.assertAlmostEqual(x.get_selection_median_length(), 1400.0)
        self.assertAlmostEqual(x.get_selection_stddev_length(), 650.0)
        self.assertAlmostEqual(x.get_selection_stderr_length(), 162.5)

        x.clear_feature_selection()
        self.assertRaises(ValueError, x.get_selection_mean_length)
        
        x.clear_feature_selection()
        x.clear_region_selection()
        x.set_name_selection("FRANK")
        regs = x.get_selection()
        self.assertEqual(len(regs),1)
        reg=regs[0]
        self.assertTrue(reg.start==1800)
        self.assertTrue(reg.stop==3200)
        self.assertTrue(reg.chrid=="chrI")
        self.assertTrue(reg.strand=="-")
        
        x.set_name_selection("frank")
        regs = x.get_selection()
        self.assertEqual(len(regs),1)
        reg=regs[0]
        self.assertTrue(reg.start==1800)
        self.assertTrue(reg.stop==3200)
        self.assertTrue(reg.chrid=="chrI")
        self.assertTrue(reg.strand=="-")
        
        x.clear_name_selection()
        x.set_attribute_filters({"parent":"FRANKBIRD"})
        regs = x.get_selection()
        self.assertEqual(len(regs),2)
        reg=regs[0]
        self.assertTrue(reg.start==7000)
        self.assertTrue(reg.stop==7500)
        self.assertTrue(reg.chrid=="chrI")
        self.assertTrue(reg.strand=="-")
        reg=regs[1]
        self.assertTrue(reg.start==7600)
        self.assertTrue(reg.stop==7700)
        self.assertTrue(reg.chrid=="chrI")
        self.assertTrue(reg.strand=="-")
        
        x.set_attribute_filters({"parent":"frankbird"})
        regs = x.get_selection()
        self.assertEqual(len(regs),2)
        reg=regs[0]
        self.assertTrue(reg.start==7000)
        self.assertTrue(reg.stop==7500)
        self.assertTrue(reg.chrid=="chrI")
        self.assertTrue(reg.strand=="-")
        reg=regs[1]
        self.assertTrue(reg.start==7600)
        self.assertTrue(reg.stop==7700)
        self.assertTrue(reg.chrid=="chrI")
        self.assertTrue(reg.strand=="-")
        
        regs = x.get_selection(return_indexes=True)
        self.assertEqual(len(regs),2)
        self.assertEqual(regs[0],11)
        self.assertEqual(regs[1],12)
        
        self.assertEqual(x.get_selection_sum_length(), 600)

if __name__ == '__main__':
    unittest.main(verbosity=2)      