#!/usr/bin/python
'''
-------------------
tests.wig_tools.py
-------------------

A module containing tests for the wig_tools module I'm 
using in :ref:`wig_tools.py <wig_tools_autodocs>`. These tests use 
the standard `unittest <http://docs.python.org/2/library/unittest.html>`_ 
package and extend the :py:class:`unittest.TestCase` class.

.. moduleauthor:: Nick Schurch <nschurch@dundee.ac.uk>

:module_version: 1.0
:created_on: 2013-11-22

'''

__version__ = "1.0"

import unittest, os, tempfile, shutil, argparse, warnings, numpy, time
from numpy.testing import *
import parsing_routines.wig_tools as wig
       
class TestWigTools(unittest.TestCase):

    ''' Test the class and functions defined in :ref:`wig_tools.py 
    <wig_tools_autodocs>`.
    
    Each function is tested with existing and non-existent instances of the 
    file or path type that it is designed for. We also test that providing non-
    string basic python objects produces string representations and doesn't 
    case a failure.
    
    '''

    def setUp(self):
        
        ''' Initialize the framework for testing. 
        
        Define and create a new system temporary directory with the
        `tempfile <http://docs.python.org/2/library/tempfile.html>`_ package
        for use as an existing directory. A range of wig files with different
        parameters are created in this directory to test the routines with. 
        '''
        
        # setup a new temp directory
        self.existing_path = tempfile.mkdtemp()
        
        # make a very simple fixedStep wig file
        self.very_simple = "%s/fixedStep_01.wig" % self.existing_path
        temp_file = open(self.very_simple, "w")
        temp_file.write("fixedStep chrom=chr1 start=1 step=1\n")
        temp_file.write("11\n")
        temp_file.write("22\n")
        temp_file.write("33\n")
        temp_file.close()
        
        # make a simple fixedStep wig file
        self.simple = "%s/fixedStep_02.wig" % self.existing_path
        temp_file = open(self.simple, "w")
        temp_file.write("fixedStep chrom=chr1 start=100 step=1\n")
        temp_file.write("11\n")
        temp_file.write("22\n")
        temp_file.write("33\n")
        temp_file.close()

        # make a complex fixedStep wig file
        self.complex = "%s/fixedStep_03.wig" % self.existing_path
        temp_file = open(self.complex, "w")
        temp_file.write("fixedStep chrom=chr1 start=100 step=50\n")
        temp_file.write("11\n")
        temp_file.write("22\n")
        temp_file.write("33\n")
        temp_file.close()

        # make a very complex fixedStep wig file
        self.very_complex = "%s/fixedStep_04.wig" % self.existing_path
        temp_file = open(self.very_complex, "w")
        temp_file.write("fixedStep chrom=chr1 start=100 step=50 span=5\n")
        temp_file.write("11\n")
        temp_file.write("22\n")
        temp_file.write("33\n")
        temp_file.close()
        
        # make a multi-entry very complex fixedStep wig file
        self.multitrack_very_complex = "%s/fixedStep_05.wig" % self.existing_path
        temp_file = open(self.multitrack_very_complex, "w")
        temp_file.write("fixedStep chrom=chr1 start=100 step=50 span=5\n")
        temp_file.write("11\n")
        temp_file.write("22\n")
        temp_file.write("33\n")
        temp_file.write("fixedStep chrom=chr2 start=200 step=50 span=5\n")
        temp_file.write("44\n")
        temp_file.write("55\n")
        temp_file.write("66\n")
        temp_file.close()      

        # make a very simple variableStep wig file
        self.simple_variable = "%s/variableStep_01.wig" % self.existing_path
        temp_file = open(self.simple_variable, "w")
        temp_file.write("variableStep chrom=chr1\n")
        temp_file.write("1 11\n")
        temp_file.write("2 22\n")
        temp_file.write("3 33\n")
        temp_file.close()
        
        # make a complex variableStep wig file
        self.complex_variable = "%s/variableStep_02.wig" % self.existing_path
        temp_file = open(self.complex_variable, "w")
        temp_file.write("variableStep chrom=chr1 span=5\n")
        temp_file.write("1 11\n")
        temp_file.write("10 22\n")
        temp_file.write("20 33\n")
        temp_file.close()
        
        # make a multi-entry complex variableStep wig file
        self.multitrack_complex_variable = "%s/variableStep_03.wig" % self.existing_path
        temp_file = open(self.multitrack_complex_variable, "w")
        temp_file.write("variableStep chrom=chr1 span=5\n")
        temp_file.write("1 11\n")
        temp_file.write("10 22\n")
        temp_file.write("20 33\n")
        temp_file.write("variableStep chrom=chr2 span=5\n")
        temp_file.write("30 44\n")
        temp_file.write("40 55\n")
        temp_file.write("50 66\n")
        temp_file.close()   
        
        # test on a 'real' wigfile
        self.realfile = "test_fixtures/test.wig" # this should be in the test_fixtures directory
        
        # test on a 'real' bigwigfile
        self.realBigWigfile = "test_fixtures/test.bigwig" # this should be in the test_fixtures directory

        # make up a mythical path and file based on the existing dir
        self.nonexisting_file = self.existing_path+"/thisdoesntexists.wig"

    def test_wigTrack_fixedStep(self):
            
        ''' Test the behaviour of wigTrack with fixedStep wig files
        
        ===================    ==========================================
        Test object            Expectation
        ===================    ==========================================
        very simple wigfile    correct wigTrack object
        simple wigfile         correct wigTrack object
        complex wigfile        correct wigTrack object
        very complex wigfile   correct wigTrack object
        ===================    ==========================================
        '''
        
        filehandle = open(self.very_simple, "r")
        filedata = filehandle.readlines()
        filehandle.close()
        
        # test very simple case
        x = wig.wigTrack(filedata)
        self.assertEqual(x.wigtype, "fixedStep")
        self.assertEqual(x.chr, "chr1")
        self.assertEqual(x.step, 1)
        self.assertEqual(x.span, 1)
        self.assertEqual(x.start, 1)
        assert_array_equal(x.data, numpy.array([11,22,33]))
        assert_array_equal(x.position, numpy.array([1,2,3]))
        self.assertEqual(x.region.chrid, "chr1")
        self.assertEqual(x.region.start, 1)
        self.assertEqual(x.region.stop, x.position.max()+1)
        
        filehandle = open(self.simple, "r")
        filedata = filehandle.readlines()
        filehandle.close()

        # test simple case
        x = wig.wigTrack(filedata)
        self.assertEqual(x.start, 100)
        assert_array_equal(x.data, numpy.array([11,22,33]))
        assert_array_equal(x.position, numpy.array([100,101,102]))

        filehandle = open(self.complex, "r")
        filedata = filehandle.readlines()
        filehandle.close()

        # test complex case
        x = wig.wigTrack(filedata)
        self.assertEqual(x.start, 100)
        self.assertEqual(x.step, 50)
        assert_array_equal(x.data, numpy.array([11,22,33]))
        assert_array_equal(x.position, numpy.array([100,150,200]))

        filehandle = open(self.very_complex, "r")
        filedata = filehandle.readlines()
        filehandle.close()
        
        # test very complex case
        x = wig.wigTrack(filedata)
        self.assertEqual(x.start, 100)
        self.assertEqual(x.step, 50)
        self.assertEqual(x.span, 5)
        assert_array_equal(x.data, numpy.array([11,11,11,11,11,22,22,22,22,22,
                                                33,33,33,33,33]))
        assert_array_equal(x.position, numpy.array([100,101,102,103,104,
                                                    150,151,152,153,154,
                                                    200,201,202,203,204]))

    def test_wigTrack_variableStep(self):
            
        ''' Test the behaviour of wigTrack with variableStep wig files
        
        ===================    ==========================================
        Test object            Expectation
        ===================    ==========================================
        simple wigfile         correct wigTrack object
        complex wigfile        correct wigTrack object
        ===================    ==========================================
        '''
        
        filehandle = open(self.simple_variable, "r")
        filedata = filehandle.readlines()
        filehandle.close()
        
        # test very simple case
        x = wig.wigTrack(filedata)
        self.assertEqual(x.wigtype, "variableStep")
        self.assertEqual(x.chr, "chr1")        
        self.assertIsNone(x.step)
        self.assertEqual(x.span, 1)
        self.assertIsNone(x.start)
        assert_array_equal(x.data, numpy.array([11,22,33]))
        assert_array_equal(x.position, numpy.array([1,2,3]))
        self.assertEqual(x.region.chrid, "chr1")
        self.assertEqual(x.region.start, 1)
        self.assertEqual(x.region.stop, x.position.max()+1)
        
        filehandle = open(self.complex_variable, "r")
        filedata = filehandle.readlines()
        filehandle.close()

        # test simple case
        x = wig.wigTrack(filedata)
        self.assertEqual(x.span, 5)
        assert_array_equal(x.data, numpy.array([11,11,11,11,11,22,22,22,22,22,
                                                33,33,33,33,33]))
        assert_array_equal(x.position, numpy.array([1,2,3,4,5,10,11,12,13,14,
                                                    20,21,22,23,24]))
    
    def test_wigData_multi_chromosome(self):
        
        ''' test the behaviour of wigData with a file with multiple chromosomes
        
        =========================    ==========================================
        Test object                  Expectation
        =========================    ==========================================
        multi_chr complex wigfile    correct wigData object with correct 
                                     wigTrack objects
        '''
        
        x=wig.wigData(self.multitrack_very_complex)
        self.assertTrue(len(x.tracks)==2)
        self.assertEqual(x.tracks["chr1"].start, 100)
        self.assertEqual(x.tracks["chr1"].step, 50)
        self.assertEqual(x.tracks["chr1"].span, 5)
        assert_array_equal(x.tracks["chr1"].data, numpy.array([11,11,11,11,11,
                                                               22,22,22,22,22,
                                                               33,33,33,33,33]))
        assert_array_equal(x.tracks["chr1"].position, 
                           numpy.array([100,101,102,103,104,
                                        150,151,152,153,154,
                                        200,201,202,203,204]))
        self.assertEqual(x.tracks["chr2"].start, 200)
        self.assertEqual(x.tracks["chr2"].step, 50)
        self.assertEqual(x.tracks["chr2"].span, 5)
        assert_array_equal(x.tracks["chr2"].data, numpy.array([44,44,44,44,44,
                                                               55,55,55,55,55,
                                                               66,66,66,66,66]))
        assert_array_equal(x.tracks["chr2"].position, 
                           numpy.array([200,201,202,203,204,
                                        250,251,252,253,254,
                                        300,301,302,303,304]))
                 
        self.assertEqual(x.region.chrid, "chr1")
        self.assertEqual(x.region.start, 1)
        self.assertEqual(x.region.stop, x.tracks["chr1"].position.max()+1)
        self.assertEqual(x.region.chrid, x.tracks["chr1"].region.chrid)
        self.assertEqual(x.region.start, x.tracks["chr1"].region.start)
        self.assertEqual(x.region.stop, x.tracks["chr1"].region.stop)
         
        
        x=wig.wigData(self.multitrack_complex_variable)
        self.assertTrue(len(x.tracks)==2)
        self.assertIsNone(x.tracks["chr1"].start)
        self.assertIsNone(x.tracks["chr2"].start)
        self.assertIsNone(x.tracks["chr1"].step)
        self.assertIsNone(x.tracks["chr2"].step)
        self.assertEqual(x.tracks["chr1"].span, 5)
        self.assertEqual(x.tracks["chr2"].span, 5)

        assert_array_equal(x.tracks["chr1"].data,
                           numpy.array([11,11,11,11,11,22,22,22,22,22,
                                        33,33,33,33,33]))
        assert_array_equal(x.tracks["chr1"].position,
                           numpy.array([1,2,3,4,5,10,11,12,13,14,
                                        20,21,22,23,24]))
        
        assert_array_equal(x.tracks["chr2"].data,
                           numpy.array([44,44,44,44,44,55,55,55,55,55,
                                        66,66,66,66,66]))
        assert_array_equal(x.tracks["chr2"].position,
                           numpy.array([30,31,32,33,34,40,41,42,43,44,
                                        50,51,52,53,54]))

    def test_wigData_and_wigTrack_set_region(self):
        
        ''' test region setter raises sensible errors in wigTrack & wigData
        
        ============================   ==========================================
        Test object                    Expectation
        ============================   ==========================================
        whole chromosome selection     correctly set region
        '''
        
        filehandle = open(self.very_simple, "r")
        filedata = filehandle.readlines()
        filehandle.close()
         
        # test whole chromosome region selection
        
        # test very simple case
        x = wig.wigTrack(filedata)
        self.assertTrue(x.region.start,1)
        self.assertTrue(x.region.stop,x.position.max()+1)

        # test complex case
        x=wig.wigData(self.multitrack_very_complex)
        x.set_region("chr1")
        self.assertTrue(x.region.start,1)
        self.assertTrue(x.region.stop,x.tracks["chr1"].position.max()+1)
    
    def test_wigTrack_region_functions(self):
        
        ''' test that the region related class functions work correctly
        
        =========================    ==========================================
        Test object                  Expectation
        =========================    ==========================================
        no region set                raise ValueError
        len chr1:10000-50000         40 kb
        npoints chr1:10000-50000     11
        data chr1:10000-50000        tuple, 0:positions 1:data
        max chr1:10000-50000         3
        min chr1:10000-50000         1
        mean datapoint               1.4692307692307693
        mean basesignal              0.0050043231063484159
        '''
        
        # setup
        chrid="chr1"
        x=wig.wigData(self.realfile)
        x.set_region("chr1:10000-50000")
        
        # test basic functions
        self.assertEqual(x.tracks[chrid].region_length(), 40000)
        self.assertEqual(x.tracks[chrid].region_ndatapoint(), 11)
        
        # check data return
        self.assertTrue(type(x.tracks[chrid].region_data()) is tuple)
        positions = [31156, 31159, 31170, 31174, 31175, 31176, 31183, 31184,
                     31185, 31187, 31189]
        assert_array_equal(x.tracks[chrid].region_data()[0], positions)
        signal = [1, 1, 1, 1, 2, 3, 1, 2, 1, 1, 1]
        assert_array_equal(x.tracks[chrid].region_data()[1], signal)
        self.assertEqual(len(x.tracks[chrid].region_data(expand=True)[0]),
                        x.tracks[chrid].region_length())
        self.assertEqual(x.tracks[chrid].region_data(expand=True)[0].min(),
                        x.tracks[chrid].region.start)
        self.assertEqual(x.tracks[chrid].region_data(expand=True)[0].max(),
                        x.tracks[chrid].region.stop-1)
        self.assertEqual(len(x.tracks[chrid].region_data(expand=True)[1]),
                        x.tracks[chrid].region_length())
        self.assertEqual(x.tracks[chrid].region_data(expand=True)[1].sum(),
                        x.tracks[chrid].region_data()[1].sum())
        self.assertEqual(x.tracks[chrid].region_data(expand=True)[1].max(),
                        x.tracks[chrid].region_data()[1].max())
        
        # check min and max values
        self.assertTrue(type(x.tracks[chrid].region_min()) is tuple)
        self.assertAlmostEqual(x.tracks[chrid].region_min()[0], 1.0)
        min_indexes = numpy.where(numpy.array(signal)==1.0)[0]
        minpos = numpy.array(positions)[min_indexes]
        assert_array_equal(x.tracks[chrid].region_min()[1], minpos)
        self.assertTrue(type(x.tracks[chrid].region_max()) is tuple)
        self.assertAlmostEqual(x.tracks[chrid].region_max()[0], 3.0)
        max_indexes = numpy.where(numpy.array(signal)==3.0)[0]
        maxpos = numpy.array(positions)[max_indexes]
        assert_array_equal(x.tracks[chrid].region_max()[1], maxpos)

        # check mean, median, stddev, stderr and allstats
        self.assertAlmostEqual(x.tracks[chrid].region_mean(),
                               numpy.array(signal).mean())
        self.assertAlmostEqual(x.tracks[chrid].region_median(),
                               numpy.median(numpy.array(signal)))
        self.assertNotEqual(x.tracks[chrid].region_stddev(),
                            numpy.array(signal).std())
        self.assertNotEqual(x.tracks[chrid].region_stddev(),
                            numpy.array(signal).std(ddof=1))
        self.assertAlmostEqual(x.tracks[chrid].region_stddev(),
                               numpy.array(signal).std(ddof=1)*1.024999999999)
        self.assertAlmostEqual(x.tracks[chrid].region_stderr(),
                               (numpy.array(signal).std(ddof=1)*1.024999999)/11)
        self.assertTrue(type(x.tracks[chrid].region_stats()) is tuple)
        self.assertTrue(len(x.tracks[chrid].region_stats())==4)
        self.assertAlmostEqual(x.tracks[chrid].region_fracbases(), 11.0/40000)
        self.assertAlmostEqual(x.tracks[chrid].region_mean_per_base(),
                               numpy.array(signal).sum()/40000.0)

    def test_wigData_BigWig(self):
        
        ''' test that the wigData class correctly reads BigWig files
        
        ================================  ======================================
        Test object                       Expectation
        ================================  ======================================
        filename not a String             TypeError
        file doesn't exist                IOError
        bigWigToWig binary doesn't exist  IOError
        corrent args                      correct wigData object
        '''

        self.assertRaises(TypeError, wig.wigData, 1, True)    
        self.assertRaises(IOError, wig.wigData, self.nonexisting_file, True)
        self.assertRaises(IOError, wig.wigData, self.realBigWigfile, True)
            
        # setup
        t0=time.time()
        x=wig.wigData(self.realBigWigfile, isBigWig=True, 
                      path_to_binary="../parsing_routines/external_libraries/bigWigToWig")
        print "loadtime was: %.2fs" % (time.time()-t0)
        self.assertEqual(len(x.tracks.keys()), 7)
        self.assertEqual(len(x.tracks["chr1"].position), 30427671)
        self.assertEqual(len(x.tracks["chr2"].position), 19698289)
        self.assertEqual(len(x.tracks["chr3"].position), 23459830)
        self.assertEqual(len(x.tracks["chr4"].position), 18585056)
        self.assertEqual(len(x.tracks["chr5"].position), 26975502)
        self.assertEqual(len(x.tracks["chrMt"].position), 366924)
        self.assertEqual(len(x.tracks["chrPt"].position), 154478)

if __name__ == '__main__':
    unittest.main(verbosity=2)