# coding: utf-8

from unittest import TestCase


class TestListContainer(TestCase):
    def _getClass(self, *args, **kwargs):
        from ..list import ListContainer
        return ListContainer(*args, **kwargs)

    def test_render_element_is_function(self):
        def test_function(i, value):
            return [value] * i

        lc = self._getClass(test_function)
        lc.render(['a', 'b', 'c'])
        self.assertEqual(
            lc,
            [
                [],
                ['b'],
                ['c', 'c'],
            ]
        )

    def test_render_normal_element(self):
        try:
            from ...model.dict import DictModel
        except ValueError:
            try:
                from model.dict import DictModel
            except ImportError:
                return

        dm = DictModel({'a': 1, 'b': 2})
        lc = self._getClass(dm)
        lc.render(3)
        self.assertEqual(
            lc,
            [
                {'a': 1, 'b': 2},
                {'a': 1, 'b': 2},
                {'a': 1, 'b': 2},
            ]
        )

        self.assertEqual(lc.count(), 3)

    def test_output(self):
        from datetime import datetime, timedelta
        from ...exceptions import OutputFileAlreadyExists
        import json
        import os
        try:
            from ...model.dict import DictModel
            from ...pattern.increment import IncrementPattern
        except ValueError:
            try:
                from model.dict import DictModel
                from pattern.increment import IncrementPattern
            except ImportError:
                return

        dm = DictModel({
            'a': IncrementPattern(),
            'b': IncrementPattern(
                datetime(2000, 1, 1),
                timedelta(days=1, hours=12)
            ),
        })

        lc = self._getClass(dm)
        lc.render(3)

        filename = 'outputtest.json'
        lc.output(filename)

        input_data = json.loads(open(filename).read())

        self.assertEqual(
            input_data,
            [
                {'a': 1, 'b': '2000-01-01T00:00:00'},
                {'a': 2, 'b': '2000-01-02T12:00:00'},
                {'a': 3, 'b': '2000-01-04T00:00:00'},
            ]
        )

        with self.assertRaises(OutputFileAlreadyExists):
            lc.output(filename)

        lc.output(filename, rewrite=True)

        os.remove(filename)
