# coding: utf-8

from __future__ import with_statement

try:
    import json
except ImportError:
    import simplejson as json

import os
from datetime import datetime
from copy import deepcopy

from ..exceptions import OutputFileAlreadyExists

json_indent = 1


class JSONEncoder(json.JSONEncoder):
    """
    Custom JSON Encoder for Container.
    """
    encoder_map = {
        datetime: (lambda dt: dt.isoformat()),
    }

    def default(self, o):
        try:
            encoder = self.encoder_map[type(o)]
            return encoder(o)
        except KeyError:
            return o


def iterize(iterable):
    """
    :param iterable: iterable objects or int or None(return None)
    :return: iterable objects
    """
    if iterable is None:
        return None

    if isinstance(iterable, int):
        try:
            return xrange(iterable)
        except NameError:
            return range(iterable)

    return iterable


class BaseContainer(object):
    """
    Base Container object.
    """
    def __init__(self, blueprint, iterable=()):
        """
        :param blueprint: pattern / model / container / function / else...
        :param iterable: iterable objects or int
        :return: container instance.
        """
        self.blueprint = blueprint
        self.iterable = iterize(iterable)

        self.set_shape(
            lambda v: v
        )
        self.set_stringify(
            lambda v: json.dumps(v, indent=json_indent, cls=JSONEncoder)
        )

    def repeat(self, iterable):
        """
        :param iterable:
        :return:
        """
        self.iterable = iterize(iterable)
        return self

    def count(self):
        return len(self)

    def stringify(self):
        """
        stringify shaped self(container) object.
        :return:
        """
        return self._stringify(self.shape())

    def set_stringify(self, stringify):
        """
        set how to convert to a string.(default: json.dumps(self, indent=1))
        :param stringify:
        :return:
        """
        self._stringify = stringify

    def shape(self):
        """
        shape self(container) object.
        :return: shaped container object
        """
        return self._shape(deepcopy(self))

    def set_shape(self, shape):
        """
        set how to shape self(container) object.(default: no operation)
        :return:
        """
        self._shape = shape

    def output(self, path, rewrite=False):
        """
        self(container) object write to file.
        :param path: output file path.
        :param rewrite: rewrite if file path is already exists.(default: False)
        :return: None
        """
        string = self.stringify()
        if os.path.exists(path):
            if not rewrite:
                raise OutputFileAlreadyExists()
        with open(path, 'wb') as f:
            f.write(string)
