# #!/usr/bin/env python

# wordseek 1.0.6 by David Hacker

import argparse
import sys
import time
import imp
import traceback
import random
import os
import copy

from pprint import pprint
from string import ascii_uppercase
from string import ascii_lowercase

letters = list(ascii_uppercase + ascii_lowercase)

def get_time(units='ms'):
    """
    Units ...
      - ms = milliseconds
      - s = seconds
    """
    if units == 's':
        if sys.platform.startswith('win'):
            return time.clock()
        else:
            return time.time()
    elif units == 'ms':
        return get_time(units='s') * 1000
 
def run():
    parser = argparse.ArgumentParser(
                                   description='''wordseek execution script
                                   
                                   WordSeek is based off of the classic Word Search game
                                   except that it uses autonomous Python 2.7 robots.
                                   The robot with the total fastest time wins the game.
                                   ''',
                                   formatter_class = argparse.RawTextHelpFormatter
                                   )
    parser.add_argument("players", nargs="+",
                        help="Player file names")
    parser.add_argument("-s", "--seed",
                        default=random.randint(0, sys.maxint),
                        help="Determines output of word grids")
    parser.add_argument("-i", "--immediate",
                        action="store_true", default=False,
                        help="Run the game immediately after startup")
    parser.add_argument("-q", "--quiet",
                        action="store_true", default=False,
                        help="Suppress game output")
    args = parser.parse_args()
        
    print "Match seed: {0}".format(args.seed)
    random.seed(args.seed)
    
    players = []
    
    def add_player(filename):
        fid = open(filename, 'r')
        new_playername = os.path.basename(fid.name).replace('.py', '')
        new_player_code = fid.read()
        fid.close()
                
        new_player = Player(new_playername, new_player_code)
        if not new_player.failed:
            if new_player not in players:
                players.append(new_player)
            else:
                raise Exception("{0} is already a contestant!".format(new_playername))
            
    for player_file in args.players:
        add_player(player_file)
                    
    print ""
    print "{0} robots:".format(len(players)).center(75)
    for player in players:
        print "{0} - {1}".format(player.name, player.author).center(75)
    print ""
    
    if len(players) < 2:
        print "In order to run, a game must have greater than 1 robots."
        return
    
    game = Game(players, args.quiet)
    if not args.immediate:
        print ""
        print "Starting in 2 seconds ...".center(75)
        time.sleep(2)
    game.run_all()
        
    print "".center(80, '=')
    
    print "Stats:".center(75)
    for player in game._players:
        name = player.name
        total_time = player.total_time
        mistakes = player.mistakes
        print "{0} took {1} milliseconds with {2} mistakes".format(name, total_time, mistakes).center(75)
        
    print ""
    
    def print_winners(place, bots):
        if len(bots) == 1:
            print (place+": {0} ({1})").format(bots[0].name, bots[0].author).center(75)
        elif len(bots) > 1:
            names = [winner.name for winner in bots]
            print (place+": {0}").format(names).center(75)
        
    remaining_players = game._players
    winners = game.get_winners(remaining_players)
    print_winners('1st', winners)
    
    remaining_players = [p for p in game._players if p not in winners]
    second_place = game.get_winners(remaining_players)
    print_winners('2nd', second_place)
        
    remaining_players = [p for p in game._players if p not in (winners + second_place)]
    if remaining_players:
        third_place = game.get_winners(remaining_players)
        print_winners('3rd', third_place)
        
class Player:
    name = ''
    author = ''
    _bot = None
    
    total_time = 0
    
    mistakes = 0
    _penalty = 0
    
    failed = False
    
    def __init__(self, name, code):
        self.name = name
        try:
            _module = imp.new_module('usercode%d' % id(self))
            exec code in _module.__dict__
            
            self._bot = _module.Robot()
            try:
                self.author = self._bot.author
            except AttributeError:
                self.author = 'Anonymous'
        except:
            print "Failed to load robot: {0}.py".format(name)
            traceback.print_exc()
            self.failed = True
        
    def penalize(self):
        self.mistakes += 1
        self._penalty += 5
        if self._penalty > 10:
            self._penalty = 10
            
    def __eq__(self, other):
        return self.name == other.name

class Game:

    _players = None
    _winner = None
    _quiet = False
    
    fresh_grids = []
    grids = {}
    word_pool = []
    turn = 0
    
    # Constants
    min_word_length = 4
    max_word_length = 9
    grid_length = 9
    max_turns = 50
    grid_num = 10
    word_num = 5
    
    def __init__(self, players, quiet):
        self._quiet = quiet
        if quiet:
            self.oldstdout = sys.stdout
            sys.stdout = NullWriter()
        
        def random_string():
            rs = []
            rsl = random.randint(self.min_word_length, self.max_word_length)
            while len(rs) < rsl:
                rs.append(random.choice(letters))
            rss = ''.join(rs)
            if rss in self.word_pool:
                return random_string()
            return rss
        
        self._players = players
        
        while len(self.word_pool) < self.max_turns:
            self.word_pool.append(random_string())
        
        generator = GridGenerator(self.grid_length)
        for i in xrange(0, self.max_turns):
            use_word = self.word_pool.pop()
            new_grid = generator.generate(use_word)
            self.fresh_grids.append(new_grid)
        random.shuffle(self.fresh_grids)
            
    def _run_next(self):
        def output(msg):
            if isinstance(msg, str):
                print msg
            else:
                pprint(msg)
        
        self.turn += 1
        output(" Turn {0} ".format(self.turn).center(75, '-'))
        
        # Pop random fresh grid, and use it
        grid_wrapper = self.fresh_grids.pop()
        grid = grid_wrapper['grid']
        word = grid_wrapper['word']
        answer = grid_wrapper['answer']
        
        output([['-']+['{0}'.format(x) for x in xrange(0, self.grid_length)]])
        formatted_grid = copy.deepcopy(grid)
        for row in formatted_grid:
            row.insert(0, '{0}'.format(formatted_grid.index(row)))
            
        output(formatted_grid)
        output("Word: '{0}'".format(word))
        output("Answer: {0}\n".format(answer))
        
        for player in self._players:
            nom = player.name
            time_start = get_time()
            bot = player._bot
            try:
                player_answer = bot.find_word(grid, word)
            except:
                traceback.print_exc()
                player_answer = []
                
            took = get_time() - time_start
            player.total_time += took
            
            output("{0} [{1:.3g}ms] returned:".format(nom, took))
            output(player_answer)
            if answer == player_answer:
                output("Correct!")
            else:
                player.penalize()
                output("Incorrect. Penalized {0} seconds.".format(player._penalty))
                player.total_time += (player._penalty * 1000) # Convert to milliseconds
            output("")
        
        # Put spent grid into used heap; these will be displayed in a GUI later
        self.grids[self.turn] = grid_wrapper
            
    def run_all(self):
        while self.turn < self.max_turns:
            self._run_next()
        if self._quiet:
            sys.stdout = self.oldstdout
        
    def get_winners(self, players):
        times = [p.total_time for p in players]
        fastest_time = min(times)
        return [p for p in self._players if p.total_time == fastest_time]
    
class GridGenerator:
    side_length = 0
    
    def __init__(self, side_length):
        self.side_length = side_length
    
    def generate(self, word):
        chars = list(word)
        length = self.side_length
        grid = []
        for i in xrange(0, length):
            row = []
            for j in xrange(0, length):
                row.append(random.choice(letters))
            grid.append(row)
            
        def encase_word():
            starting_point = (random.randint(0, length-1), random.randint(0, length-1))
            operation = (random.randint(-1, 1), random.randint(-1, 1))
            if operation[0] == operation[1] == 0:
                return encase_word()
            
            willwork = 0 <= (starting_point[0] + operation[0] * (len(chars)-1)) < length and 0 <= (starting_point[1] + operation[1] * (len(chars)-1)) < length
            if not willwork:
                return encase_word()
            
            answer = []
            for i, curr_char in enumerate(chars):
                current_point = ((starting_point[0] + (operation[0] * i)), (starting_point[1] + (operation[1] * i)))
                xvar = current_point[0]
                yvar = current_point[1]
                new_row = []
                for index, dud_char in enumerate(grid[yvar]):
                    if index == xvar:
                        new_row.append(curr_char)
                    else:
                        new_row.append(dud_char)
                grid[yvar] = new_row
                answer.append(current_point)
                
            return answer
        
        word_location = encase_word()
        return {'word':word, 'grid':grid, "answer":word_location}

class NullWriter:
    
    def write(self, arg):
        pass
    
if __name__ == '__main__':
    run()