###############################################################
# Tools figure generation
#
# Authors:
#  NOVELTIS: Cedric Bacour / Ivan Price
###############################################################



#from __init__ import *

import numpy as np
import sys
import tempfile
import os
import random
import math
from PIL import Image
import calendar

# matplotlib doesn't work without a writable home directory, as maybe the case when
# run by the web server for the web site. If this is so then we assign a temp dir as 
# it's home directory to keep it happy
home_dir = os.getenv('USERPROFILE') or os.getenv('HOME')
if not os.path.exists(home_dir):
    os.environ['MPLCONFIGDIR'] = tempfile.gettempdir()

import matplotlib
# need to set 'Agg' as the backend when running by web, this is set in web_run_calc.py
if os.getenv('ADAM_WEB_REQUEST', '') == 'true':
    matplotlib.use("Agg")
from mpl_toolkits.mplot3d import Axes3D
import matplotlib.pyplot as plt
from matplotlib.patches import Circle as circle


import adam





# ===================================================================
# Display map
#
# Inputs:
#  - map [nlon,nlat] to be displayed
#  - latitudes (vector of nlat)
#  - longitudes (vector of nlon)
#  - enlarge (optional): if True enlarge by offset in map in lat,lon
# - fig (optional): figure number
#  - save (optional): save the figure in a file
#  - save_format (optional): output format of the saved file
#  - path (optional): output path of the saved file
#  - filename (optional): output file name
#
#
# - TODO :
# different colormaps for land and ocean...
# gestion of the missval pixels
#
# ===================================================================
def map(data_map, lats, lons, enlarge = False, fig = None,
        save = False, save_format = 'png', path = None, filename = None,
        missval = None, fill_missval = 0):

    xint = lons[1] - lons[0]
    yint = lats[1] - lats[0]


    offset = 0
    if enlarge == True: offset = 2 #enlarge the map on lat,lon by "offset" degrees
    if fig == None: fig = 1


    if save == True:
        if path == None or filename == None:
            sys.exit('\n*2 *** ERROR in show map: save = True but the output path/file name is not defined')


    data_map = np.transpose(data_map.copy())

    if missval != None:
        mapm = np.ma.masked_equal(data_map, missval)
        data_map = np.ma.filled(mapm,fill_missval)


    fig = plt.figure(fig,frameon=False)
    plt.clf()

    range_lon = np.minimum.reduce(lons), np.maximum.reduce(lons)
    range_lat = np.minimum.reduce(lats), np.maximum.reduce(lats)
    lon0 = (range_lon[1]-range_lon[0])/2+range_lon[0]
    lat0 = (range_lat[1]-range_lat[0])/2+range_lat[0]


    (minx,miny,maxx,maxy) = (range_lon[0]-offset, range_lat[0]-offset, range_lon[1]+offset, range_lat[1]+offset)

    print 'make map with %s, %s, %s, %s' % (minx,miny,maxx,maxy)



#    map = Basemap(projection='cyl',lon_0=lon0,lat_0=lat0,
#                  llcrnrlat=range_lat[0]-offset,urcrnrlat=range_lat[1]+offset,
#                  llcrnrlon=range_lon[0]-offset,urcrnrlon=range_lon[1]+offset,
#                  rsphere=6371200.,resolution='l',area_thresh=100)

    world_file = '%s\n' % str( (maxx-minx)/360)
    world_file += '0.0\n'
    world_file += '0.0\n'
    world_file += '%s\n' % str( -(maxy-miny)/359)
    world_file += '%s\n' % str(range_lon[0]-offset)
    world_file += '%s\n' % str(range_lat[1]+offset)

    coords_file = open(filename+ '.pgw', 'wt')
    coords_file.write(world_file)
    coords_file.close()

    #map.drawcoastlines()
    #map.drawcountries()
    #map.drawmapboundary()

    # draw lat/lon grid lines every 30 degrees.
    #map.drawmeridians(np.arange(0, 360, 30))
    #map.drawparallels(np.arange(-90, 90, 30))

    #x, y = map(lons, lats)
    #map.pcolor(x, y, data_map)
    array = np.ma.masked_equal(data_map, -9999)
    #plt.imshow(np.flipud(array), vmin=0, vmax=0.7)
    plt.imshow(array, vmin=0, vmax=0.7, interpolation='nearest')

    plt.xticks([])
    plt.yticks([])
    plt.subplots_adjust(left=0, right=1, top=1, bottom=0, hspace=0, wspace=0)

    if save == True:
        plt.savefig(filename+'.'+save_format, dpi=72, facecolor='None', edgecolor='None',
                orientation='portrait', papertype=None, format= save_format,
                transparent=True, bbox_inches='tight')
    else:
        plt.show()

    plt.close(fig)
# END map
# ===================================================================





# ===================================================================
# Display map using only PIL for margin-free, fast maps.
#
#
# TODO:
# different colormaps for land and ocean...
#
# ===================================================================
def map_pil(adam_job, spectral_domain_index=0, width=250, output_dir=None):

    # get the data variable
    data = adam_job.data['reflectance_averaged']

    
    data = np.ma.masked_equal(data, -9999)
    # if this is 3d, it means we were passed an array with different spectra, we should
    # map only the one we are asked to map
    if len(data.shape) > 2:
        data = data[:,:, spectral_domain_index]

    # if we are given a specific output dir we should use it, otherwise use the one for this job
    if output_dir == None:
        filename = '%s/output-%s' % (adam_job.job_output_dir, random.random())
    else:
        filename = '%s/output-%s' % (output_dir, random.random())


    extent = adam_job.extent
    minx, miny, maxx, maxy = extent['minx'], extent['miny'], extent['maxx'], extent['maxy']

    # determing the dimensions of the map in pixels according to the data dimensions in lat/longs
    width = float(width)
    ratio = abs  ( (maxx - minx ) / float(maxy - miny) )
    height = math.ceil(width/ratio)


    # write the world file to georeference the map
    world_file = '%s\n' % str( (maxx-minx)/width)
    world_file += '0.0\n'
    world_file += '0.0\n'
    world_file += '%s\n' % str( -(maxy-miny)/height)
    world_file += '%s\n' % minx
    world_file += '%s\n' % maxy

    world_file_extension = '.pgw'
    coords_file = open(filename + world_file_extension, 'wt')
    coords_file.write(world_file)
    coords_file.close()

    # mask water
    #data = np.ma.masked_equal(data, -1)

    # load the default min and max colour ranges obtain by
    # inspecting the data for the whole globe for each spectral range

    range_dict = {
      0: {'min': 0, 'max': 1.03365},
      1: {'min': 0, 'max': 1.03555 },
      2: {'min': 0, 'max': 1.17559},
      3: {'min': 0, 'max': 1.41174},
      4: {'min': 0, 'max': 1.26724}
    }
    



    try:
        min_value = range_dict[spectral_domain_index]['min']
        max_value = range_dict[spectral_domain_index]['max']
    except:
        min_value = 0
        max_value = 0.5



    # build our color map lookerupper using matplotlib machinery
    classes = 255
    mult = classes / (max_value - min_value)
    cmap = matplotlib.cm.get_cmap('jet', classes)
    cmap.set_over('r', 1.0)


#
#    def get_colour_hex(value):
#        ''' returns the hex string for a particular data value from an existing colour map '''
#        corrected_value = (value - min_value) * mult
#        colour = matplotlib.colors.rgb2hex(cmap(int(corrected_value)))
#        return colour


    def get_rgb_tuple(value):
        ''' returns a r,g,b,a tuple from a particular data value from an existing colour map '''
        if value is np.ma.masked:
            return (0,0,0,0)

        # convert the value into a positive number between 0 and number of classes (255)
        corrected_value = (value - min_value) * mult
        # obtain the colour hex code for that value
        hex_code = matplotlib.colors.rgb2hex(cmap(int(corrected_value)))
        # turn the hexcode into rgb values where the values are between 0 and 1
        r,g,b = matplotlib.colors.hex2color(hex_code)

        # scale the numbers to between 0 and 255
        return (r*255, g*255, b*255, 255)


    # initialise the new image (map)
    pil_im = Image.new('RGBA', (width, height))
    # get the image access object
    pil_acc = pil_im.load()

    # i don't like this loop but for now its like this
    # loop across the x
    for i in range(data.shape[0]):
        # loop across the y
        for j in range(data.shape[1]):
            # for each pixel get the rgba value

            pil_acc[i, j] = get_rgb_tuple(data[i,j])
            #print 'got %s from %s' % (pil_acc[i, j], data[i,j])


    # save the PIL image to disk
    pil_im.save(filename + '.png', "PNG")


# END map_pil
# ===================================================================



# ===================================================================
# output a tif file representing the raw reflectance values
# used to make the colour composite maps
#
#
# ===================================================================
def map_raw_values(adam_job, spectral_domain_index=0, width=250, output_dir=None):

    import osgeo.gdal as gdal
    import osgeo.osr as osr

    # get the data variable
    data = adam_job.data['reflectance_averaged']
    
    #data = adam_job.data['reflectance_MODIS']

    # if this is 3d, it means we were passed an array with different spectra, we should
    # map only the one we are asked to map
    if len(data.shape) > 2:
        data = data[:,:, spectral_domain_index]

    # if we are given a specific output dir we should use it, otherwise use the one for this job
    if output_dir == None:
        filename = '%s/output-%s' % (adam_job.job_output_dir, random.random())
    else:
        filename = '%s/output-%s' % (output_dir, random.random())


    extent = adam_job.extent
    minx, miny, maxx, maxy = extent['minx'], extent['miny'], extent['maxx'], extent['maxy']

    # determing the dimensions of the map in pixels according to the data dimensions in lat/longs
    width = float(width)
    ratio = abs  ( (maxx - minx ) / float(maxy - miny) )
    height = width/ratio


    # write the world file to georeference the map
    world_file = '%s\n' % str( (maxx-minx)/width)
    world_file += '0.0\n'
    world_file += '0.0\n'
    world_file += '%s\n' % str( -(maxy-miny)/height)
    world_file += '%s\n' % minx
    world_file += '%s\n' % maxy

    world_file_extension = '.tfw'
    coords_file = open(filename + world_file_extension, 'wt')
    coords_file.write(world_file)
    coords_file.close()

    # mask water
    #data = np.ma.masked_equal(data, -1)

    # dodgy scaling hacks
    #data = np.ma.filled(data, 0.)

# reflectance_averaged
    range_dict = {
      0: {'min': 0, 'max': 1.03365},
      1: {'min': 0, 'max': 1.03555 },
      2: {'min': 0, 'max': 0.842136},
      3: {'min': 0, 'max': 0.546611},
      4: {'min': 0, 'max': 0.441895}
    }
    # modis cedric, too light
#    range_dict = {
#      0: {'min': 0, 'max': 0.215},
#      1: {'min': 0, 'max': 0.266 },
#      2: {'min': 0, 'max': 0.233},
#      3: {'min': 0, 'max': 0.53 },
#      4: {'min': 0, 'max': 0.56},
#      5: {'min': 0, 'max': 0.69},
#      6: {'min': 0, 'max': 0.66}
#    }
# modis mine
#    range_dict = {
#      0: {'min': 0, 'max': 0.5  },
#      1: {'min': 0, 'max': 0.5  },
#      2: {'min': 0, 'max': 0.5  },
#      3: {'min': 0, 'max': 0.62 },
#      4: {'min': 0, 'max': 0.7 },
#      5: {'min': 0, 'max': 0.7 },
#      6: {'min': 0, 'max': 0.7 }
#    }
    
    
    this_range = range_dict[spectral_domain_index]['max'] - range_dict[spectral_domain_index]['min']
    #stretched = data.clip(range_dict[spectral_domain_index]['min'], range_dict[spectral_domain_index]['max'])
    #stretched = data * 400/range_dict[spectral_domain_index]['max']
    stretched = data * 255/range_dict[spectral_domain_index]['max']
    stretched = stretched.clip(0, 255)
    #stretched = data * (255/range_dict[spectral_domain_index]['max'])
    #stretched = data

    #print stretched.mask
    #stretched = np.ma.masked_less_equal(stretched, 0)
    #stretched = stretched.filled(255)

    #stretched = stretched.astype(np.uint)



    output_driver = gdal.GetDriverByName( "GTiff" )
    output_dataset = output_driver.Create( filename + '.tif', int(width), int(height), 1, gdal.GDT_Float32 )
    #output_dataset = output_driver.Create( filename + '.tif', int(width), int(height), 1, gdal.GDT_Int16 )
    
    #output_dataset = output_driver.Create( filename + '.tif', int(width), int(height), 1, gdal.GDT_Byte )

    #lon = adam_job.data['longitude']
    #lat = adam_job.data['latitude']

    #x_int = lon[1] - lon[0]
    #output_dataset.SetGeoTransform( [ lon.min(), (lon.max() + x_int- lon.min()) / width, 0, lat.max(), 0, -(lat.max()- lat.min()) / height ] )
    #srs = osr.SpatialReference()
    #srs.SetWellKnownGeogCS( 'EPSG:4326' )
    #output_dataset.SetProjection( srs.ExportToWkt() )

    output_dataset.GetRasterBand(1).SetNoDataValue( -9999 )
    # write the data to the first band
    output_dataset.GetRasterBand(1).WriteArray( stretched.T )
    # force save
    output_dataset = None

# END map_raw_values
# ===================================================================








# ===================================================================
# Display Reflectance spectra
#
# Input
#  - data dictionnary containing mean, max, min, and std
# ===================================================================
def main_ref_spectra(job, case = None, indices=None, title = None, graph_error_indices=None):

    cfg = job.cfg
    # -
    # - Processing for individual pixels
    # -
    if case == 'pixels':

        if len(indices) < 1:
            # if we were given an empty array, or the length of the data is for some reason zero
            # we just return
            return
            
        # get the data for the pixel graph, store its original shape
        original_shape = job.data['BRDF'].shape
        
        # if the BRDF variable is more than 2d (i.e. has lat,long and spectra),
        # we need to flatten the data to a 1d array of reflectances per wavelength
        if len(job.data['BRDF'].shape) > 2:
            job.data['BRDF'] = job.data['BRDF'].reshape(job.data['BRDF'].shape[0] * job.data['BRDF'].shape[1], len(cfg.lmbd))


        # if we weren't given the indices to graph, assume all
        if indices == None:
            indices = range(len(job.data['BRDF']))

        # initialise the graphable data
        ref = np.zeros((len(cfg.lmbd), len(indices)))

        legend = []
        # we need a set of coordinate pairs from the lat/longs,
        # so build a grid using meshgrid
        x,y = np.meshgrid(job.data['longitude'], job.data['latitude'])

        # cycle through each pixel
        for i in range(len(indices)):
            # load the graphable data
            ref[:,i] = np.ma.array(job.data['BRDF'][indices[i]])
            # get the lat long using the same index in our meshgrid
            lat = y.ravel()[indices[i]]
            lon = x.ravel()[indices[i]]

            txt_lat = u"%4.2f\u00b0" % (lat)
            txt_lon = u"%4.2f\u00b0" % (lon)

            # build legend text
            legend.extend(['pix %i (%s,%s)'%(indices[i],txt_lat,txt_lon) ])
        

        error_array = job.data['reflectance_err_land']
        if len(error_array.shape) > 2:
            error_array = error_array.reshape(error_array.shape[0] * error_array.shape[1], error_array.shape[2] )
        error_array = error_array[indices,:]

        draw_ref_spectra(cfg.lmbd, ref, job, title=title, legend=legend, error_array=error_array)

        job.data['BRDF'] = job.data['BRDF'].reshape(original_shape)
        job.data['reflectance_err_land'] = job.data['reflectance_err_land'].reshape(original_shape)

        return



    # -
    # - Processing statistics
    # -
#    if type(data) == type({}):
    elif case == 'stats':
        data = job.data['stats']
        surfaces = data.keys()

        for surface in surfaces:
            if data[surface] != None:

                fig = plt.figure(figsize=(9,6))

                plt.errorbar(cfg.lmbd, data[surface]['mean'], yerr = data[surface]['std'],color = '0.8')
                plt.plot(cfg.lmbd, data[surface]['mean'],'r', label = 'mean')
                plt.plot(cfg.lmbd, data[surface]['min'],'g', label = 'min')
                plt.plot(cfg.lmbd, data[surface]['max'],'b', label ='max')
                plt.xlim(400,4000)
                plt.xlabel('wavelength (nm)')
                plt.ylabel('reflectance')
                plt.legend(loc='upper right')

                extent = '%s,%s %s,%s' % (job.extent['minx'], job.extent['miny'], job.extent['maxx'], job.extent['maxy'])
                plt.title('Spectrum Graph - %s pixels (%s), Extent: %s' % (surface, data[surface]['npts'], extent))

                # add the text explaination
                plt.subplots_adjust(left=0.1, right=0.8, top=0.95, bottom=0.1)
                annotation_x = 535
                annotation_y = 390
                plt.annotate('ADAM', xy=(annotation_x, annotation_y), fontsize=22, xycoords='figure pixels')
                annotation_y -= 20
                plt.annotate('Reflectance Database', xy=(annotation_x, annotation_y), fontsize=10, xycoords='figure pixels')
                annotation_y -= 150


                annotation = 'Month: %s\n' % job.get_month_name()
                annotation += 'Lats:\n   %.03f:%.03f\n' % (job.data['latitude'].min(), job.data['latitude'].max())
                annotation += 'Lons:\n   %.03f:%.03f\n' % (job.data['longitude'].min(), job.data['longitude'].max())
                annotation += 'SZA: %s\n' % job.sza
                annotation += 'VZA: %s\n' % job.vza
                annotation += 'phi: %s\n' % job.phi

                plt.annotate(annotation, xy=(annotation_x, annotation_y), fontsize=12, xycoords='figure pixels')

                filename = job.build_filename('spectrum_%s' % surface, 'png')

                plt.savefig(filename, dpi=72, facecolor='None', edgecolor='None',
                            orientation='portrait', papertype=None, format= 'png', bbox_inches=None)

                plt.close(fig)

                job.add_output('brdf_spectrum',
                               filename,
                               case='stats',
                               title='Spectrum Graph - %s' % surface)



# END ref_spectra
# ===================================================================


# ===================================================================
# Display Reflectance spectra
#
# Input
#  - data dictionnary containing mean, max, min, and std
# ===================================================================
def draw_ref_spectra(x, y, job, title=None, legend=None, yerr=None, error_array=None):

    fig = plt.figure(figsize=(9,6))
    if title==None:
        title = ''
    plt.plot(x,y)

    if error_array is not None:
        # error array was pixel, spect: we need spect, pixel
        error_array = np.swapaxes(error_array, 0, 1)
        minimum = y - error_array
        maximum = y + error_array
        plt.plot(x, minimum,'grey')
        plt.plot(x, maximum,'grey')
        
        
    plt.xlabel('wavelength (nm)')
    plt.ylabel('reflectance')
    plt.legend(legend, loc='upper right')
    plt.xlim(400,4000)
    plt.title(title)

    # add the text explaination
    plt.subplots_adjust(left=0.1, right=0.8, top=0.95, bottom=0.1)
    annotation_x = 535
    annotation_y = 390
    plt.annotate('ADAM', xy=(annotation_x, annotation_y), fontsize=22, xycoords='figure pixels')
    annotation_y -= 20
    plt.annotate('Reflectance Database', xy=(annotation_x, annotation_y), fontsize=10, xycoords='figure pixels')

    annotation_y -= 150

    annotation = 'Month: %s\n' % job.get_month_name()
    annotation += 'Lats:\n   %.03f:%.03f\n' % (job.data['latitude'].min(), job.data['latitude'].max())
    annotation += 'Lons:\n   %.03f:%.03f\n' % (job.data['longitude'].min(), job.data['longitude'].max())
    annotation += 'SZA: %s\n' % job.sza
    annotation += 'VZA: %s\n' % job.vza
    annotation += 'phi: %s\n' % job.phi

    plt.annotate(annotation, xy=(annotation_x, annotation_y), fontsize=12, xycoords='figure pixels')



    filename = job.build_filename('spectrum_pixels', 'png')

    plt.savefig(filename, dpi=72, facecolor='None', edgecolor='None',
                orientation='portrait', papertype=None, format= 'png', bbox_inches=None)


    plt.close(fig)

    job.add_output('brdf_spectrum', filename, case='pixels', title=title)

# END draw_ref_spectra
# ===================================================================

def main_brdf_time(job, case=None):

    surfaces = ['land', 'ocean', 'snow']

    if case == 'pixels':

        data = job.time_series
        data = np.array(data)
        print data.shape
        # reshape data from (12, x_pixels, y_pixels, 1) -> (12, total)
        data = data.reshape((data.shape[0], data.shape[1] * data.shape[2] ))
        

        for surface in surfaces:
            title = 'BRDF Time Series - Pixels - %s' % surface
            
            # we need a list of all the unique pixel indexes for this surface type that occur during the year
            # in order to follow each relevant pixels throughout the year, regardless of if it disappears for 
            # certain months
            possible_indexes = []
            for month_number in range(12):
                monthly_pixel_index = job.time_series_idxs[month_number][surface]
                for idx in monthly_pixel_index:
                    if not idx in possible_indexes:
                        possible_indexes.append(idx)
            possible_indexes = np.array(possible_indexes)
            
            job.set_status('possible pixel indexes for surface %s (across all months) are: %s' % (surface, [possible_indexes]))
            
            # if there is not a single pixel for any month we will skip this surface
            if len(possible_indexes) < 1:
                continue
            
            
            #graph_data is the array to be graphed, it has shape 12 months x the max number of pixels for this surface type
            graph_data = np.empty((12, len(possible_indexes)))
            graph_data[:] = np.NaN
            
            for month_index in range(12):
                indices = job.time_series_idxs[month_index][surface]
                job.set_status('for month %s adding data for indexes: %s' % (month_index, [indices]))
                # for each pixel for this month/surface combo, fill the appropriate slot in the graph_data array
                for idx in indices:
                    graph_data[month_index][possible_indexes == idx] = data[month_index][idx]


                

            # init the x axis values 0 -> 11
            x = np.arange(12)
            y = graph_data

            # init the vars that contain the lat lon coords for the legend text
            lons, lats = np.meshgrid(job.data['longitude'], job.data['latitude'])
            legend = []
            for i in range(len(possible_indexes)):
                lat = lats.ravel()[possible_indexes[i]]
                lon = lons.ravel()[possible_indexes[i]]
                txt_lat = u"%4.2f\u00b0" % (lat)
                txt_lon = u"%4.2f\u00b0" % (lon)
                # build legend text
                legend.extend(['pix %i (%s,%s)'%(possible_indexes[i],txt_lat,txt_lon) ])

            # init the graph
            fig = plt.figure(figsize=(9,6))
            plt.clf()

            # plot the data
            plt.plot(x, y, 'o-')

            plt.xlabel('')
            plt.ylabel('Reflectance')
            # legend
            leg = plt.legend(legend, loc='upper right')
            ltext  = leg.get_texts()
            plt.setp(ltext, fontsize='small')

            plt.xlim(x.min(),x.max())
            plt.xticks( np.arange(12), calendar.month_abbr[1:13])
            plt.title(title)

            # add the text explaination
            plt.subplots_adjust(left=0.1, right=0.8, top=0.95, bottom=0.1)
            annotation_x = 535
            annotation_y = 390
            plt.annotate('ADAM', xy=(annotation_x, annotation_y), fontsize=22, xycoords='figure pixels')
            annotation_y -= 20
            plt.annotate('Reflectance Database', xy=(annotation_x, annotation_y), fontsize=10, xycoords='figure pixels')
            annotation_y -= 150

            annotation = 'Month: %s\n' % job.get_month_name()
            annotation += 'Lats:\n   %.03f:%.03f\n' % (job.data['latitude'].min(), job.data['latitude'].max())
            annotation += 'Lons:\n   %.03f:%.03f\n' % (job.data['longitude'].min(), job.data['longitude'].max())
            annotation += 'SZA: %s\n' % job.sza
            annotation += 'VZA: %s\n' % job.vza
            annotation += 'Wavelength:\n   Min: %s nm\n   Max: %s nm' % (int(job.lmbd.min()), int(job.lmbd.max()))

            plt.annotate(annotation, xy=(annotation_x, annotation_y), fontsize=12, xycoords='figure pixels')



            # save the graph
            filename = job.build_filename('BRDFtime_pixels_%s' % surface, 'png')
            plt.savefig(filename, dpi=72, facecolor='None', edgecolor='None',
                        orientation='portrait', papertype=None, format= 'png', bbox_inches=None)
            plt.close(fig)


            job.add_output('brdf_time_series', filename, case='pixels', title=title)

    else:
        # this is a case='stats' graph

        for surface in surfaces:
            title = 'Time Series - Stats - %s' % surface
            data = job.data['stats']

            # init x axis data
            x = np.arange(12)

            # if the maximum number of pixels for this surface over all the months is zero we skip this surface
            if data[surface]['npts'] == 0:
                continue

            # init the graph
            fig = plt.figure(figsize=(9,6))
            plt.clf()
            plt.errorbar(x, data[surface]['mean'], yerr = data[surface]['std'], color = '0.8')
            plt.plot(x, data[surface]['mean'],'ro-', label = 'mean')
            plt.plot(x, data[surface]['min'], 'go-', label = 'min')
            plt.plot(x, data[surface]['max'], 'bo-', label = 'max')
            plt.xlabel('')
            plt.xticks( np.arange(12), calendar.month_abbr[1:13])
            plt.ylabel('reflectance')
            # legend
            loc='upper left'
            if job.sza < 0:
                loc='upper right'
            leg = plt.legend(loc=loc)
            ltext = leg.get_texts()
            plt.setp(ltext, fontsize='small')

            plt.xlim(np.amin(x),np.max(x))

            extent = '%s,%s %s,%s' % (job.extent['minx'], job.extent['miny'], job.extent['maxx'], job.extent['maxy'])
            plt.title('Time Series - %s pixels (max %s), Extent: %s' % (surface, data[surface]['npts'], extent))

            # add the text explaination
            plt.subplots_adjust(left=0.1, right=0.8, top=0.95, bottom=0.1)
            annotation_x = 535
            annotation_y = 390
            plt.annotate('ADAM', xy=(annotation_x, annotation_y), fontsize=22, xycoords='figure pixels')
            annotation_y -= 20
            plt.annotate('Reflectance Database', xy=(annotation_x, annotation_y), fontsize=10, xycoords='figure pixels')
            annotation_y -= 150

            annotation = 'Month: %s\n' % job.get_month_name()
            annotation += 'Lats:\n   %.03f:%.03f\n' % (job.data['latitude'].min(), job.data['latitude'].max())
            annotation += 'Lons:\n   %.03f:%.03f\n' % (job.data['longitude'].min(), job.data['longitude'].max())
            annotation += 'SZA: %s\n' % job.sza
            annotation += 'VZA: %s\n' % job.vza
            annotation += 'Wavelength:\n   Min: %s nm\n   Max: %s nm' % (int(job.lmbd.min()), int(job.lmbd.max()))

            plt.annotate(annotation, xy=(annotation_x, annotation_y), fontsize=12, xycoords='figure pixels')


            # save the graph
            filename = job.build_filename('BRDFplane_%s' % surface, 'png')
            plt.savefig(filename, dpi=72, facecolor='None', edgecolor='None',
                        orientation='portrait', papertype=None, format= 'png', bbox_inches=None)
            plt.close(fig)


            job.add_output('brdf_time_series',
                           filename,
                           case='stats',
                           title='BRDF Time Series - %s' % surface)




# ===================================================================
# Display BRDF in the principal plane
#
# Inputs
#  - data to be displayed (dictionnary)
#  - waveband domain
#  - viewing zenith angles
#
# ===================================================================
def main_brdf_principal_plane(job, graph_error=False, case=None, indices=None, title=None, spectral_domain_index=0, phi_index=0):

    vza = job.vza

    # -
    # - Processing for individual pixels
    # -
    if case == 'pixels':

        if len(indices) < 1:
            # if we were given an empty array, or the length of the data is for some reason zero
            # we just return
            return
        
        
        data = job.data['BRDF'][:,:, spectral_domain_index,:, phi_index]
        #data = job.data['BRDF']
        data_err = job.data['BRDF_err_land'][:,:, spectral_domain_index,:, phi_index]
        #data.dimensions = x,y,spectral_domain_index,vza: see job.spectral_domains
        data = data.reshape(data.shape[0] * data.shape[1], data.shape[-1])
        #data = data.reshape(data.shape[0] * data.shape[1], data.shape[2], data.shape[3])
        data_err = data_err.reshape(data.shape)

        # if we weren't given the indices to graph, assume all
        if indices == None:
            indices = range(len(data))

        x,y = np.meshgrid(job.data['longitude'], job.data['latitude'])


        # initialise the graphable data
        brdf = np.zeros((vza.shape[0],len(indices)))
        brdf_err = np.zeros((vza.shape[0],len(indices)))
        
        legend = []
        # cycle through each pixel
        for i in range(len(indices)):
            # get the lat long using the same index in our meshgrid
            lat = y.ravel()[indices[i]]
            lon = x.ravel()[indices[i]]
            txt_lat = u"%4.2f\u00b0" % (lat)
            txt_lon = u"%4.2f\u00b0" % (lon)

            # build legend text
            legend.extend(['pix %i (%s,%s)'%(indices[i],txt_lat,txt_lon) ])

            brdf[:,i] = np.ma.array(data[indices[i]])
            brdf_err[:,i] = np.ma.array(data_err[indices[i]])
            #lat = data[indices[i]]['lat_user']
            #txt_lat = u"%4.1f\u00b0" % (data[indices[i]]['lat_user'])
            #txt_lon = u"%4.1f\u00b0" % (data[indices[i]]['lon_user'])
            #legend.extend(['pix %i (%s,%s)'%(indices[i],txt_lat,txt_lon) ])

        draw_brdf_principal_plane(vza, brdf, job, error_array=brdf_err, title=title, legend=legend)

    # -
    # - Processing statistics for each surface type
    # -
    # case == 'stats'
    else:
        data = job.data['stats']
        surfaces = data.keys()

        for surface in surfaces:
            fig = plt.figure(figsize=(9,6))

            # ivan changed this
            # if this is none it means no ocean or land pixels were found
            if data[surface] != None:
                plt.errorbar(vza, data[surface]['mean'], yerr = data[surface]['std'],color = '0.8')
                plt.plot(vza, data[surface]['mean'],'r', label = 'mean')
                plt.plot(vza, data[surface]['min'],'g', label = 'min')
                plt.plot(vza, data[surface]['max'],'b', label ='max')
                plt.xlabel('view zenith angle (deg)')
                plt.ylabel('reflectance')
                loc='upper left'
                if job.sza < 0:
                    loc='upper right'
                plt.legend(loc=loc)
                plt.xlim(np.amin(vza),np.max(vza))
                #plt.title(surf+' pixels (%i)' % (data[surf]['npts']))
                extent = '%s,%s %s,%s' % (job.extent['minx'], job.extent['miny'], job.extent['maxx'], job.extent['maxy'])
                plt.title('BRDF Graph - %s pixels (%s), Extent: %s' % (surface, data[surface]['npts'], extent))


                # add the text explaination
                plt.subplots_adjust(left=0.1, right=0.8, top=0.95, bottom=0.1)
                annotation_x = 535
                annotation_y = 390
                plt.annotate('ADAM', xy=(annotation_x, annotation_y), fontsize=22, xycoords='figure pixels')
                annotation_y -= 20
                plt.annotate('Reflectance Database', xy=(annotation_x, annotation_y), fontsize=10, xycoords='figure pixels')
                annotation_y -= 150

                annotation = 'Month: %s\n' % job.get_month_name()
                annotation += 'Lats:\n   %.03f:%.03f\n' % (job.data['latitude'].min(), job.data['latitude'].max())
                annotation += 'Lons:\n   %.03f:%.03f\n' % (job.data['longitude'].min(), job.data['longitude'].max())
                annotation += 'SZA: %s\n' % job.sza

                annotation += 'Wavelength:\n   Min: %s nm\n   Max: %s nm' % (int(job.lmbd.min()), int(job.lmbd.max()))

                plt.annotate(annotation, xy=(annotation_x, annotation_y), fontsize=12, xycoords='figure pixels')



                filename = job.build_filename('BRDFplane_%s' % surface, 'png')
                plt.savefig(filename, dpi=72, facecolor='None', edgecolor='None',
                            orientation='portrait', papertype=None, format= 'png', bbox_inches=None)
                plt.close(fig)


                job.add_output('brdf_principle_plane',
                               filename,
                               case='stats',
                               title='BRDF Principal Plane - %s' % surface)

# END brdf_principal_plane
# ===================================================================



# ===================================================================
# Display BRDF in the principal plane
#
# Inputs
#  - data to be displayed (dictionnary)
#  - waveband domain
#  - viewing zenith angles
#
# ===================================================================
def draw_brdf_principal_plane(x, y, job, error_array=None, title=None, legend=None, filename=None):


    fig = plt.figure(figsize=(9,6))
    plt.clf()
    lplt = plt.plot(x,y)
    

    if error_array is not None:

        #error_array = np.swapaxes(error_array, 0, 1)

        minimum = y - error_array
        maximum = y + error_array
        #legend.extend(['min' ])
        #legend.extend(['max' ])

        plt.plot(x, minimum,'grey')
        plt.plot(x, maximum,'grey')
    
    xl  = plt.xlabel('view zenith angle (deg)')
    yl  = plt.ylabel('reflectance')
    
    # try to detect where the peak is to put the legend on the other side
    if np.argmax(y) < 25:
        leg = plt.legend(legend, loc='upper right')
    else:
        leg = plt.legend(legend, loc='upper left')
        
    xlim = plt.xlim(x.min(),x.max())
    ltitle = plt.title(title)


    plt.subplots_adjust(left=0.1, right=0.8, top=0.95, bottom=0.1)
    annotation_x = 535
    annotation_y = 390
    plt.annotate('ADAM', xy=(annotation_x, annotation_y), fontsize=22, xycoords='figure pixels')
    annotation_y -= 20
    plt.annotate('Reflectance Database', xy=(annotation_x, annotation_y), fontsize=10, xycoords='figure pixels')
    annotation_y -= 130

    annotation = 'Month: %s\n' % job.get_month_name()
    annotation += 'Lats:\n   %.05f:%.05f\n' % (job.data['latitude'].min(), job.data['latitude'].max())
    annotation += 'Lons:\n   %.05f:%.05f\n' % (job.data['longitude'].min(), job.data['longitude'].max())
    annotation += 'SZA: %s\n' % job.sza
    annotation += 'Wavelength:\n   Min: %s nm\n   Max: %s nm' % (int(job.lmbd.min()), int(job.lmbd.max()))

    plt.annotate(annotation, xy=(annotation_x, annotation_y), fontsize=10, xycoords='figure pixels')

    filename = job.build_filename('BRDFplane_pixels', 'png')

    plt.savefig(filename, dpi=72, facecolor='None', edgecolor='None',
                orientation='portrait', papertype=None, format= 'png', bbox_inches=None)
    plt.close(fig)


    job.add_output('brdf_principle_plane', filename, case='pixels', title=title)
# END brdf_principal_plane
# ===================================================================



# ===================================================================
# Display BRDF as a polar plot
#
# Inputs
#  - data to be displayed (dictionnary)
#  - waveband domain
#  - viewing zenith angles (radius = vza ; angle = phi)
#
# TODO : add a colorbar
#
# ===================================================================
def main_brdf_polar_plot(job, vza, phi, case = 'pixels', indices = None, title=None, three_d=False, waveband_index=0):


    # -
    # - Processing for individual pixels
    # -
    if case == 'pixels':

        # normally data shape is long x lat x wavebands x vza
        # we want an array or single pixels each with the vza array
        data = job.data['BRDF'][:,:, waveband_index, :]
        data = data.reshape(data.shape[0] * data.shape[1], data.shape[2])

        # if we weren't given the indices to graph, assume all
        if indices == None:
            indices = range(len(data))

        if len(indices) < 1:
            # if we were given an empty array, or the length of the data is for some reason zero
            # we just return
            return


        x,y = np.meshgrid(job.data['longitude'], job.data['latitude'])

        # initialise the graphable data
        brdf = np.zeros((vza.shape[0],len(indices)))

        legend = []
        # cycle through each pixel
        for i in range(len(indices)):
            # get the lat long using the same index in our meshgrid
            lat = y.ravel()[indices[i]]
            lon = x.ravel()[indices[i]]
            txt_lat = u"%4.2f\u00b0" % (lat)
            txt_lon = u"%4.2f\u00b0" % (lon)

            # build legend text
            legend.extend(['pix %i (%s,%s)'%(indices[i],txt_lat,txt_lon) ])
            label = ' pixel %i (%s,%s)' % (indices[i],txt_lat, txt_lon)


            if (three_d):
                filename = job.build_filename('BRDF_3D', 'png', lats=[lat], lons=[lon])
                draw_3d_plot(data[indices[i]], vza, phi, job, label=label, filename=filename, title='%s - %s' % (title, i))
            else:
                filename = job.build_filename('BRDF_polar', 'png', lats=[lat], lons=[lon])
                draw_polar_plot(data[indices[i]], vza, phi, job, label=label, filename=filename, title='%s - %s' % (title, i))




    # -
    # - Processing statistics for each surface type
    # -
    if case == 'stats':

        surfaces = data.keys()

        var2draw = ['mean','min','max']
        ifig = 0
        for i in range(len(surfaces)):
            surf = surfaces[i]

            if type(data[surf]) == type({}):

                for j in range(len(var2draw)):
                    brdf = data[surf][var2draw[j]]
                    draw_polar_plot(brdf, vza, phi, job, title = var2draw[j]+' BRDF for '+surf+'pixels (%i)'% (data[surf]['npts']) )

                    ifig = ifig+1






def draw_3d_plot(brdf, vza, phi, job, title=None, label=None, filename=None):


    # - reshape brdf
    sz = brdf.shape[0]
    vza = np.reshape(vza,  (np.sqrt(sz),np.sqrt(sz))  )
    phi = np.reshape(phi,  (np.sqrt(sz),np.sqrt(sz))  )
    brdf = np.reshape(brdf, (np.sqrt(sz),np.sqrt(sz)) )

    # - Polar coordinates
    x = vza*np.cos(phi*np.pi/180)
    y = vza*np.sin(phi*np.pi/180)

    fig = plt.figure(figsize=(9,6))

    # need to allow for the older version of matplotlib on the linux machines
    ax = plt.gca(projection='3d')
    #ax = Axes3D(fig)
    #ax.plot_wireframe(y,x,brdf)
    ax.plot_surface(y, x, brdf, rstride=1, cstride=1, cmap=matplotlib.cm.jet, linewidth=0.5, antialiased=True, alpha=0.8)

    ax.set_zlabel('Reflectance')
    # this only works for matplotlib version > 1
    ax.set_title('%s : %s' % (title, label))

    # rotate the view to make it look like the polar graph
    # ax.view_init(elev, azim)
    ax.view_init(30, 5)


    plt.subplots_adjust(left=0, right=0.95, top=1, bottom=0, wspace=0, hspace=0)
    annotation_x = 535
    annotation_y = 390
    plt.annotate('ADAM', xy=(annotation_x, annotation_y), fontsize=22, xycoords='figure pixels')
    annotation_y -= 20
    plt.annotate('Reflectance Database', xy=(annotation_x, annotation_y), fontsize=10, xycoords='figure pixels')
    annotation_y -= 90

    annotation = 'Month: %s\n' % job.get_month_name()
    annotation += 'SZA: %s\n' % job.sza
    annotation += 'Wavelength:\n  Min: %s nm\n  Max: %s nm' % (int(job.lmbd.min()), int(job.lmbd.max()))

    plt.annotate(annotation, xy=(annotation_x, annotation_y), fontsize=12, xycoords='figure pixels')


    if filename is None:
        filename = job.build_filename('BRDF_3D', 'png')

    plt.savefig(filename, dpi=72, facecolor='None', edgecolor='None',
                orientation='portrait', papertype=None, format= 'png', bbox_inches=None)
    #plt.close(fig)


    job.add_output('brdf_plot_3d', filename, title=title)



# ===================================================================
# Display BRDF as a polar plot
#
# Inputs
#  - data to be displayed (dictionnary)
#  - waveband domain
#  - viewing zenith angles (radius = vza ; angle = phi)
#
# TODO : add a colorbar
#
# ===================================================================
def draw_polar_plot(brdf, vza, phi, job, title=None, label=None, filename=None):


    fig = plt.figure(figsize=(9,6))

    # - reshape brdf
    sz = brdf.shape[0]

    vza = np.reshape(vza,(np.sqrt(sz),np.sqrt(sz)))
    phi = np.reshape(phi,(np.sqrt(sz),np.sqrt(sz)))
    brdf = np.reshape(brdf,(np.sqrt(sz),np.sqrt(sz)))

    # -
    # - Draw contourf
    # -

    # - Polar coordinates
    maxrho = vza.max()
    x = vza*np.cos(phi*np.pi/180)
    y = vza*np.sin(phi*np.pi/180)

    # - draw surface
    ax = plt.subplot(111)

    ax.set_aspect('equal')
    ax.set_xticklabels([])
    ax.set_yticklabels([])
    id_cont = plt.contourf(x,y,brdf,20)


    # - Draw circles
    ncircles = 6

    radius = np.linspace(0,maxrho,ncircles)
    for i in range(ncircles):
        circ = circle((0,0), radius = radius[i], linestyle = 'dotted',facecolor = 'none', edgecolor = 'k')
        ax = plt.gca()
        ax.add_patch(circ)

        # labels
        if i > 0:
            if np.float(i)/2 != np.int(i/2):
                angle_txt = u"%d\u00b0" % (radius[i])
                plt.text(radius[i]*np.cos(np.pi/3)-0.2, radius[i]*np.sin(np.pi/3)-0.2,angle_txt,
                         horizontalalignment = 'center', fontsize = 7 )


    # - Axes every 45 degrees
    nax = 8
    xaxe = np.zeros((nax,10))
    yaxe = np.zeros((nax,10))
    for num_axe in range(nax):
        xmax = np.cos(num_axe*np.pi/4)*maxrho
        ymax = np.sin(num_axe*np.pi/4)*maxrho
        xaxe[num_axe,:] = np.linspace(0,xmax,10)
        if num_axe == 0:
            yaxe[num_axe,:] = np.zeros((1,10))
        else:
            yaxe[num_axe,:] = np.linspace(0,ymax,10)


        if np.float(num_axe)/2 == np.int(num_axe/2):
            linestyle = '--'
        else:
            linestyle = ':'

        plt.plot(xaxe[num_axe,:],yaxe[num_axe,:],linestyle,color = 'k', markersize =1)

        # labels
        angle_txt = u"%d\u00b0" % (num_axe*45)
        plt.text(1.1*xmax,1.1*ymax,angle_txt,horizontalalignment = 'center', fontsize = 8)

    # - Add colorbar, 'fraction' positions it left and right
    plt.colorbar(id_cont, fraction=0.3)
    plt.title('%s : %s' % (title, label))


    plt.subplots_adjust(left=0.05, right=0.6, top=0.90, bottom=0.05 )
    annotation_x = 535
    annotation_y = 390
    plt.annotate('ADAM', xy=(annotation_x, annotation_y), fontsize=22, xycoords='figure pixels')
    annotation_y -= 15
    plt.annotate('Reflectance Database', xy=(annotation_x, annotation_y), fontsize=10, xycoords='figure pixels')
    annotation_y -= 90

    annotation = 'Month: %s\n' % job.get_month_name()
    annotation += 'SZA: %s\n' % job.sza
    annotation += 'Wavelength:\n  Min: %s nm\n  Max: %s nm' % (int(job.lmbd.min()), int(job.lmbd.max()))

    plt.annotate(annotation, xy=(annotation_x, annotation_y), fontsize=12, xycoords='figure pixels')



    if filename is None:
        filename = job.build_filename('BRDF_polar', 'png')
    plt.savefig(filename, dpi=72, facecolor='None', edgecolor='None',
                orientation='portrait', papertype=None, format= 'png', bbox_inches=None)
    plt.close(fig)

    job.add_output('brdf_polar_plot', filename, title=title)

# END draw_polat_plot
# ===================================================================
