# -*- coding: utf-8 -*-
"""
title           : brdf_commandline.py
description     : script using the ADAM Toolkit demonstating taking user input from the commandline
author          : Ivan.Price@noveltis.fr
date            : Tue Jul 31 16:55:26 2012
version         : 0.1
notes           :

EXAMPLE USAGE:

python brdf_commandline.py --sza 30.0 --vza 40.0 --phi 0.0 --lat1 36.0 --lon1 5.2 --lat2 36.2 --lon2 5.5 --wave-min 437 --wave-max 437 --month jan --operation-type brdf --beyond_za_limit nan --sza_max_limit 30 --netcdf-output c://output.nc

"""


import numpy as np
import sys
#import argparse
from optparse import OptionParser

# this script may live in the test/ directory and the adam class my not be in the PATH,
# so we add this here to allow for that case
if not '../' in sys.path:
    sys.path.insert(0, '../')


# import the adam libraries
import adam
import process_brdf
import adam_config as adam_config

########################################################################
########################################################################
# user-configurable config here

# get an instance of an adam 'config' object, this will be attached to any jobs we run
#my_path_data       = 'p://3895-ADAM/code/data/'
#my_output_root_dir = 'Y://web-root/adam/web-output/'
#my_config = adam_config.adam_config(path_data = my_path_data, output_root_dir = my_output_root_dir)
my_config = adam_config.adam_config()


# end of user-configurable code !






########################################################################
########################################################################
########################################################################
# command line input

parser = OptionParser(description='ADAM commandline options')
parser.add_option('--sza', help='Sun Zenith Angle (default 45)', default=45)
parser.add_option('--vza', help='View Zenith Angle (default 0)', default=0) # MODIF_CB : changed 45 to 0
parser.add_option('--phi', help='Phi (default 0)', default=0)
parser.add_option('--sza_max_limit', help='Maximum value of sza authorized', default=70)
parser.add_option('--vza_max_limit', help='Maximum value of vza authorized', default=70)
parser.add_option('--beyond_za_limit', help='Behaviour of the BRDF models when sza of vza are beyond the max limit values', default='0', choices=['0','nan','<any constant>'])
parser.add_option('--lat1', help='Latitude 1')
parser.add_option('--lon1', help='Longitude 1')
parser.add_option('--lat2', help='Latitude 2')
parser.add_option('--lon2', help='Longitude 1')
parser.add_option('--wave-min', type=int, help='Wavelength min')
parser.add_option('--wave-max', type=int, help='Wavelength max')
parser.add_option('--month', help='Month', default='jan', choices=['jan', 'feb', 'mar', 'apr', 'may', 'jun', 'jul', 'aug', 'sep', 'oct', 'nov', 'dec'])
parser.add_option('--operation-type', help='Type of operation to perform', choices=['brdf','spectrum'])
parser.add_option('--netcdf-output', help='Name of netcdf output filename')


options, args = parser.parse_args()
def usage_and_exit():
    print '** Error parsing input **'
    parser.print_help()
    print '''Example usage:\npython brdf_commandline.py --sza 30.0 --vza 40.0 --phi 0.0 --lat1 36.0 --lon1 5.2 --lat2 36.2 --lon2 5.5 --wave-min 437 --wave-max 437 --month jan --operation-type brdf --beyond_za_limit nan --sza_max_limit 30 --netcdf-output c://output.nc\n\n'''
    sys.exit()
    
if None in [options.lat1, options.lon1, options.lat2, options.lat2, options.operation_type]:
    usage_and_exit()


########################################################################
########################################################################
########################################################################





# define the type of adam calculation we wish to perform..
request_dict = { 'fieldOperationType'  : options.operation_type,
                 'fieldSpectralDomain' : '%s-%s' % (options.wave_min, options.wave_max),
                 'fieldSunZenith'      : options.sza,
                 'fieldViewZenith'     : options.vza,
                 'fieldRelAzimuth'     : options.phi,

                 'fieldSunZenithMaxLimit'     : options.sza_max_limit,
                 'fieldViewZenithMaxLimit'    : options.vza_max_limit,
                 'fieldBeyondZALimit'         : options.beyond_za_limit,

                 
                 'fieldCorner1Lat'     : options.lat1,
                 'fieldCorner1Lon'     : options.lon1,
                 'fieldCorner2Lat'     : options.lat2,
                 'fieldCorner2Lon'     : options.lon2,

                }

# initialise the job, providing us with a blank object ('job') to work with
print 'init job/config'
job = adam.adam_job( cfg=my_config )

# now validate our request against certain logic rules, and populate any unpopulated required fields
# with default values
if not job.validate_input(request_dict):
    raise Exception('request input invalid !')


# Add something is the job similar to the following in order to retrieve:
# either the individual wavebands or the wavelengths associated to the spectral domains
#--------
#TODO: need to explain what is happening here
domains = job.spectral_domains
ndomains = len(domains)
do_spectral_averaging = None
try:
    len(domains[0])
    do_spectral_averaging = True
except:
    do_spectral_averaging = False
lmbd_user = []
if do_spectral_averaging == True:
    for i in range(len(domains)):
        d= np.linspace( domains[i][0],domains[i][1],domains[i][1]-domains[i][0]+1)
        lmbd_user.append(d)
else:
    lmbd_user = domains
lmbd_user = lmbd_user[0]
#--------

# now we are ready to load the data from our source database into our job object
print 'load data...'
job.load_data()

# print a little output to prove we know where we are in the world
print "%s land pixels, %s ocean pixels, %s snow pixels" % (len(job.data['idx_land']),
                                                           len(job.data['idx_ocean']),
                                                           len(job.data['idx_snow']))

# call the convenience functions to populate the various variables
print 'perform reflectance calc...'
job.process_reflectance()

# Increase vza sampling depending on sza (increase resolution near sza)
print 'define_vza_hotspot...'
job.vza_values = process_brdf.define_vza_hotspot(job.sza, job.cfg.vza_values)


print 'Perform BRDF calculation...'
# populates job.data['BRDF'] and job.data['BRDF_err_land']
job.process_brdf_averaged(lmbd=lmbd_user, do_spectral_averaging=do_spectral_averaging)


print 'save netcdf'
job.save_netcdf(output_filename=options.netcdf_output)


if options.netcdf_output is not None:
    print 'finished, netcdf written to: %s' % options.netcdf_output
else:
    print 'finished, netcdf written to: %s' % job.job_output_dir
