###############################################################
# Definition of the default configuration parameters
#
# Authors:
#  NOVELTIS: Cedric Bacour / Ivan Price
###############################################################

#from __init__ import *

import os
import numpy as np
import netCDF4
import socket
import logging


class adam_config():
    ''' class to be used as a singleton by all adam modules for storing config info for the adam system '''

    def __init__(self, path_data=None, output_root_dir=None, debug_log_file=None, http_root=None):

        ##############################################################################################
        # -
        # - Input/Output directories & files
        # -

        # these are normally supplied as parameters at the top of the script, as given in the examples:
        # my_config = adam_config.adam_config(path_data = my_path_data, output_root_dir = my_output_root_dir)
        # job = adam.adam_job( cfg=my_config )

        # the root directory to the data files
        self.path_data = path_data
        # the root directory into which job output directories will be created
        self.output_root_dir = output_root_dir

        # debug info will be written to this file if it is defined, used primarily by the website
        self.debug_log_file = debug_log_file

        # used by the website
        self.http_root = http_root



        ##############################################################################################



        # these are here to allow default values when no directories are supplied, primarily for
        # development / debugging by NOVELTIS.
        # adding your own value here means you don't need to explicity set the location of
        # your data in your scripts, as long as your hostname matches a key here

        config_dict = {'pc-price': {'path_data'       : 'p://3895-ADAM/code/data/',
                                    'output_root_dir' : 'Y://web-root/adam/web-output/',
                                    'debug_log_file'  : 'Y://wsgi-apps/adam/wsgi-log.log',
                                    'http_root'       : 'http://price.ws.noveltis.dev/adam/web-output/'
                                   },
                      'webdev':    {'path_data'       : '/PROJETS/3895-ADAM/code/data/',
                                    'output_root_dir' : '/NOVELTIS/price/workspace/web-root/adam/web-output/',
                                    'debug_log_file'  : '/NOVELTIS/price/workspace/wsgi-apps/adam/wsgi-log.log',
                                    'http_root'       : 'http://price.ws.noveltis.dev/adam/web-output/'
                                    },
                      'prod1':     {'path_data'       : '/PROJETS/3895-ADAM/code/data/',
                                    'output_root_dir' : '/home/www/data/adam/web-root/web-output/',
                                    'debug_log_file'  : '/home/www/data/adam/wsgi-apps/wsgi-log.log',
                                    'http_root'       : 'http://www.noveltis.com/adam-demo/web-output/'
                                   },
                      'cedric':    {'path_data'       : '/home/satellites9/cbacour/ADAM/Data/V1/',
                                    'output_root_dir' : '/home/satellites9/cbacour/ADAM/OUTPUT/',
                                    'debug_log_file'  : '/home/satellites9/cbacour/ADAM/OUTPUT/logfile.txt',
                                    'http_root'       : '/home/satellites9/cbacour/ADAM/OUTPUT/'
                                   }
                       }


        if config_dict.has_key(socket.gethostname()):
            if path_data is None        : self.path_data       = config_dict[socket.gethostname()]['path_data']
            if output_root_dir is None  : self.output_root_dir = config_dict[socket.gethostname()]['output_root_dir']
            if debug_log_file is None   : self.debug_log_file  = config_dict[socket.gethostname()]['debug_log_file']
            if http_root is None        : self.http_root       = config_dict[socket.gethostname()]['http_root']



        self.netcdf_datafile = 'ADAM_V1_PlateCarree.nc'
        self.netcdf_file_covarland_template = 'ADAM_V1_ERR_PlateCarree_M%02i.nc'
        self.path_ascii = os.path.join(self.path_data, 'ascii')                     # path where the ascii files are saved
        self.path_maps = os.path.join(self.path_data,'maps_raw')                    # output path for saving maps

        # - Land
        # refractive index from http://refractiveindex.info/?group=CRYSTALS&material=H2O-ice,
        #  Warren S.G (1984, Optical constants of ice from the ultraviolet to the microwave, Applied Optics, 23:1206-1225)
        self.file_land_IceRefInd = 'ADAM_V1_IceComplexRefractiveIndex_300-4000.txt'
        # - Ocean
        self.file_ocean_ref_std = 'ADAM_V1_OceanTypicalReflectance_300-800.txt'
        # refractive index from http://refractiveindex.info/?group=LIQUIDS&material=Water
        #  G. M. Hale and M. R. Querry. Optical Constants of Water in the 200-nm to 200-microm Wavelength Region, Appl. Opt. 12, 555-563 (1973) doi:10.1364/AO.12.000555
        self.file_ocean_WaterRefInd = 'ADAM_V1_WaterRefractiveIndex_300-4000.txt'
        # absorption coefficient of water
        # from  D. J. Segelstein, "The complex refractive index of water," University of Missouri-Kansas City, (1981). (http://omlc.ogi.edu/spectra/water/data/segelstein81.dat)
        self.file_ocean_WaterAbs = 'ADAM_V1_WaterAbsorption_300-4000.txt' # CHANGE_CB 



        #OUTPUT_ROOT_DIR = '/home/satellites9/cbacour/ADAM/OUTPUT/'
        #DEBUG_LOG_FILE  = '/home/satellites9/cbacour/ADAM/OUTPUT/'
        #HTTP_ROOT       = '/home/satellites9/cbacour/ADAM/OUTPUT/'

        self.netcdf_datafile = os.path.join(self.path_data, self.netcdf_datafile)
        self.netcdf_file_covarland_template = os.path.join(self.path_data, self.netcdf_file_covarland_template)

        logging.basicConfig(filename=debug_log_file, level=logging.DEBUG, format='%(asctime)s %(levelname)s %(message)s')


    # -
    # - Global parameters
    # -

        # the maximum allowable size in degrees for a (non-download) request (20x20)
        self.limit_square_degrees = 410

        #maximum limit to perform pixel-level calculations
        self.limit_degrees_pixel_calcs = 0.1

        # -
        # - Names of the variables to read in the gridded ADAM NetCDF file
        # -
        self.vars_land = ['ref_land']   # respect this order!
        self.vars_ocean = ['wind_speed','chloro_conc']      # respect this order!


        # - standard viewing geometry (in degrees)
        self.sza_std = 40
        self.vza_std = 0
        self.phi_std = 0

        vza_range  = [-65,65]
        vza_step   = 5
        self.vza_values = np.arange(vza_range[0], vza_range[1]+5, vza_step)
        self.phi_values = np.arange(0,360+10,10)

        self.step_polar = 41 # discretisation step for drawing polar plots

        # maximum value authorized
        self.sza_max_limit = 70
        self.vza_max_limit = 70
        # behaviour of the BRDF models when sza of vza are beyond the max limit values:
        #  - '0': set the corresponding BRDF values to 0
        #  - 'nan': set the corresponding BRDF values to NaN
        #  - 'constant': set the corresponding BRDF values to the values it would have for sza/vza = max limit
        self.beyond_za_limit = 'constant'


        # missing values
        self.missval_1b = 255
        self.missval_2b = -9999.

        self.lmbd = np.arange(4000-300+1)+300
        self.spectral_domains = [[300,400],[400,700],[700,1300],[1300,2500],[2500,4000]]

        # maximum number of pixels to display individual data. Beyond mx_pts_analysis, the statistics are drawn
        self.max_pts_analysis = 6

        # parameters for the RGB composite image
        self.lmbd_R = 670
        self.lmbd_G = 460
        self.lmbd_B = 560
        self.ilmbd_R = np.where(self.lmbd == self.lmbd_R)[0][0]
        self.ilmbd_G = np.where(self.lmbd == self.lmbd_G)[0][0]
        self.ilmbd_B = np.where(self.lmbd == self.lmbd_B)[0][0]

        try:
            source_dataset = netCDF4.Dataset(self.netcdf_datafile, 'r')
        except:
            raise Exception('Error opening main source data file, location given was: %s' % self.netcdf_datafile)

        # load the modis wavebands for convenience later
        self.modis_wavebands = np.round(source_dataset.variables['wavebands'][:])
        source_dataset.close()

        self.land , self.ocean = self._get_land_ocean_configs()

    # - Land/Snow/Sea masks (INFO FOR IVAN : will be populated in process_reflectance)
#    mask_land = None
#    mask_snow = None
#    mask_ocean = None
#    idx_land = []
#    idx_snow = []
#    idx_ocean = []


    def _get_land_ocean_configs(self):


        # - define Global class
        #    cfg = defCglobal(limit_square_degrees, limit_degrees_pixel_calcs, vars_land,vars_ocean,
        #                     sza_std,vza_std,phi_std,vza_range,vza_step,vza_values,phi_values,step_polar,
        #                     missval_1b,missval_2b,spectral_domains,
        #                     max_pts_analysis,
        #                     lmbd,lmbd_R,lmbd_G,lmbd_B,ilmbd_R,ilmbd_G,ilmbd_B,
        #                     path_data,netcdf_datafile,path_figures,path_maps,
        #                     netcdf_file_covarland_template,
        #                     OUTPUT_ROOT_DIR,DEBUG_LOG_FILE,HTTP_ROOT,
        #                     mask_land,mask_snow,mask_ocean,idx_land,idx_snow,idx_ocean)

        # -
        # - Land
        # -
        file_acp =  os.path.join(self.path_data,'ADAM_V1_ACP.nc')
        try:
            dataACP = netCDF4.Dataset(file_acp, 'r')
        except:
            raise Exception('Error opening ACP data file, location given was: %s' % file_acp)

        kice_data = np.loadtxt(os.path.join(self.path_ascii, self.file_land_IceRefInd), comments='#', skiprows=0, unpack=True)

        AttributeDict
        config_land = AttributeDict({
            'missval'             : -8421,
            'add_offset_land'     : 0.,
            'scale_factor_land'   : 0.0001,

            # - Spectrum parameters (ACP)
            'file_acp'            : (os.path.join(self.path_data,'ADAM_V1_ACP.nc')),
            'ACP_eigenvectors'    : dataACP.variables['Eigenvectors'][:],
            'ACP_ObsMean'         : dataACP.variables['ObsMean'][:],
            'ACP_SpecMean'        : dataACP.variables['SpecMean'][:],
            'ACP_prod'            : dataACP.variables['prodACP'][:],
            'ACP_lmbd'            : dataACP.variables['wavelength'][:],
            'ACP_neigenvectors'   : dataACP.variables['neigenvectors'][:],


            # - BRDF parameters
              # vis / nir
            'slope_R'                 : np.array([0.2, -0.05]),
            'slope_V'                 : np.array([1, 2]),
            'intercept_R'             : np.array([0.1, 0.15]),
            'intercept_V'             : np.array([0.5, 0.5]),
            'alpha_RV'                : 0.5,
            'err_R'                   : 0.05,
            'err_V'                   : 0.15,

            # - snow
            'snow_size_grain'         : 0.12e-3,
            'chanB'                   : 0, #CHANGE_CB 
            'chanR'                   : 2,
            'chanPIR'                 : 3,
            'chanPIR2'                : 4,
            'cond_snow_1'             : 0.4,
            'cond_snow_2'             : 0.,

            # - ice complex refractive index

            'lmbd_kice'               : kice_data[0,:],
            'kice'                    : kice_data[1,:]
        })

        #ACP_eigenvectors = ACP_eigenvectors[:,ACP_neigenvectors][:]
        #ACP_mat = np.dot(ACP_eigenvectors,ACP_prod)
        config_land['ACP_eigenvectors'] = config_land['ACP_eigenvectors'][:, config_land['ACP_neigenvectors']][:]
        # can may be computed only once
        config_land['ACP_mat'] = np.dot(config_land['ACP_eigenvectors'], config_land['ACP_prod'])

        dataACP.close()



    # -
    # - Ocean
    # -

        nwater_data = np.loadtxt(os.path.join(self.path_ascii, self.file_ocean_WaterRefInd), comments='#', skiprows=0, unpack=True)

        config_ocean = AttributeDict({
            'missval'           : -32768,
            'add_offset_ws'     : 15.97722,
            'scale_factor_ws'   : 0.0004026239,
            'add_offset_chl'    : -0.1103447,
            'scale_factor_chl'  : 5.676205e-05,

            # - Water refractive Index
            'lmbd_nwater'       : nwater_data[0,:],
            'nwater'            : nwater_data[1,:]
        })

        # - typical ocean reflectance spectra wrt chlorophyll content
        fic = open(os.path.join(self.path_ascii, self.file_ocean_ref_std), mode='rU')
        lines = fic.readlines()
        fic.close()

        nl = len(lines)
        nlmbd  = 501
        nchl = 6
        ref_ocean_chl_std = np.zeros((nchl, nlmbd), np.float)
        lmbd_ocean_chl_std = np.zeros(nlmbd, np.float)
        buf = (lines[0].strip()).split()
        buf = [float(el) for el in buf[1:]]

        chl_ocean = np.array(buf[:])

        for i in range(nl):
            if i == 0: continue
            buf = (lines[i].strip()).split()
            lmbd_ocean_chl_std[i-1] = float(buf[0])
            buf = [float(el) for el in buf[1:]]
            ref_ocean_chl_std[:,i-1] = np.array(buf)

        config_ocean['chl'] = chl_ocean
        config_ocean['ref_chl_std'] = ref_ocean_chl_std
        config_ocean['lmbd_chl_std'] = lmbd_ocean_chl_std

        # > MODIF_CB
        # - foam reflectance as a function of foam (from Koepke, 1984)
        data = np.array([[4,  0.00],[5,  0.00],[6,  0.00],[7,  0.00], [8,  0.10],[9,  0.20],\
                         [10, 0.20],[11, 0.30],[12, 0.40],[13, 0.50],[14, 0.70],[15, 0.90],\
                         [16, 1.10],[17, 1.30],[18, 1.60],[19, 2.00],[20, 2.30],[25, 5.10]])
        ws_0 = data[:,0].ravel()
        ref_0 = (data[:,1].ravel())/100.
        #  interpolation for wind speed between [0 ; 30] m.s-1
        ws_foam_std = np.arange(31)
        ref_foam_std = np.interp(ws_foam_std, ws_0, ref_0)
        config_ocean['foam_ws_std'] = ws_foam_std
        config_ocean['foam_ref_std'] = ref_foam_std

        # - Spectral variation of the foam contribution
        # (see Kokhanovsky A.A. (2004), Spectral reflectance of whitecaps, Journal of Geophysical Research, 109, C05021, doi:10.1029/2003JC002177)
        if not os.path.exists(os.path.join(self.path_ascii, self.file_ocean_WaterAbs)):
            raise Exception('Error finding foam datafile at: %s !' % os.path.join(self.path_ascii, self.file_ocean_WaterAbs))
        data = np.loadtxt(os.path.join(self.path_ascii, self.file_ocean_WaterAbs), comments='#', skiprows=0, unpack=True)
        abs_water_foam = data[1,:]
        b_foam = 1.72*1e-3
        c_foam = 0.00027*1e6
        config_ocean['foam_specmod'] = np.exp(-np.sqrt((abs_water_foam+c_foam)*b_foam))
        ilmbd_VIS = np.where(self.lmbd == 400)[0][0]
        config_ocean['foam_normfac'] = config_ocean['foam_specmod'][ilmbd_VIS]

        # < MODIF_CB

    # -
    # - Return
    # -
        return config_land, config_ocean


class AttributeDict(dict):
    ''' simple override class to provide a dictionary that allows for access of its elements
        like attributes.. for example: dict.value = 'xxxx' instead of dict['value'] = 'xxxx'

        ATTENTION because we cannot expect dict['my key'] to work like dict.my key
        this is here because it keeps the code a bit cleaner and we are aware of this danger
    '''

    def __getattr__(self, attr):
        return self[attr]

    def __setattr__(self, attr, value):
        self[attr] = value


#
## ===================================================================
## land Class
## ===================================================================
#class defCland:
#    """
#    Land class containing all information required to compute
#    reflectance spectra and BRDF for land pixels
#    """
#    ##@param[in] self   python implicit argument
#    ##@param[in]
#    ##@param[in]
#    ##@param[in]
#    ##@param[in]
#    def __init__(self,slope_R,slope_V,intercept_R,intercept_V,alpha_RV,err_R,err_V,
#                 missval,add_offset,scale_factor,
#                 snow_size_grain,chanPIR2,chanPIR,chanR,cond_snow_1,cond_snow_2,
#                 lmbd_kice,kice,
#                 ACP_eigenvectors,ACP_ObsMean,ACP_SpecMean,ACP_prod,ACP_lmbd,ACP_neigenvectors,ACP_mat):
#        self.slope_R         = slope_R
#        self.slope_V         = slope_V
#        self.intercept_R     = intercept_R
#        self.intercept_V     = intercept_V
#        self.alpha_RV        = alpha_RV
#        self.err_R           = err_R
#        self.err_V           = err_V
#        self.missval         = missval
#        self.add_offset      = add_offset
#        self.scale_factor    = scale_factor
#        self.lmbd_kice       = lmbd_kice
#        self.kice            = kice
#        self.snow_size_grain = snow_size_grain
#        self.chanPIR2        = chanPIR2
#        self.chanPIR         = chanPIR
#        self.chanR           = chanR
#        self.cond_snow_1     = cond_snow_1
#        self.cond_snow_2     = cond_snow_2
#        self.ACP_eigenvectors = ACP_eigenvectors
#        self.ACP_ObsMean     = ACP_ObsMean
#        self.ACP_SpecMean    = ACP_SpecMean
#        self.ACP_prod        = ACP_prod
#        self.ACP_lmbd        = ACP_lmbd
#        self.ACP_neigenvectors = ACP_neigenvectors
#        self.ACP_mat = ACP_mat
#
#
## ===================================================================
## ocean Class
## ===================================================================
#class defCocean:
#    """
#    Ocean class containing all information required to compute
#    reflectance spectra and BRDF for ocean pixels
#    """
#    ##@param[in] self   python implicit argument
#    ##@param[in]
#    ##@param[in]
#    ##@param[in]
#    ##@param[in]
#    def __init__(self,
#                 missval,add_offset_ws,scale_factor_ws,add_offset_chl,scale_factor_chl,
#                 lmbd_nwater,nwater,chl,ref_chl_std,lmbd_chl_std):
#        self.missval          = missval
#        self.add_offset_ws    = add_offset_ws
#        self.scale_factor_ws  = scale_factor_ws
#        self.add_offset_chl   = add_offset_chl
#        self.scale_factor_chl = scale_factor_chl
#        self.lmbd_nwater      = lmbd_nwater
#        self.nwater           = nwater
#        self.chl              = chl
#        self.ref_chl_std      = ref_chl_std
#        self.lmbd_chl_std     = lmbd_chl_std
#
#
## ===================================================================
## Global Class
## ===================================================================
#class defCglobal:
#    """
#    Gloabl class containing all information required to compute
#    reflectance spectra and BRDF for land  & oceanpixels
#    """
#    def __init__(self,limit_square_degrees, limit_degrees_pixel_calcs, vars_land,vars_ocean,
#                 sza_std,vza_std,phi_std,vza_range,vza_step,vza_values,phi_values,step_polar,
#                 missval_1b,missval_2b,spectral_domains,
#                 max_pts_analysis,
#                 lmbd,lmbd_R,lmbd_G,lmbd_B,ilmbd_R,ilmbd_G,ilmbd_B,
#                 path_data,netcdf_datafile,path_figures,path_maps,
#                 netcdf_file_covarland_template,
#                 OUTPUT_ROOT_DIR,DEBUG_LOG_FILE,HTTP_ROOT,
#                 mask_land,mask_snow,mask_ocean,idx_land,idx_snow,idx_ocean):
#
#
#        self.limit_square_degrees = limit_square_degrees
#        self.limit_degrees_pixel_calcs = limit_degrees_pixel_calcs
#        self.vars_land  = vars_land
#        self.vars_ocean = vars_ocean
#        self.sza_std = sza_std
#        self.vza_std = vza_std
#        self.phi_std = phi_std
#        self.vza_range = vza_range
#        self.vza_step = vza_step
#        self.vza_values = vza_values
#        self.phi_values = phi_values
#        self.step_polar = step_polar
#        self.missval_1b = missval_1b
#        self.missval_2b = missval_2b
#        self.spectral_domains = spectral_domains
#        self.max_pts_analysis = max_pts_analysis
#        self.lmbd = lmbd
#        self.lmbd_R = lmbd_R
#        self.lmbd_G = lmbd_G
#        self.lmbd_B = lmbd_B
#        self.ilmbd_R = ilmbd_R
#        self.ilmbd_G = ilmbd_G
#        self.ilmbd_B = ilmbd_B
#        self.path_data = path_data
#        self.netcdf_datafile = netcdf_datafile
#        self.netcdf_file_covarland_template = netcdf_file_covarland_template
#        self.path_figures = path_figures
#        self.path_maps = path_maps
#        self.OUTPUT_ROOT_DIR = OUTPUT_ROOT_DIR
#        self.DEBUG_LOG_FILE = DEBUG_LOG_FILE
#        self.HTTP_ROOT = HTTP_ROOT
#        self.mask_land = mask_land
#        self.mask_snow = mask_snow
#        self.mask_ocean = mask_ocean
#        self.idx_land = idx_land
#        self.idx_snow = idx_snow
#        self.idx_ocean = idx_ocean

