# -*- coding: utf-8 -*-
"""
title           : brdf_multidir_brem.py
description     : script using the ADAM Toolkit demonstating the parallelisation of multiple
                  ADAM jobs each representing different observation and illumination angles,
                  specialised for the BREM-use case.
                  
version         : 0.1
usage           : python brdf_multidir_brem.py --help
                : python brdf_multidir_brem.py --insza 30.0 --lat1 36.0 --lon1 5.2 --lat2 36.2 --lon2 5.5 --wave-min 437 --wave-max 447 --month jan --netcdf-output /tmp/foo.nc
notes           : before use ensure that the 4 parameters are correct

"""

import numpy as np
import sys
import time
import multiprocessing
import traceback
from optparse import OptionParser
# this script may live in the test/ directory and the adam class my not be in the PATH,
# so we add this here to allow for that case
if not '../' in sys.path: sys.path.insert(0, '../')
# import the adam libraries
import adam
import adam_config as adam_config

####################################################################################
####################################################################################
####################################################################################
####################################################################################
#
# BEGIN CONFIGURABLE PARAMETERS

# get an instance of an adam 'config' object, this will be attached to any jobs we run
# will need to setup site-specific config here
## BREM INFO FOLLOWS:
#my_path_data       = '/data/ADAMANDEVE/NOVELTIS/ADAMv1/Data/' 
#my_output_root_dir = '/data/ADAMANDEVE/NOVELTIS/ADAMv1/OUTPUT/' 
#my_config = adam_config.adam_config(path_data = my_path_data, output_root_dir = my_output_root_dir)

my_config = adam_config.adam_config()




# declare our user-defined parameters for input
NUM_PROCESS_LIMIT = 2
# VZA_VALUES        = range(0,4,2)
# SZA_VALUES        = range(0,8,2)
# PHI_VALUES        = range(0,100)
# LAT1              = 0
# LON1              = 0
# LAT2              = 5
# LON2              = 5

#16 Gauss angles
GAU_ANG = np.array([89.696,88.412,86.148,82.975,78.985,74.277,
                    68.949,63.096,56.804,50.148,43.197,36.008,
                    28.634,21.122,13.520, 5.901])

# SZA
#1 - Incidence angles [deg]:
#;last entry - solar zenith angles [deg] of the satellite measurement
# for BREM these will be populated from the command line !!!!
SZA_VALUES = np.array([])

# VZA
#2 - Reflection angles [deg]:
#last entry - nadir direction 0 [deg]
VZA_VALUES = np.append(GAU_ANG, 0.000)
 
# PHI
#3 - Azimuthal angles [deg] - 100 elements
PHI_VALUES = np.array([0.026,  0.136,  0.333,  0.619,  0.991,  1.451,
                      1.997,  2.629,  3.347,  4.149,  5.035,  6.004,
                      7.055,  8.187,  9.399, 10.690, 12.059, 13.503,
                      15.022, 16.615, 18.279, 20.013, 21.816, 23.685,
                      25.620, 27.617, 29.675, 31.791, 33.965, 36.194,
                      38.475, 40.806, 43.186, 45.611, 48.080, 50.589,
                      53.137, 55.721, 58.339, 60.988, 63.664, 66.367,
                      69.093, 71.839, 74.603, 77.382, 80.173, 82.974,
                      85.782, 88.593, 91.407, 94.218, 97.026, 99.827,
                      102.618,105.397,108.161,110.907,113.633,116.336,
                      119.012,121.661,124.279,126.863,129.411,131.920,
                      134.389,136.814,139.194,141.525,143.806,146.035,
                      148.209,150.325,152.383,154.380,156.315,158.184,
                      159.987,161.721,163.385,164.978,166.497,167.941,
                      169.310,170.601,171.813,172.945,173.996,174.965,
                      175.851,176.653,177.371,178.003,178.549,179.009,
                      179.381,179.667,179.864,179.974])


# END CONFIGURABLE PARAMETERS
####################################################################################
####################################################################################
####################################################################################
####################################################################################






def run_single_job(request_dict):
    ''' The worker function, called once for each iteration withing a test run.
    This will be run by all the child processes that we will spawn for the
    various permutations to be calculated.
    '''    
    
    # instantiate an ADAM job using the input dictionary. This is performed in a 'try'
    # to catch an error, allowing us to notify the parent if neccessary
    try:
        # the job is initialised without creating a working directory,
        # as we want to save the output of many jobs to the same location
        job = adam.adam_job( cfg=my_config, create_working_dir=False )
        job.validate_input(request_dict)
        job.load_data()
        job.process_reflectance()
        # specify that the vza values are an array
        job.vza = VZA_VALUES
        # specify that the phi values are an array
        job.phi = PHI_VALUES
        # populate the job.data['BRDF'] array
        job.process_brdf(do_spectral_averaging=True)
        # save the job to a specified location
        
        filename = options.output.replace('.nc', '')
        filename = '%s-%s.nc' % (filename, request_dict['fieldSunZenith']) 
        job.save_netcdf(output_filename=filename)


        #print 'finished for sza: %s' % (request_dict['fieldSunZenith'])
        # return the data to the parent for use later if neccessary
        return (request_dict['fieldSunZenith'], job.data['BRDF'])
    
    # if there was an error during the calculation, we catch it here
    except:
        # if there is an error anywhere in the processing, OR we receive a ctrl-c break
        # print the error and allow the child to die
        trace_back = traceback.format_exception(sys.exc_info()[0], sys.exc_info()[1], sys.exc_info()[2])
        trace_back = '\n'.join(trace_back)
        print 'ERROR captured in child process, message follows:'
        print trace_back
        pass
        
        
        
        
def launch_test_cycle(options):
    ''' parent function, called with options populated from the commandline '''
    
    # for time keeping, record the start time
    time_start = time.time() 
    
    # now the worker(child) function is defined, we build the parent that will call it

    # first define the type of adam calculation we wish to perform..
    # this is the same as the request_dict object in the other examples
    request_dict = { 'fieldOperationType' : 'brdf',    
                     'fieldSpectralDomain': '%s-%s' % (options.wave_min, options.wave_max),
                     'fieldBRType'        : 'principal',
                     'fieldViewZenith'    : 0,
                     'fieldRelAzimuth'    : 0,
                     'fieldMonth'         : options.month,
                     'fieldCorner1Lat'    : options.lat1,
                     'fieldCorner1Lon'    : options.lon1,
                     'fieldCorner2Lat'    : options.lat2,
                     'fieldCorner2Lon'    : options.lon2 
                    }
    
    # we will build an array of request_dict objects, calling our worker children 
    # with one each. Each request_dict object varies by vza and sza value
    request_dicts = []
    for sza in SZA_VALUES:
        request_dict['fieldSunZenith'] = sza
        request_dicts.append(request_dict.copy())
    
    
    # initialise the processing pool with the correct number of concurrent processors
    pool = multiprocessing.Pool(processes=NUM_PROCESS_LIMIT)        
    
    # launch the calculations.. map_async will launch le worker function defined above with each
    # element in the request_dicts array, effectively launching one ADAM job for each request_dict
    # we have defined
    results = pool.map_async(run_single_job, request_dicts)
    
    # now we wait for all calculations to complete
    pool.close()
    pool.join()
    
    # load the results into an array and build a dictionary with which we can traverse them easily
    results =  results.get()
    # we will load all the results into a master dictionary
    r = {}
    for result in results:
        # if there was an error in processing the result will be None, 
        # if so we catch that here
        if result is None:
            print 'ERROR, a child returned none, which means there was an error during processing'
            print 'Review the error above and re-try'
            sys.exit()
        # otherwise this child must have executed sucessfully
        sza, data_brdf = result
        r[sza] = data_brdf
    
    
    
    # we're all done.. now output some interesting statistics
    cases = len(VZA_VALUES) * len(SZA_VALUES) * len(PHI_VALUES)
    time_taken = time.time() - time_start
    square_degrees = abs(request_dict['fieldCorner1Lat'] - request_dict['fieldCorner2Lat']) * float(abs(request_dict['fieldCorner1Lon'] - request_dict['fieldCorner2Lon']))
    pixels = square_degrees * 100
    efficiency = pixels * cases / time_taken
    
    print 'made array of data each of shape: %s (pixels_x, pixels_y, wavebands, vza, phi)' % [data_brdf.shape]
    print 'with %s cpus performed: %s calculations in: %s seconds for: %s square degrees' % (NUM_PROCESS_LIMIT, cases, time_taken, square_degrees)
    print 'efficiency: %s pixel-cases / second' % efficiency

    # end of program
    sys.exit()



if __name__ == "__main__":
    ''' initial function executed when this script is run from the command line.
    Performs command line parsing and calls the parent function with the result. '''

    # ---------------------------
    #  Parse options & arguments
    # ---------------------------
    parser = OptionParser()
    
    parser.add_option("--insza",    dest="insza",    metavar="Sun Zenith Angle",   help="")
    parser.add_option("--lat1",     dest="lat1",     metavar="Latitude 1",      help="")
    parser.add_option("--lat2",     dest="lat2",     metavar="Latitude 2",      help="")
    parser.add_option("--lon1",     dest="lon1",     metavar="Longitude 1",     help="")
    parser.add_option("--lon2",     dest="lon2",     metavar="Longitude 2",     help="")
    parser.add_option("--wave-min", dest="wave_min", metavar="Wavelength Min",  help="")
    parser.add_option("--wave-max", dest="wave_max", metavar="Wavelength Max",  help="")
    parser.add_option("--month",    dest="month",    metavar="Month",          help="Name of Month ('Jan' / 'Feb' / 'Mar'..)")
    parser.add_option("--netcdf-output",   dest="output",   metavar="filename",   help="Filename for NetCDF output")
    
    (options, args) = parser.parse_args()
    
    def usage_and_exit():
        print '** Error parsing input **'
        parser.print_help()
        print '''Example usage:\npython brdf_multidir_brem.py --insza 30.0 --lat1 36.0 --lon1 5.2 --lat2 36.2 --lon2 5.5 --wave-min 437 --wave-max 437 --month jan 
--netcdf-output output.nc\n\n'''
        sys.exit()
        
    if None in [options.insza,options.lat1,options.lat2,options.lon1,options.lon2,
        options.wave_min,options.wave_max,options.month,options.output]:
        usage_and_exit()

    # if any of the numeric options can't be converted to floats throw an error    
    try:
        options.insza = np.float(options.insza)
        options.lat1  = np.float(options.lat1)
        options.lat2  = np.float(options.lat2)
        options.lon1  = np.float(options.lon1)
        options.lon2  = np.float(options.lon2)
    except:
        usage_and_exit()
    
    # the sza_values contains the list of GAU_ANG calues plus our variable one added at the end (?)
    SZA_VALUES = np.append(GAU_ANG, options.insza)
    
    # finally launch the calculations
    launch_test_cycle(options)