# -*- coding: utf-8 -*-
###############################################################
# Authors:
#  NOVELTIS: Cedric Bacour / Ivan Price
###############################################################

"""

WSGI application executed by a web server to run jobs for the ADAM project

"""
import sys
import numpy as np
import cgi
import traceback
import json
from socket import gethostname
import time
import os
# this is a signal to the adam system to indicate it is being run for the web
os.environ['ADAM_WEB_REQUEST'] = 'true'


# need to tell the web server to look in our wsgi scripts path to find the modules we will be requesting
# we insert at position 0 to exert precedence over existing python modules, which is why we
# can't use site.addsitedir()


# the web server needs to add these to his path as he does not run the code from the current directory
for string in ['/NOVELTIS/price/workspace/wsgi-apps/adam/','/home/www/data/adam/wsgi-apps/']:
    if not string in sys.path:
        sys.path.insert(0, string)




import adam
import adam_config
import process_brdf


# get an instance of an adam 'config' object, this will be attached to any jobs we run
# we supply no path details here as the adam_config file knows us and will build the correct values
# based on the hostname of the machine running the code
my_config = adam_config.adam_config()


def application(environ, start_response):

    def httpResponse(response_status,response_body):
        response_headers = [('Content-Type', 'text/html'), ('Content-Length', str(len(response_body))) ]
        start_response(response_status, response_headers)
        return [response_body]
        
    response_status = '200 OK'
    
    d = cgi.parse_qs(environ['QUERY_STRING'])
    request_dict = {}
    for key in d.keys():
        request_dict[key] = cgi.escape( d.get(key, [''])[0] ).lower()

    operation_type = request_dict.get('fieldOperationType', '').strip()

    if request_dict.has_key('_dc'):
        del(request_dict['_dc'])
        
    ##################################
    if operation_type == 'brdf-time':
        try:
            htmlResult = performBRDFTime(request_dict)
        except:
            trace_back = traceback.format_exception(sys.exc_info()[0], sys.exc_info()[1], sys.exc_info()[2])
            response_status = '514 BAD BRDF Time'
            htmlResult = json.dumps({'success': 'false',  'message': '\n'.join(trace_back)})
    
    ##################################
    elif operation_type == 'brdf':
        try:
            htmlResult = performBRDF(request_dict)
        except:
            trace_back = traceback.format_exception(sys.exc_info()[0], sys.exc_info()[1], sys.exc_info()[2])
            response_status = '514 BAD BRDF'
            htmlResult = json.dumps({'success': 'false',  'message': '\n'.join(trace_back)})
    
    ###################################
    elif operation_type == 'spectrum':
        try:
            htmlResult = performSpectrum(request_dict)
        except:
            trace_back = traceback.format_exception(sys.exc_info()[0], sys.exc_info()[1], sys.exc_info()[2])
            response_status = '514 BAD SPECTRUM'
            htmlResult = json.dumps({'success': 'false',  'message': '\n'.join(trace_back)})

    ##################################
    elif operation_type == 'download':
        try:
            htmlResult = performDownload(request_dict)
        except:
            trace_back = traceback.format_exception(sys.exc_info()[0], sys.exc_info()[1], sys.exc_info()[2])
            response_status = '514 BAD DOWNLOAD'
            htmlResult = json.dumps({'success': 'false',  'message': '\n'.join(trace_back)})

    ##################################
    
    else:
        response_status = '515 BAD OPERATION'
        htmlResult = json.dumps({'success': 'false',  'message': 'no operation type found !'})
        

    
    response_body = htmlResult
    return httpResponse(response_status,response_body)

def performDownload(request_dict):
    
    job = adam.adam_job(cfg=my_config)
    
    job.set_status('begin download job on host: %s' % gethostname())
    job.set_status('parsing job parameters')
    if not job.validate_input(request_dict):
        raise Exception('request input invalid !')
    
    job.set_status('all input was validated')
    job.set_status('extent is: %.05f, %.05f, %.05f, %.05f' % (job.extent['minx'], job.extent['miny'], job.extent['maxx'], job.extent['maxy'], ))
    job.set_status('month_index is %s: ' % job.month_index)

    job.set_status('loading netcdf data')
    job.load_data()
    
    job.set_status('saving netcdf data')
    job.save_netcdf()
    
    job.end_time = time.time()
    job.set_status('job finished, took %s seconds' % str(job.end_time - job.start_time))

    return json.dumps({'success': 'true',  'log': job.running_status, 'outputs': job.outputs}, indent=4)
    
def performBRDFTime(request_dict):
    
    job = adam.adam_job(cfg=my_config)
    
    job.set_status('begin brdf time series job on host: %s' % gethostname())
    job.set_status('parsing job parameters')
    if not job.validate_input(request_dict):
        raise Exception('request input invalid !')
    
    job.set_status('all input was validated')
    job.set_status('extent is: %.05f, %.05f, %.05f, %.05f' % (job.extent['minx'], job.extent['miny'], job.extent['maxx'], job.extent['maxy'], ))
    job.graph_brdf_time()
    
    #job.set_status('saving netcdf data')
    #job.save_netcdf()
    
    job.end_time = time.time()
    job.set_status('job finished, took %s seconds' % str(job.end_time - job.start_time))

    return json.dumps({'success': 'true',  'log': job.running_status, 'outputs': job.outputs}, indent=4)


    
def performSpectrum(request_dict):
    
    job = adam.adam_job(cfg=my_config)
    job.set_status('begin spectrum job on host: %s' % gethostname())
    job.set_status('received request dictionary: %s' % request_dict)
    job.set_status('parsing job parameters')
    if not job.validate_input(request_dict):
        raise Exception('request input invalid !')
    
    job.set_status('all input was validated')
    job.set_status('extent is: %.05f, %.05f, %.05f, %.05f' % (job.extent['minx'], job.extent['miny'], job.extent['maxx'], job.extent['maxy'], ))
    job.set_status('month_index is %s: ' % job.month_index)
    job.set_status('sza is %s: ' % job.sza)
    job.set_status('vza is %s: ' % job.vza)
    job.set_status('phi is %s: ' % job.phi)
    
    
    job.set_status('loading netcdf data')
    job.load_data()
    

    job.set_status('loading reflectance data')
    job.process_reflectance(do_compute_error=True)

    job.set_status('calulate brdf')
    job.process_brdf(do_spectral_averaging=False)



    
    

    if job.is_pixel_request():
        job.set_status('call make pixels spectral graph')
            
        job.set_status('write land pixel graph (%s pixels)' % len(job.data['idx_land']))
        for i, pixel_index in enumerate(job.data['idx_land']):
            idx = [pixel_index]
            job.graph_main_ref_spectra(case='pixels', indices=idx, title='Land pixels')
        
        
        #job.graph_main_ref_spectra(case='pixels', indices=data['idx_land'],  title='Land pixel info', graph_error=True)
        
        job.set_status('write ocean pixel graph (%s pixels)' % len(job.data['idx_ocean']))
        job.graph_main_ref_spectra(case='pixels', indices=job.data['idx_ocean'], title='Ocean pixels')

        job.set_status('write snow pixel graph (%s pixels)' % len(job.data['idx_snow']))
        for i, pixel_index in enumerate(job.data['idx_snow']):
            idx = [pixel_index]
            job.graph_main_ref_spectra(case='pixels', indices=idx, title='Snow pixels')
        


    else:
        # otherwise do stats
        job.set_status('perform stats analysis')
        job.calculate_stats(job.data['BRDF'])
        
        job.set_status('call make stats graph')
        job.graph_main_ref_spectra(case='stats')
    
    
    job.set_status('saving netcdf data')
    job.save_netcdf()
    
    job.end_time = time.time()
    job.set_status('job finished, took %s seconds' % str(job.end_time - job.start_time))

    return json.dumps({'success': 'true',  'log': job.running_status, 'outputs': job.outputs}, indent=4)



    
def performBRDF(request_dict):
    
    job = adam.adam_job(cfg=my_config)
    job.set_status('begin BRDF job on host: %s' % gethostname())
    job.set_status('received request dictionary: %s' % request_dict)
    job.set_status('parsing BRDF input')
    
    if not job.validate_input(request_dict):
        raise Exception('request input invalid !')

    ############################################
    # input is validated and variables populated    
    job.set_status('all input was validated')
    
    
    job.set_status('extent is: %.05f, %.05f, %.05f, %.05f' % (job.extent['minx'], job.extent['miny'], job.extent['maxx'], job.extent['maxy'], ))
    job.set_status('month_index is %s: ' % job.month_index)
    job.set_status('spectral ranges are %s: ' % job.spectral_domains)
    job.set_status('sza is %s: ' % job.sza)
    job.set_status('phi is %s: ' % job.phi)


    
    
    job.set_status('requesting data from source file')
    job.load_data()

    job.set_status('%s land pixels, %s ocean pixels, %s snow pixels' % (len(job.data['idx_land']),
                                                                       len(job.data['idx_ocean']),
                                                                       len(job.data['idx_snow'])))
                                                           
    job.set_status('process_reflectance')
    job.process_reflectance()



    if 'polar' in job.brdf_graph_type:
#            job.vza, job.phi = process_brdf.define_angles_polar_plot(job.cfg.vza_values, job.cfg.step_polar)
        
        print 'polar'
        
        job.phi = np.linspace(0, 360, job.cfg.step_polar)
        job.vza = np.linspace(0, job.cfg.vza_values.max(), job.cfg.step_polar)
        job.process_brdf(do_spectral_averaging=True)
        
        if job.is_pixel_request():
            
            print 'pixel'
            job.set_status('call polar pixel graphs')
            
            job.graph_polar_plot(case='pixels', indices=job.data['idx_land'], title='Land Polar Graph', three_d=False)
            job.graph_polar_plot(case='pixels', indices=job.data['idx_land'], title='Land Graph 3D', three_d=True)

            job.graph_polar_plot(case='pixels', indices=job.data['idx_ocean'], title='Ocean Polar Graph', three_d=False)
            job.graph_polar_plot(case='pixels', indices=job.data['idx_ocean'], title='Ocean Graph 3D', three_d=True)

            job.graph_polar_plot(case='pixels', indices=job.data['idx_snow'], title='Snow Polar Graph', three_d=False)
            job.graph_polar_plot(case='pixels', indices=job.data['idx_snow'], title='Snow Graph 3D', three_d=True)
            
        else:

            job.set_status('Perform BRDF stats analysis')
            job.calculate_stats(job.data['BRDF'])
            
            job.set_status('call polar stats graphs')
            job.graph_polar_plot(case='stats', title='Polar Graph', three_d=False)
            job.graph_polar_plot(case='stats', title='3D Graph', three_d=True)

            
    ##############
    # principal plane            
    else:

        job.set_status('define_vza_hotspot')
        job.define_vza_hotspot()
    
        job.set_status('calculate brdf and brdf error')
        job.process_brdf(do_spectral_averaging=True)
            
        if job.is_pixel_request():
            
            job.set_status('draw graph, principal plane, case pixels')
            
            job.graph_principal_plane(case='pixels', indices=job.data['idx_land'],  title='BRDF Land pixels', graph_error=True)
            job.graph_principal_plane(case='pixels', indices=job.data['idx_ocean'], title='BRDF Ocean pixels')
            job.graph_principal_plane(case='pixels', indices=job.data['idx_snow'],  title='BRDF Snow pixels')   

        else:

            job.set_status('Perform BRDF stats analysis')
            job.calculate_stats(job.data['BRDF'])
            
            job.set_status('draw graph, principal plane, case stats')
            job.graph_principal_plane(case = 'stats')

    
    
    # finished either pixel or stats job
    job.set_status('saving netcdf data')
    job.save_netcdf()
    
    job.end_time = time.time()
    job.set_status('job finished, took %s seconds' % str(job.end_time - job.start_time))
    
    
    return json.dumps({'success': 'true',  'log': job.running_status, 'outputs': job.outputs}, indent=4)
        
        

        
