###############################################################
# Definition of the default configuration parameters
#
# Authors:
#  NOVELTIS: Cedric Bacour / Ivan Price
###############################################################


import os
import numpy as np
import netCDF4
import socket
import logging
import process_brdf


class adam_config():
    ''' class to be used as a singleton by all adam modules for storing config info for the adam system '''

    def __init__(self, path_data=None, output_root_dir=None, debug_log_file=None, http_root=None):

        ##############################################################################################
        # -
        # - Input/Output directories & files
        # -

        # these are normally supplied as parameters at the top of the script, as given in the examples:
        # my_config = adam_config.adam_config(path_data = my_path_data, output_root_dir = my_output_root_dir)
        # job = adam.adam_job( cfg=my_config )

        # the root directory to the data files
        self.path_data = path_data
        # the root directory into which job output directories will be created
        self.output_root_dir = output_root_dir

        # debug info will be written to this file if it is defined, used primarily by the website
        self.debug_log_file = debug_log_file

        # used by the website
        self.http_root = http_root



         
        ##############################################################################################



        # these are here to allow default values when no directories are supplied, primarily for
        # development / debugging by NOVELTIS.
        # adding your own value here means you don't need to explicity set the location of
        # your data in your scripts, as long as your hostname matches a key here

        config_dict = {'pc-price': {'path_data'       : 'p://3895-ADAM/code/data/',
                                    'output_root_dir' : 'Y://web-root/adam/web-output/',
                                    'debug_log_file'  : 'Y://wsgi-apps/adam/wsgi-log.log',
                                    'http_root'       : 'http://price.ws.noveltis.dev/adam/web-output/'
                                   },
                      'webdev':    {'path_data'       : '/PROJETS/3895-ADAM/code/data/',
                                    'output_root_dir' : '/NOVELTIS/price/workspace/web-root/adam/web-output/',
                                    'debug_log_file'  : '/NOVELTIS/price/workspace/wsgi-apps/adam/wsgi-log.log',
                                    'http_root'       : 'http://price.ws.noveltis.dev/adam/web-output/'
                                    },
                      'prod1':     {'path_data'       : '/PROJETS/3895-ADAM/code/data/',
                                    'output_root_dir' : '/home/www/data/adam/web-root/web-output/',
                                    'debug_log_file'  : '/home/www/data/adam/wsgi-apps/wsgi-log.log',
                                    'http_root'       : 'http://adam.noveltis.fr/web-output/'
                                   },
                      'adam-srv':     {'path_data'       : '/PROJETS/3895-ADAM/code/data/',
                                    'output_root_dir' : '/home/www/data/adam/web-root/web-output/',
                                    'debug_log_file'  : '/home/www/data/adam/wsgi-apps/wsgi-log.log',
                                    'http_root'       : 'http://adam.noveltis.fr/web-output/'
                                   },
                      'cedric':    {'path_data'       : '/home/satellites9/cbacour/ADAM/Data/V1/',
                                    'output_root_dir' : '/home/satellites9/cbacour/ADAM/OUTPUT/',
                                    'debug_log_file'  : '/home/satellites9/cbacour/ADAM/OUTPUT/logfile.txt',
                                    'http_root'       : '/home/satellites9/cbacour/ADAM/OUTPUT/'
                                   }
                       }


        if config_dict.has_key(socket.gethostname()):
            if path_data is None        : self.path_data       = config_dict[socket.gethostname()]['path_data']
            if output_root_dir is None  : self.output_root_dir = config_dict[socket.gethostname()]['output_root_dir']
            if debug_log_file is None   : self.debug_log_file  = config_dict[socket.gethostname()]['debug_log_file']
            if http_root is None        : self.http_root       = config_dict[socket.gethostname()]['http_root']



        self.netcdf_datafile = 'ADAM_V2_PlateCarree.nc'
        self.netcdf_file_covarland_template = 'ADAM_V2_ERR_PlateCarree_M%02i.nc'
        self.path_ascii = os.path.join(self.path_data, 'ascii')                     # path where the ascii files are saved
        self.path_maps = os.path.join(self.path_data,'maps_raw')                    # output path for saving maps

        # - Land
        # refractive index from http://refractiveindex.info/?group=CRYSTALS&material=H2O-ice,
        #  Warren S.G (1984, Optical constants of ice from the ultraviolet to the microwave, Applied Optics, 23:1206-1225)
        self.file_land_IceRefInd = 'ADAM_IceComplexRefractiveIndex_300-4000.txt'
        # - Ocean
        self.file_ocean_ref_std = 'ADAM_OceanTypicalReflectance_300-800.txt'
        # refractive index from http://refractiveindex.info/?group=LIQUIDS&material=Water
        #  G. M. Hale and M. R. Querry. Optical Constants of Water in the 200-nm to 200-microm Wavelength Region, Appl. Opt. 12, 555-563 (1973) doi:10.1364/AO.12.000555
        self.file_ocean_WaterRefInd = 'ADAM_WaterRefractiveIndex_300-4000.txt'
        # absorption coefficient of water
        # from  D. J. Segelstein, "The complex refractive index of water," University of Missouri-Kansas City, (1981). (http://omlc.ogi.edu/spectra/water/data/segelstein81.dat)
        self.file_ocean_WaterAbs = 'ADAM_WaterAbsorption_300-4000.txt'


        self.netcdf_datafile = os.path.join(self.path_data, self.netcdf_datafile)
        self.netcdf_file_covarland_template = os.path.join(self.path_data, self.netcdf_file_covarland_template)

        logging.basicConfig(filename=self.debug_log_file, level=logging.DEBUG, format='%(asctime)s %(levelname)s %(message)s')


    # -
    # - Global parameters
    # -

        # limits of reflectance for the various wavelengths used to define the colour scales
        # for maps generated for the ADAM website.
        # the spectral range is the key, from which reasonable max ranges can be had for global maps.
        # the incrementing numbers (0->11) are the indexes used by the make_maps scripts, and
        # correspond to the names of the images used by the website.
        
        self.reflectance_limits = {
            # averaged 0
            (300, 400):   {'land':  {'min': 0, 'max': 1.02},
                           'ocean': {'min': 0, 'max': 0.10442} },
            # averaged 1
            (400, 700):   {'land':  {'min': 0, 'max': 1.02 },
                           'ocean': {'min': 0, 'max': 0.0262 } },
            # averaged 2
            (700, 1300):  {'land':  {'min': 0, 'max': 0.928},
                           'ocean': {'min': 0, 'max': 0.00044} },
            # averaged 3
            (1300, 2500): {'land':  {'min': 0, 'max': 0.69},
                           'ocean': {'min': 0, 'max': 0.0001} },
            # averaged 4
            (2500, 4000): {'land':  {'min': 0, 'max': 0.55},
                           'ocean': {'min': 0, 'max': 0.0001} },
            # MODIS iii ## 5
            (469,469):    {'land':  {'min': 0, 'max': 1.027 },
                           'ocean': {'min': 0, 'max': 0.0512 } },
            # MODIS iv  ## 6
            (555,555):    {'land':  {'min': 0, 'max': 1.023},
                           'ocean': {'min': 0, 'max': 0.0222} },
            # MODIS i   ## 7
            (645,645):    {'land':  {'min': 0, 'max': 1.01 },
                           'ocean': {'min': 0, 'max': 0.0082 } },
            # MODIS ii  ## 8
            (858,858):    {'land':  {'min': 0, 'max': 0.947},
                           'ocean': {'min': 0, 'max': 0.0001} },
            # MODIS v   ## 9
            (1240,1240):  {'land':  {'min': 0, 'max': 0.621 },
                          'ocean':  {'min': 0, 'max': 0.0001 } },
            # MODIS vi  ## 10
            (1640,1640):  {'land':  {'min': 0, 'max': 0.511},
                           'ocean': {'min': 0, 'max': 0.0001} },
            # MODIS vii ## 11
            (2130,2130):  {'land':  {'min': 0, 'max': 0.493 },
                           'ocean': {'min': 0, 'max': 0.0001 } }
        }
         
        # arrays of the wavebands, each element either length=1 for narrow band 
        # or length=2 to use spectral averaging.
        self.instrument_wavebands = {
            'modis'           : [469, 555, 645, 858, 1240, 1640, 2130],
            'vgt'             : [450, 645, 840, 1665],
            'avhrr'           : [635, 862, 1610, 3740],
            'meris'           : [413, 443, 490, 510, 560, 620, 665, 681, 705, 754, 760, 775, 865, 885, 900],
            'misr'            : [446, 558, 672, 867],
            'seawifs  '       : [412, 443, 490, 510, 555, 670, 765, 865],
            'polder_parasol'  : [445, 492, 565, 670, 763, 763, 861, 908],
            'msg_seviri'      : [600, 800, 1600, 3900],
            'sentinel3_olci'  : [413, 443, 490, 510, 555, 560, 620, 659, 665, 682, 709, 754, 761, 779, 865, 885, 900, 940, 1020, 1376, 1610, 2250, 3740],
            'omi'             : [309, 312, 320, 328, 335, 343, 345, 354, 367, 373, 376, 380, 388, 406, 416, 418, 425, 440, 442, 452, 463, 471, 477, 488, 494, 499],
            'orbview_2'       : [412, 443, 490, 510, 555, 670, 765, 865],
            'sentinel2'       : [443, 490, 560, 665, 705, 741, 783, 842, 865, 945, 1375, 1610, 2190],
            'aster'           : [560, 660, 810, 1650, 2165, 2205, 2260, 2330, 2395],
            'spot_5'          : [545, 645, 835, 1665],
            'spot_67'         : [490, 560, 660, 825],
            'landsat_7'       : [485, 560, 660, 835, 1650, 2220],
            'landsat_8'       : [443, 482, 562, 655, 865, 1375, 1610, 2200]
        }
        
        # the gome instrument is every integer value between (and including) 240 and 790
        # actually, adam has a minimum of 300 so we set to 300 here
        self.instrument_wavebands['gome'] = range(300,791) 
        

        # the maximum allowable size in degrees for a (non-download) request (20x20)
        self.limit_square_degrees = 410

        # maximum limit to perform pixel-level calculations
        self.limit_degrees_pixel_calcs = 0.1

        # -
        # - Names of the variables to read in the gridded ADAM NetCDF file
        # -
        self.vars_land = ['ref_land']                       # respect this order!
        self.vars_ocean = ['wind_speed','chloro_conc']      # respect this order!


        # - standard viewing geometry (in degrees)
        self.sza_std = 45
        self.vza_std = 0
        self.phi_std = 0


        # maximum value authorized
        self.sza_max_limit = 70
        self.vza_max_limit = 70
        # behaviour of the BRDF models when sza of vza are beyond the max limit values:
        #  - '0': set the corresponding BRDF values to 0
        #  - 'nan': set the corresponding BRDF values to NaN
        #  - 'constant': set the corresponding BRDF values to the values it would have for sza/vza = max limit
        self.beyond_za_limit = 'constant'


        vza_range  = [0,self.vza_max_limit]
        vza_step   = 5
        self.vza_values = np.arange(-vza_range[1], vza_range[1]+5, vza_step)
        self.phi_values = np.arange(0,360+10,10)
        self.step_polar = 41 # discretisation step for drawing polar plots

        # standard BRDF F1 & F2 functions over land (in the standard observation geometry)
        self.f1_std = process_brdf.brdf_land_f1(self.sza_std,self.vza_std,self.vza_std)
        self.f2_std = process_brdf.brdf_land_f2(self.sza_std,self.vza_std,self.vza_std)


        # missing values
        self.missval_1b = 255
        self.missval_2b = -9999.

        self.lmbd = np.arange(4000-300+1)+300
        self.spectral_domains = [[300,400],[400,700],[700,1300],[1300,2500],[2500,4000]]

        # maximum number of pixels to display individual data. Beyond mx_pts_analysis, the statistics are drawn
        self.max_pts_analysis = 6

        # parameters for the RGB composite image
        self.lmbd_R = 670
        self.lmbd_G = 460
        self.lmbd_B = 560
        self.ilmbd_R = np.where(self.lmbd == self.lmbd_R)[0][0]
        self.ilmbd_G = np.where(self.lmbd == self.lmbd_G)[0][0]
        self.ilmbd_B = np.where(self.lmbd == self.lmbd_B)[0][0]


        try:
            source_dataset = netCDF4.Dataset(self.netcdf_datafile, 'r')
        except:
            raise Exception('Error opening main source data file, location given was: %s' % self.netcdf_datafile)


        # load the modis wavebands
        self.modis_wavebands = np.round(source_dataset.variables['wavebands'][:])
        source_dataset.close()
        self.indices_modis_bands = []
        for wl in self.modis_wavebands:
            ind = np.ma.nonzero(self.lmbd == wl )[0]
            self.indices_modis_bands.extend(ind.tolist())


        self.land , self.ocean = self._get_land_ocean_configs()


    def _get_land_ocean_configs(self):

        # -
        # - Land
        # -
        file_acp =  os.path.join(self.path_data,'ADAM_V1_ACP.nc')

        try:
            dataACP = netCDF4.Dataset(file_acp, 'r')
        except:
            raise Exception('Error opening ACP data file, location given was: %s' % file_acp)

        kice_data = np.loadtxt(os.path.join(self.path_ascii, self.file_land_IceRefInd), comments='#', skiprows=0, unpack=True)

        config_land = AttributeDict({
            'missval'             : -8421,
            'add_offset_land'     : 0.,
            'scale_factor_land'   : 0.0001,

            # - Spectrum parameters (ACP)
            'file_acp'            : file_acp,
            'ACP_ObsMean'         : dataACP.variables['ObsMean'][:],
            'ACP_SpecMean'        : dataACP.variables['SpecMean'][:],
            'ACP_prod'            : dataACP.variables['prodACP'][:],
            'ACP_lmbd'            : dataACP.variables['wavelength'][:],
            'ACP_neigenvectors'   : dataACP.variables['neigenvectors'][:],


            # - BRDF parameters
              # vis / nir
            'slope_R'                 : np.array([0.2, -0.05]),
            'slope_V'                 : np.array([1, 2]),
            'intercept_R'             : np.array([0.1, 0.15]),
            'intercept_V'             : np.array([0.5, 0.5]),
            'alpha_RV'                : 0.5,
            'err_R'                   : 0.05,
            'err_V'                   : 0.15,

            # - snow
            'snow_grain_size'         : 0.12e-3,
            'chanB'                   : 0,
            'chanR'                   : 2,
            'chanPIR'                 : 3,
            'chanPIR2'                : 4,
            'cond_snow_1'             : 0.4,
            'cond_snow_2'             : 0.,

            # - ice complex refractive index

            'lmbd_kice'               : kice_data[0,:],
            'kice'                    : kice_data[1,:]
        })


        # this consumes ~5 seconds !
        config_land['ACP_eigenvectors'] = dataACP.variables['Eigenvectors'][:, config_land['ACP_neigenvectors']]
        config_land['ACP_mat'] = np.dot(config_land['ACP_eigenvectors'], config_land['ACP_prod'])

        dataACP.close()



    # -
    # - Ocean
    # -
        nwater_data = np.loadtxt(os.path.join(self.path_ascii, self.file_ocean_WaterRefInd), comments='#', skiprows=0, unpack=True)

        config_ocean = AttributeDict({
            'missval'           : -32768,
            'add_offset_ws'     : 15.97722,
            'scale_factor_ws'   : 0.0004026239,
            'add_offset_chl'    : -0.1103447,
            'scale_factor_chl'  : 5.676205e-05,

            # - Water refractive Index
            'lmbd_nwater'       : nwater_data[0,:],
            'nwater'            : nwater_data[1,:]
        })

        # - typical ocean reflectance spectra wrt chlorophyll content
        fic = open(os.path.join(self.path_ascii, self.file_ocean_ref_std), mode='rU')
        lines = fic.readlines()
        fic.close()

        nl = len(lines)
        nlmbd  = 501
        nchl = 6
        ref_ocean_chl_std = np.zeros((nchl, nlmbd), np.float)
        lmbd_ocean_chl_std = np.zeros(nlmbd, np.float)
        buf = (lines[0].strip()).split()
        buf = [float(el) for el in buf[1:]]

        chl_ocean = np.array(buf[:])

        for i in range(nl):
            if i == 0: continue
            buf = (lines[i].strip()).split()
            lmbd_ocean_chl_std[i-1] = float(buf[0])
            buf = [float(el) for el in buf[1:]]
            ref_ocean_chl_std[:,i-1] = np.array(buf)

        config_ocean['chl'] = chl_ocean
        config_ocean['ref_chl_std'] = ref_ocean_chl_std
        config_ocean['lmbd_chl_std'] = lmbd_ocean_chl_std

        # - foam reflectance as a function of foam (from Koepke, 1984)
        data = np.array([[4,  0.00],[5,  0.00],[6,  0.00],[7,  0.00], [8,  0.10],[9,  0.20],\
                         [10, 0.20],[11, 0.30],[12, 0.40],[13, 0.50],[14, 0.70],[15, 0.90],\
                         [16, 1.10],[17, 1.30],[18, 1.60],[19, 2.00],[20, 2.30],[25, 5.10]])
        ws_0 = data[:,0].ravel()
        ref_0 = (data[:,1].ravel())/100.

        # - interpolation for wind speed between [0 ; 30] m.s-1
        ws_foam_std = np.arange(31)
        ref_foam_std = np.interp(ws_foam_std, ws_0, ref_0)
        config_ocean['foam_ws_std'] = ws_foam_std
        config_ocean['foam_ref_std'] = ref_foam_std

        # - Spectral variation of the foam contribution
        # (see Kokhanovsky A.A. (2004), Spectral reflectance of whitecaps, Journal of Geophysical Research, 109, C05021, doi:10.1029/2003JC002177)
        if not os.path.exists(os.path.join(self.path_ascii, self.file_ocean_WaterAbs)):
            raise Exception('Error finding foam datafile at: %s !' % os.path.join(self.path_ascii, self.file_ocean_WaterAbs))
        data = np.loadtxt(os.path.join(self.path_ascii, self.file_ocean_WaterAbs), comments='#', skiprows=0, unpack=True)
        abs_water_foam = data[1,:]
        b_foam = 1.72*1e-3
        c_foam = 0.00027*1e6
        config_ocean['foam_specmod'] = np.exp(-np.sqrt((abs_water_foam+c_foam)*b_foam))
        ilmbd_VIS = np.where(self.lmbd == 400)[0][0]
        config_ocean['foam_normfac'] = config_ocean['foam_specmod'][ilmbd_VIS]


    # -
    # - Return
    # -
        return config_land, config_ocean


class AttributeDict(dict):
    ''' simple override class to provide a dictionary that allows for access of its elements
        like attributes.. for example: dict.value = 'xxxx' instead of dict['value'] = 'xxxx'

        ATTENTION because we cannot expect dict['my key'] to work like dict.my key
        this is here because it keeps the code a bit cleaner and we are aware of this danger
    '''

    def __getattr__(self, attr):
        return self[attr]

    def __setattr__(self, attr, value):
        self[attr] = value

