

import numpy as np
import sys
import os
import time
from datetime import datetime
import tempfile
from multiprocessing import Pool
import multiprocessing
import traceback
import errno


# this is only required to calculate the percentile values of the maxes..
# if scipy is not available and your numpy is recent you can alternatively use numpy.percentile
from scipy import stats



# if this script is to be run as a user without a home directory the following is necessary
home_dir = os.getenv('USERPROFILE') or os.getenv('HOME')
if not os.path.exists(home_dir):
    os.environ['MPLCONFIGDIR'] = tempfile.gettempdir()
    

# this script may live in the test/ directory and the adam class my not be in the PATH,
# so we add this here to allow for that case
if not '../' in sys.path:
    sys.path.insert(0, '../')

# import the main adam library
import adam
import adam_config as adam_config

#my_path_data       = 'p://3895-ADAM/code/data/'
#my_output_root_dir = 'Y://web-root/adam/web-output/'

# get an instance of an adam 'config' object, this will be attached to any jobs we run
#my_config = adam_config.adam_config(path_data = my_path_data, output_root_dir = my_output_root_dir)
my_config = adam_config.adam_config()

my_config.path_maps = 'e://maps/'
NUM_PROCESSES = 2
LONS = np.array([1, 2])
LATS =  np.array([38, 39])
TILE_WIDTH_DEGREES = 2

my_config.path_maps = '/data_no_saved/adam/maps/'
NUM_PROCESSES = 15
LONS = np.linspace(-180,170,36)
LATS = np.linspace(-90,80,18)
TILE_WIDTH_DEGREES = 10

MAP_ROOT = my_config.path_maps

MONTHS = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']




SPECTRAL_DOMAINS = [[300, 400], [400, 700], [700, 1300], [1300, 2500], [2500, 4000],
                    [469,469], [555,555], [645,645], [858,858], [1240,1240], [1640,1640], [2130, 2130]]


def main():
    ''' master function that builds the array of job definitions and manages the job queue '''
    
    
    pool = Pool(processes=NUM_PROCESSES)
    
    request_dicts = []
    for month in MONTHS:

        # loop right across all the left-side corners of map output tiles
        for minx in LONS:
            # and loop up for all the bottom corners of map output tiles
            for miny in LATS:
                # find the upper left corner by adding 10 degrees to the bottom left
                maxx = minx + TILE_WIDTH_DEGREES
                maxy = miny + TILE_WIDTH_DEGREES
                
                # now we can build our request dictionary based on the dimensions of the tile
                request_dict = {'fieldOperationType': 'brdf',
                                'fieldMonth': month,
                                'fieldSunZenith': 45,
                                'fieldRelAzimuth': 0,
                                'fieldViewZenith': 0,
                                'fieldBRDFType': 'plane',
                                'fieldLonMin': minx,
                                'fieldLatMin': miny,
                                'fieldLonMax': maxx,
                                'fieldLatMax': maxy
                                }
    
                request_dicts.append(request_dict)


    print 'Launching %s jobs now..' % len(request_dicts)
    

    # launch the pool with the array of job definitions
    start_time = time.time()
    results = pool.map_async(run_job, request_dicts, chunksize=1)
    # indicate we have finished adding data to the queue
    pool.close()
    
    # now wait for all jobs to terminate
    while (True):
        remaining =  results._number_left
        print 'Jobs remaining: %s' % remaining
        if results.ready(): break
        time.sleep(5)
        
        

    # all jobs are done, gather the results
    pool.join()
    results = results.get()
    
    print '\n\nfinished all tasks in %s seconds\n\n' % (time.time() - start_time)    
    
    # we want to do some stats on the results, so we convert them to a numpy array,
    # the dimensions will be job, spectrum, land/ocean
    results_matrix = np.array(results)
    
    for i in range(len(SPECTRAL_DOMAINS)):
        
        land_maxes  = results_matrix[:,i,0]
        ocean_maxes = results_matrix[:,i,1]
        
        print 'spec: %s land max: %s' % (i, land_maxes.max())
        print 'spec: %s land 95 percentile is: %s'  % (i, stats.scoreatpercentile(land_maxes, 95) )
        print 'spec: %s land 98 percentile is: %s'  % (i, stats.scoreatpercentile(land_maxes, 98) )
        print 'spec: %s ocean min: %s' % (i, ocean_maxes.min())
        print 'spec: %s ocean max: %s' % (i, ocean_maxes.max())
        print 'spec: %s ocean 95 percentile is: %s' % (i, stats.scoreatpercentile(ocean_maxes, 95) )
        print 'spec: %s ocean 98 percentile is: %s' % (i, stats.scoreatpercentile(ocean_maxes, 98) )

    return results_matrix
    


                                         
def run_job(request_dict):

    # initialise this job
    job = adam.adam_job( cfg=my_config, create_working_dir=False )
    job.validate_input(request_dict)
    
    # set spectral domains
    job.spectral_domains = SPECTRAL_DOMAINS

    job.lmbd = np.arange(300,4001)
    job.nlmbd = len(job.lmbd)
    
    # load the data
    job.load_data()
    
    job.process_reflectance(do_spectral_averaging=True)

    #job.process_brdf(do_spectral_averaging=True)
    
    maxes = []
    for i in range(len(job.spectral_domains)):

        
        output_dir = '%s/%s/%s' % (MAP_ROOT, request_dict['fieldMonth'], i)

        # we will try to make the directory, if it already exists we will continue
        # silently, if there was any other form of problem then throw an exception.
        try:
            os.makedirs(output_dir)
        except OSError as exception:
            if exception.errno != errno.EEXIST:
                print 'ERROR CREATING DIRECTORY: %s' % output_dir
                raise
        
        
        try:
            # creates 3-channel pngs
            #job.map_ref_averaged_spectra(spectral_domain_index=i, width=TILE_WIDTH_DEGREES * 10, output_dir=output_dir)
            # creates 1-channel tifs
            job.map_raw_values(spectral_domain_index=i, width=TILE_WIDTH_DEGREES * 10, output_dir=output_dir)
        except:
            # if there is an error anywhere in the processing, OR we receive a ctrl-c break
            # print the error and allow the child to die
            trace_back = traceback.format_exception(sys.exc_info()[0], sys.exc_info()[1], sys.exc_info()[2])
            trace_back = '\n'.join(trace_back)
            print 'ERROR captured in child process, message follows:'
            print trace_back
            sys.exit()
            
        # creates single-channel tiffs
        #job.map_raw_values(spectral_domain_index=i, width=100, output_dir='%s/%s' % (output_dir, i))
        

        job.data['reflectance_averaged'] = np.ma.masked_equal(job.data['reflectance_averaged'], -9999)
        land_max = 0
        ocean_max = 0
        
        # snow and land are considered together
        land_indexes = np.concatenate((job.data['idx_land'], job.data['idx_snow']))
        if len(land_indexes) > 0:
            land_max =   job.data['reflectance_averaged'][:,:,i].flatten()[land_indexes].max()
            
        if len(job.data['idx_ocean']) > 0:
            ocean_max =  job.data['reflectance_averaged'][:,:,i].flatten()[job.data['idx_ocean']].max()
        
        maxes.append((land_max, ocean_max))
        
    return maxes
    

        

if __name__ == "__main__":

    results_matrix = main()
    
    
    print 'finished at: %s' % datetime.now()
    print '''
    maybe you want to run:
    
    # create a single tif from a collection of pngs:
    gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 *.png 0.tif
    
    # create several tiffs in parallel for the 5 directories:
    gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 0/*.png 0.tif &
    gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 1/*.png 1.tif &
    gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 2/*.png 2.tif &
    gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 3/*.png 3.tif &
    gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 4/*.png 4.tif &
    
    # same, if the output was tifs:
    gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 0/*.tif 0.tif &
    gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 1/*.tif 1.tif &
    gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 2/*.tif 2.tif &
    gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 3/*.tif 3.tif &
    gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 4/*.tif 4.tif &
    gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 5/*.tif 5.tif &
    gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 6/*.tif 6.tif &
    
    
    # the result needs to be manually set to global extents
    gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 0.tif adam-0.tif &
    gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 1.tif adam-1.tif &
    gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 2.tif adam-2.tif &
    gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 3.tif adam-3.tif &
    gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 4.tif adam-4.tif &
    gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 5.tif adam-5.tif &
    gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 6.tif adam-6.tif &
    
    
    # add averaging to speed up access
    gdaladdo -r average adam-0.tif 2 4 8 16 &
    gdaladdo -r average adam-1.tif 2 4 8 16 &
    gdaladdo -r average adam-2.tif 2 4 8 16 &
    gdaladdo -r average adam-3.tif 2 4 8 16 &
    gdaladdo -r average adam-4.tif 2 4 8 16 &
    
    # for merging multiple bands to a single (tif) image:
    gdal_merge.py -of GTIFF -separate -n 0 -o merged.tif *.tif 


 
 
    '''






#a gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 0/*.png 00.tif &
#a gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 1/*.png 01.tif &
#a gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 2/*.png 02.tif &
#a gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 3/*.png 03.tif &
#a gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 4/*.png 04.tif &
#a gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 5/*.png 05.tif &
#a gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 6/*.png 06.tif &
#a gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 7/*.png 07.tif &
#a gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 8/*.png 08.tif &
#a gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 9/*.png 09.tif &
#a gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 10/*.png 10.tif &
#a gdalwarp -s_srs EPSG:4326 -t_srs EPSG:4326 11/*.png 11.tif &
#
#a gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 00.tif adam-00.tif &
#a gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 01.tif adam-01.tif &
#a gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 02.tif adam-02.tif &
#a gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 03.tif adam-03.tif &
#a gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 04.tif adam-04.tif &
#a gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 05.tif adam-05.tif &
#a gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 06.tif adam-06.tif &
#a gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 07.tif adam-07.tif &
#a gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 08.tif adam-08.tif &
#a gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 09.tif adam-09.tif &
#a gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 10.tif adam-10.tif &
#a gdal_translate -a_srs EPSG:4326 -a_ullr -180 90 180 -90 11.tif adam-11.tif &
#
#a gdaladdo -r average adam-00.tif 2 4 8 16 &
#a gdaladdo -r average adam-01.tif 2 4 8 16 &
#a gdaladdo -r average adam-02.tif 2 4 8 16 &
#a gdaladdo -r average adam-03.tif 2 4 8 16 &
#a gdaladdo -r average adam-04.tif 2 4 8 16 &
#a gdaladdo -r average adam-05.tif 2 4 8 16 &
#a gdaladdo -r average adam-06.tif 2 4 8 16 &
#a gdaladdo -r average adam-07.tif 2 4 8 16 &
#a gdaladdo -r average adam-08.tif 2 4 8 16 &
#a gdaladdo -r average adam-09.tif 2 4 8 16 &
#a gdaladdo -r average adam-10.tif 2 4 8 16 &
#a gdaladdo -r average adam-11.tif 2 4 8 16 &



