# -*- coding: utf-8 -*-
###############################################################
# Authors:
#  NOVELTIS: Cedric Bacour / Ivan Price
###############################################################

"""

WSGI application executed by a web server to run jobs for the ADAM project

"""
import sys
import numpy as np
import cgi
import traceback
import json
from socket import gethostname
import time
import os
import smtplib
from email.mime.text import MIMEText
import pytz
from datetime import datetime
import urllib, urllib2
from PIL import Image, ImageDraw, ImageFont
from cStringIO import StringIO
from osgeo import gdal
from PIL import Image
import numpy as np
from cStringIO import StringIO
import tempfile

DATA_LICENCE_EMAIL_FROM = 'ivan.price@noveltis.fr'
DATA_LICENCE_EMAIL_TO   = ['Ivan.Price@noveltis.fr', 'Pascal.Prunet@noveltis.fr' ,'Cedric.Bacour@noveltis.fr']


# this is a signal to the adam system to indicate it is being run for the web
os.environ['ADAM_WEB_REQUEST'] = 'true'


# need to tell the web server to look in our wsgi scripts path to find the modules we will be requesting
# we insert at position 0 to exert precedence over existing python modules, which is why we
# can't use site.addsitedir()


# the web server needs to add these to his path as he does not run the code from the current directory
for string in ['/NOVELTIS/price/workspace/wsgi-apps/adam/','/home/www/data/adam/wsgi-apps/']:
    if not string in sys.path:
        sys.path.insert(0, string)




import adam
import adam_config



# get an instance of an adam 'config' object, this will be attached to any jobs we run
# we supply no path details here as the adam_config file knows us and will build the correct values
# based on the hostname of the machine running the code
my_config = adam_config.adam_config()


def application(environ, start_response):

    def httpResponse(response_status,response_body, content_type='text/html'):
        response_headers = [('Content-Type', content_type), ('Content-Length', str(len(response_body))) ]
        start_response(response_status, response_headers)
        return [response_body]

    response_status = '200 OK'

    d = cgi.parse_qs(environ['QUERY_STRING'])
    request_dict = {}
    
    
    

    for key in d.keys():
        request_dict[key] = cgi.escape( d.get(key, [''])[0] )#.lower()
    if request_dict.has_key('_dc'): del(request_dict['_dc'])
    
    operation_type = request_dict.get('fieldOperationType', '').lower().strip()

    # for anything other than a map request we force all input parameters into lowercase    
    if operation_type != 'map':
        for key in request_dict.keys():
            request_dict[key] = request_dict[key].lower()


    content_type = 'text/html'
    ##################################
    if operation_type == 'brdf-time':
        try:
            htmlResult = performBRDFTime(request_dict)
        except:
            trace_back = traceback.format_exception(sys.exc_info()[0], sys.exc_info()[1], sys.exc_info()[2])
            response_status = '514 BAD BRDF Time'
            htmlResult = json.dumps({'success': 'false',  'message': '\n'.join(trace_back)}, indent=4)

    ##################################
    elif operation_type == 'brdf':
        try:
            htmlResult = performBRDF(request_dict)
        except:
            trace_back = traceback.format_exception(sys.exc_info()[0], sys.exc_info()[1], sys.exc_info()[2])
            response_status = '514 BAD BRDF'
            htmlResult = json.dumps({'success': 'false',  'message': '\n'.join(trace_back)}, indent=4)

    ###################################
    elif operation_type == 'spectrum':
        try:
            htmlResult = performSpectrum(request_dict)
        except:
            trace_back = traceback.format_exception(sys.exc_info()[0], sys.exc_info()[1], sys.exc_info()[2])
            response_status = '514 BAD SPECTRUM'
            htmlResult = json.dumps({'success': 'false',  'message': '\n'.join(trace_back)}, indent=4)

    ##################################
    elif operation_type == 'download':
        try:
            htmlResult = performDownload(request_dict)
        except:
            trace_back = traceback.format_exception(sys.exc_info()[0], sys.exc_info()[1], sys.exc_info()[2])
            response_status = '514 BAD DOWNLOAD'
            htmlResult = json.dumps({'success': 'false',  'message': '\n'.join(trace_back)}, indent=4)

    ##################################
    elif operation_type == 'email':
        try:
            htmlResult = performEmail(request_dict)
        except:
            raise
            trace_back = traceback.format_exception(sys.exc_info()[0], sys.exc_info()[1], sys.exc_info()[2])
            response_status = '514 BAD EMAIL'
            htmlResult = json.dumps({'success': 'false',  'message': '\n'.join(trace_back)}, indent=4)

    ##################################
    elif operation_type == 'map':
        try:
            htmlResult, content_type = performMap(request_dict)
        except:
            raise
            trace_back = traceback.format_exception(sys.exc_info()[0], sys.exc_info()[1], sys.exc_info()[2])
            response_status = '514 BAD MAP'
            htmlResult = json.dumps({'success': 'false',  'message': '\n'.join(trace_back)}, indent=4)
            
            
    ##################################

    else:
        response_status = '515 BAD OPERATION'
        htmlResult = json.dumps({'success': 'false',  'message': 'no operation type found !: %s' % operation_type}, indent=4)



    response_body = htmlResult
    return httpResponse(response_status, response_body, content_type=content_type)

def performMap(request_dict):
    ''' function to act as a proxy between the user and geoserver, stamping
    the image credits text onto the image.
    
    made complicated because we need to use gdal to modify the geotiff in order to preserve the 
    georeferencing.
    '''
    
    url = 'http://localhost:8080/geoserver/wms'
    
    # firstly fix the stle parameter that has had < and > symbols translated
    for key in request_dict.keys():
        request_dict[key] = request_dict[key].replace('&lt;', '<')
        request_dict[key] = request_dict[key].replace('&gt;', '>')

    # re-make the same request, this time to geoserver
    data = urllib.urlencode(request_dict)
    req = urllib2.Request(url, data)
    response = urllib2.urlopen(req)
    

    
    tfile = tempfile.NamedTemporaryFile(delete=False)
    tfile.write(response.read())
    tfile.close()
    

    # unfortunately we can't use this feature coz the version of gdal on the server !> 1.8
    ## read response into a file-like object
    ##image_data = StringIO(response.read())
    ##gdal.FileFromMemBuffer('/vsimem/tiffinmem', image_data)
    ##src_ds = gdal.Open('/vsimem/tiffinmem') 
    # so instead we have to write temp files :(
    src_ds = gdal.Open(tfile.name) 
    num_bands = src_ds.RasterCount
    
    gdal_array = src_ds.ReadAsArray()
    pil_array = np.rollaxis(gdal_array,0,3)
    
    pil_image = Image.fromarray(pil_array)

    # now we have the geoserver image both as a gdal dataset and as a PIL image,
    # we can now use PIL to stamp our image attribution
    ################################################
    image_width, image_height = pil_image.size

    # get an object with which to modify the image
    draw = ImageDraw.Draw(pil_image)

    # draw the rectangle on the image
    x = image_width-140
    y = image_height-20
    draw.rectangle([(x-10, y-10),(image_width, image_height)], fill=(255,255,255))

    # find our font file relative to this script
    path = os.path.dirname(os.path.abspath(__file__))
    font = ImageFont.truetype('%s/FreeSans.ttf' % path, 12, encoding='utf-8')
    
    # draw the text on the image, with the text centered horizontally and at the bottom vertically
    draw.text((x, y), 'http://adam.noveltis.com/', fill="black", font=font)
    
    pil_array = np.asarray(pil_image)
    
    os.remove(tfile.name)
    
    # now we have an array representing the image data with the attribution
    ################################################

    # if the image is to be a geotiff, we need to create a new geotiff with the 
    # modified data
    content_type = 'image/png'
    if 'tiff' in src_ds.GetDriver().ShortName.lower():
        content_type = 'image/tiff'
        
        tfile = tempfile.NamedTemporaryFile()
        temp_filename = tfile.name
        tfile.close()
    
        copy_drv = src_ds.GetDriver()
        copy_ds = copy_drv.CreateCopy(temp_filename, src_ds, 0 )
            
        for band_num in range(num_bands):
            copy_ds.GetRasterBand(band_num+1).WriteArray(pil_array[:,:,band_num])
        copy_ds = None
        src_ds = None
        #gdal.Unlink('/vsimem/tiffinmem')
        image_data = open(temp_filename, 'rb').read()
    
    # otherwise if the image is to be a PNG we can simply save it out
    else:
        image_data = StringIO()
        pil_image.save(image_data, format='PNG')
        image_data.seek(0)
        image_data = image_data.read()
        

    return image_data, content_type

    
def performEmail(request_dict):


    assert request_dict.get('email') is not None

    date = datetime.now(tz=pytz.UTC).strftime('%Y-%m-%d %H:%M %z')
    message = '''
A user of the ADAM service accepted the data licence agreement.

Their details are as follows:

Name: %s

Email: %s

Organisation: %s

Address: %s

Intended Use: %s

Country: %s

Have a nice day.

%s

    ''' % ( request_dict.get('name', ''),
            request_dict.get('email', ''),
            request_dict.get('organisation', ''),
            request_dict.get('address', ''),
            request_dict.get('use', ''),
            request_dict.get('country', ''),
            date)

    msg = MIMEText(message, 'plain', 'utf-8')
    msg['Subject'] = 'ADAM data licence agreement: %s' % request_dict['name']
    msg['From'] = DATA_LICENCE_EMAIL_FROM
    msg['To'] = ', '.join(DATA_LICENCE_EMAIL_TO)

    s = smtplib.SMTP('mail.noveltis.fr')
    s.sendmail( DATA_LICENCE_EMAIL_FROM, DATA_LICENCE_EMAIL_TO, msg.as_string() )


    return json.dumps({'success': 'true'}, indent=4)




def performDownload(request_dict):

    job = adam.adam_job(cfg=my_config)

    job.set_status('begin download job on host: %s' % gethostname())
    job.set_status('parsing job parameters')
    if not job.validate_input(request_dict):
        raise Exception('request input invalid !')

    job.set_status('all input was validated')
    job.set_status('extent is: %.05f, %.05f, %.05f, %.05f' % (job.extent['minx'], job.extent['miny'], job.extent['maxx'], job.extent['maxy'], ))
    job.set_status('month_index is %s: ' % job.month_index)

    job.set_status('begin loading netcdf data')
    job.load_data()

    job.set_status('begin saving netcdf data')
    job.save_netcdf()

    job.end_time = time.time()
    job.set_status('job finished, took %s seconds' % str(job.end_time - job.start_time))

    return json.dumps({'success': 'true',  'log': job.running_status, 'outputs': job.outputs}, indent=4)

def performBRDFTime(request_dict):

    job = adam.adam_job(cfg=my_config)

    job.set_status('begin brdf time series job on host: %s' % gethostname())
    job.set_status('parsing job parameters')
    if not job.validate_input(request_dict):
        raise Exception('request input invalid !')

    job.set_status('all input was validated')
    job.set_status('extent is: %.05f, %.05f, %.05f, %.05f' % (job.extent['minx'], job.extent['miny'], job.extent['maxx'], job.extent['maxy'], ))
    job.set_status('compute error is: %s' % job.do_compute_error)
    job.set_status('generate graphs is: %s' % job.generate_graphs)
    job.graph_brdf_time()

    #job.set_status('saving netcdf data')
    #job.save_netcdf()

    job.end_time = time.time()
    job.set_status('job finished, took %s seconds' % str(job.end_time - job.start_time))

    return json.dumps({'success': 'true',  'log': job.running_status, 'outputs': job.outputs}, indent=4)



def performSpectrum(request_dict):

    job = adam.adam_job(cfg=my_config)
    job.set_status('begin spectrum job on host: %s' % gethostname())
    job.set_status('received request dictionary: %s' % request_dict)
    job.set_status('parsing job parameters')
    if not job.validate_input(request_dict):
        raise Exception('request input invalid !')

    job.set_status('all input was validated')
    job.set_status('extent is: %.05f, %.05f, %.05f, %.05f' % (job.extent['minx'], job.extent['miny'], job.extent['maxx'], job.extent['maxy'], ))
    job.set_status('month_index is: %s' % job.month_index)
    job.set_status('spectral ranges are: %s' % job.spectral_domains)
    job.set_status('sza is: %s' % job.sza)
    job.set_status('phi is: %s' % job.phi)
    job.set_status('compute error is: %s' % job.do_compute_error)
    job.set_status('generate graphs is: %s' % job.generate_graphs)
    

    job.set_status('begin loading netcdf data')
    job.load_data()


    job.set_status('begin calculate reflectance spectra')
    job.process_reflectance()

    job.set_status('begin calculate brdf')
    job.process_brdf(do_spectral_averaging=False)






    if job.is_pixel_request():
        job.set_status('call make pixels spectral graph')

        job.set_status('write land pixel graph (%s pixels)' % len(job.data['idx_land']))
        for i, pixel_index in enumerate(job.data['idx_land']):
            idx = [pixel_index]
            job.graph_main_ref_spectra(case='pixels', indices=idx, title='Land pixels')


        #job.graph_main_ref_spectra(case='pixels', indices=data['idx_land'],  title='Land pixel info', graph_error=True)

        job.set_status('write ocean pixel graph (%s pixels)' % len(job.data['idx_ocean']))
        job.graph_main_ref_spectra(case='pixels', indices=job.data['idx_ocean'], title='Ocean pixels')

        job.set_status('write snow pixel graph (%s pixels)' % len(job.data['idx_snow']))
        for i, pixel_index in enumerate(job.data['idx_snow']):
            idx = [pixel_index]
            job.graph_main_ref_spectra(case='pixels', indices=idx, title='Snow pixels')



    else:
        # otherwise do stats
        job.set_status('begin stats analysis')
        job.calculate_stats(job.data['BRDF'])

        job.set_status('call make stats graph')
        job.graph_main_ref_spectra(case='stats')


    job.set_status('saving netcdf data')
    job.save_netcdf()

    job.end_time = time.time()
    job.set_status('job finished, took %s seconds' % str(job.end_time - job.start_time))

    return json.dumps({'success': 'true',  'log': job.running_status, 'outputs': job.outputs}, indent=4)




def performBRDF(request_dict):

    job = adam.adam_job(cfg=my_config)
    job.set_status('begin BRDF job on host: %s' % gethostname())
    job.set_status('received request dictionary: %s' % request_dict)
    job.set_status('parsing BRDF input')

    if not job.validate_input(request_dict):
        raise Exception('request input invalid !')

    ############################################
    # input is validated and variables populated
    job.set_status('all input was validated')


    job.set_status('extent is: %.05f, %.05f, %.05f, %.05f' % (job.extent['minx'], job.extent['miny'], job.extent['maxx'], job.extent['maxy'], ))
    job.set_status('month_index is: %s' % job.month_index)
    job.set_status('spectral ranges are: %s' % job.spectral_domains)
    job.set_status('sza is: %s' % job.sza)
    job.set_status('phi is: %s' % job.phi)
    job.set_status('compute error is: %s' % job.do_compute_error)
    job.set_status('generate graphs is: %s' % job.generate_graphs)



    job.set_status('begin loading requested data from source file')
    job.load_data()

    job.set_status('%s land pixels, %s ocean pixels, %s snow pixels' % (len(job.data['idx_land']),
                                                                       len(job.data['idx_ocean']),
                                                                       len(job.data['idx_snow'])))

    job.set_status('begin calculating reflectance spectra')
    job.process_reflectance()



    if 'polar' in job.brdf_graph_type:
#            job.vza, job.phi = process_brdf.define_angles_polar_plot(job.cfg.vza_values, job.cfg.step_polar)

        job.phi = np.linspace(0, 360, job.cfg.step_polar)
        job.vza = np.linspace(0, job.cfg.vza_values.max(), job.cfg.step_polar)
        job.set_status('begin calculating brdf')
        job.process_brdf(do_spectral_averaging=True)

        if job.is_pixel_request():

            job.set_status('call polar pixel graphs')

            job.graph_polar_plot(case='pixels', indices=job.data['idx_land'], title='Land Polar Graph', three_d=False)
            job.graph_polar_plot(case='pixels', indices=job.data['idx_land'], title='Land Graph 3D', three_d=True)

            job.graph_polar_plot(case='pixels', indices=job.data['idx_ocean'], title='Ocean Polar Graph', three_d=False)
            job.graph_polar_plot(case='pixels', indices=job.data['idx_ocean'], title='Ocean Graph 3D', three_d=True)

            job.graph_polar_plot(case='pixels', indices=job.data['idx_snow'], title='Snow Polar Graph', three_d=False)
            job.graph_polar_plot(case='pixels', indices=job.data['idx_snow'], title='Snow Graph 3D', three_d=True)

        else:

            job.set_status('perform BRDF stats analysis')
            job.calculate_stats(job.data['BRDF'])

            job.set_status('call polar stats graphs')
            job.graph_polar_plot(case='stats', title='Polar Graph', three_d=False)
            job.graph_polar_plot(case='stats', title='3D Graph', three_d=True)


    ##############
    # transect plane
    else:

        job.set_status('define_vza_hotspot')
        job.define_vza_hotspot()

        job.set_status('begin calculating brdf and brdf error')
        job.process_brdf(do_spectral_averaging=True)

        
                        
        if job.is_pixel_request():

            job.set_status('draw graph, transect plane, case pixels')

            job.graph_transect_plane(case='pixels', indices=job.data['idx_land'],  title='BRDF Land pixels', graph_error=True)
            job.graph_transect_plane(case='pixels', indices=job.data['idx_ocean'], title='BRDF Ocean pixels')
            job.graph_transect_plane(case='pixels', indices=job.data['idx_snow'],  title='BRDF Snow pixels')

        else:

            job.set_status('perform BRDF stats analysis')
            job.calculate_stats(job.data['BRDF'])

            job.set_status('draw graph, transect plane, case stats')
            job.graph_transect_plane(case = 'stats')



    # finished either pixel or stats job
    job.set_status('saving netcdf data')
    job.save_netcdf()

    job.end_time = time.time()
    job.set_status('job finished, took %s seconds' % str(job.end_time - job.start_time))


    return json.dumps({'success': 'true',  'log': job.running_status, 'outputs': job.outputs}, indent=4)




