import re

from adaptation import *

__all__ = ['MimeType', 'MimeTypedData']

class MimeDataMetaclass(type):
    implementations = {}
    def __call__(cls, value, content_type=None, charset=None):
        if content_type is None:
            if cls.default_mimetype is None:
                if charset is not None:
                    raise ValueError(charset)
                return adapt(value, cls)
            else:
                content_type = cls.default_mimetype
        if type(value) is unicode:
            charset = charset or 'UTF-8'
            value = value.encode(charset)
        mimetype, charset = parse_content_type(content_type, charset)
        for subtype in mimetype.implied:
            implementation = cls.implementations.get(subtype)
            if implementation is not None:
                if not issubclass(implementation, cls):
                    raise TypeError("Not a subclass:", implementation, cls)
                break
        else:
            implementation = cls
        return type.__call__(implementation, value, mimetype, charset)

class MimeTypedData(str):
    __metaclass__ = MimeDataMetaclass
    default_mimetype = None

    def __new__(cls, value, mimetype, charset):
        self = super(MimeTypedData, cls).__new__(cls, value)
        self.mimetype = mimetype
        self.charset = charset
        return self

    def __unicode__(self):
        return self.decode(self.charset or 'ASCII')


class MimeTypeMetaclass(type):
    registry = {}
    def __call__(cls, *args, **kw):
        if len(args) == 1 and not kw and isinstance(args[0], MimeType):
            return args[0]
        key = split_mimetype(*args, **kw)
        if key not in cls.registry:
            major, minor = key
            if major not in valid_major_mimetypes or \
                rx_mime_minortype.match(minor) is None:
                    raise ValueError("Invalid MimeType: %s/%s" % (major, minor))
            cls.registry[key] = mimetype = type.__call__(cls, major, minor)
            #cls.init_adapters(mimetype)
        return cls.registry[key]


class MimeType(object):
    __metaclass__ = MimeTypeMetaclass

    def __init__(self, major, minor):
        self.major, self.minor = major, minor
        self.minor_components = set(minor.split('+'))
        self.typestr = '%s/%s' % (self.major, self.minor)
        self.is_wildcard = (self.major == '*' or self.minor == '*')

    __implied = None

    @property
    def implied(self):
        if self.__implied is None:
            self.__implied = [self]
            if '+' in self.minor:
                for minor_comp in self.minor.split('+'):
                    self.__implied.append(MimeType(major=self.major, minor=minor_comp))
            if self.minor != '*':
                self.__implied.append(MimeType(major=self.major))
            if self.major != '*':
                self.__implied.append(MimeType.ANY)
        return self.__implied

    def __contains__(self, other):
        if isinstance(other, MimeType):
            return (other is self
                    or self is MimeType(major=other.major)
                    or self is MimeType.ANY
                    or self in other.implied)
        else:
            return False

    def __repr__(self):
        return 'MimeType(%r)' % self.typestr

    def __str__(self):
        return self.typestr

    @classmethod
    def register_constant(cls, name, typestr):
        if name != name.upper():
            raise ValueError("MimeType constant name must be in uppercase: %s" % name)
        setattr(cls, name, cls(typestr))

    def register_implementation(self, cls):
        if self in MimeDataMetaclass.implementations:
            raise KeyError(self)
        MimeDataMetaclass.implementations[self] = cls
        cls.default_mimetype = self

    def __call__(self, data, charset=None):
        if type(data) is unicode:
            if charset is None:
                charset = 'UTF-8'
            data = data.encode(charset)
        if type(data) is str:
            return MimeTypedData(data, self, charset)
        else:
            mimedata = adapt(data, MimeTypedData)
            if self not in mimedata.mimetype.implied:
                raise TypeError(data)
            return mimedata

valid_major_mimetypes = frozenset(['multipart', 'message', 'application', 'text', 'video', 'audio', 'image', '*'])
rx_mime_minortype = re.compile(r'([a-z+-]+)|\*')
rx_charset = re.compile('charset=(.+)')

def split_mimetype(typestr=None, major=None, minor=None):
    if bool(typestr) == bool(major or minor):
        if not typestr:
            raise ValueError("No MIME type specified")
        else:
            raise ValueError("Parameters conflict: %r, %r" % (typestr, (major, minor)))
    if typestr:
        try:
            major, minor = typestr.split('/')
        except:
            raise ValueError("Invalid MIME type: %s" % typestr)
    elif minor is None:
        minor = '*'
    return major, minor


def parse_content_type(content_type, charset=None):
    if isinstance(content_type, str) and ';' in content_type:
        mimetype, extra = content_type.split(';', 1)
        match = rx_charset.search(extra)
        if match is not None:
            if charset is not None:
                if charset != match.group(1):
                    raise ValueError("Conflicting charset values: %s %s" % (content_type, charset))
            else:
                charset = match.group(1)
    else:
        mimetype = content_type

    mimetype = MimeType(mimetype)
    if mimetype.is_wildcard:
        raise ValueError("Invalid MIME type for data: %s" % mimetype)

    return mimetype, charset

MimeType.register_constant('ANY', '*/*')

mimemap = dict( MULTIPART = 'multipart/*', MESSAGE = 'message/*', APPLICATION = 'application/*',
                AUDIO = 'audio/*', VIDEO = 'video/*', IMAGE = 'image/*',
                TEXT = 'text/*', PLAIN = 'text/plain', TEXT_PLAIN = 'text/plain', HTML = 'text/html',
                XHTML = 'application/xhtml+xml',
                XML = 'application/xml', XML_TEXT = 'text/xml',
                GIF = 'image/gif', PNG = 'image/png',
                JAVASCRIPT = 'application/x-javascript',
                URLENCODED = 'application/x-www-urlencoded', FORM_URLENCODED = 'application/x-www-form-urlencoded',
    )


for name, typestr in mimemap.iteritems():
    MimeType.register_constant(name, typestr)


