# -*- coding: utf-8 -*-

# Copyright 2013 Vincent Jacques
# vincent@vincent-jacques.net

# This file is part of AnotherPyGraphvizAgain. http://jacquev6.github.com/AnotherPyGraphvizAgain

# AnotherPyGraphvizAgain is free software: you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.

# AnotherPyGraphvizAgain is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.

# You should have received a copy of the GNU Lesser General Public License along with AnotherPyGraphvizAgain.  If not, see <http://www.gnu.org/licenses/>.

from .Framework import ConstructionTestCase

import AnotherPyGraphvizAgain.Raw as gv


class VerboseConstruction(ConstructionTestCase(gv.Graph)):
    def testBadAdd(self):
        with self.assertRaises(Exception):
            self.g.add("This is a string")

    def testAttributedGraph(self):
        self.g.set("dir", "LR")

        self.expect("""digraph "First graph" {dir="LR";}""")

    def testResetAttribute(self):
        self.g.set("dir", "LR")
        self.g.set("dir", None)

        self.expect("""digraph "First graph" {}""")

    def testSimpleNode(self):
        self.g.add(gv.Node("First"))

        self.expect("""digraph "First graph" {First;}""")

    def testAttributedNode(self):
        n = gv.Node("First")
        n.set("label", "First node")
        self.g.add(n)

        self.expect("""digraph "First graph" {First[label="First node"];}""")

    def testAttributedWithQuotes(self):
        n = gv.Node("First")
        n.set("attr", 'xx"yy"zz')
        self.g.add(n)

        self.expect("""digraph "First graph" {First[attr="xx\\\"yy\\\"zz"];}""")

    def testSimpleLink(self):
        n1 = gv.Node("First")
        n2 = gv.Node("Second")
        self.g.add(n1)
        self.g.add(n2)
        self.g.add(gv.Link(n1, n2))

        self.expect("""digraph "First graph" {First;Second;First->Second;}""")

    def testLabeledLink(self):
        n1 = gv.Node("First")
        n2 = gv.Node("Second")
        self.g.add(n1)
        self.g.add(n2)
        l = gv.Link(n1, n2)
        l.set("label", "First link")
        self.g.add(l)

        self.expect("""digraph "First graph" {First;Second;First->Second[label="First link"];}""")

    def testEdgeAttributes(self):
        self.g.edgeAttr.set("color", "red")

        self.expect("""digraph "First graph" {edge [color="red"];}""")

    def testNodeAttributes(self):
        self.g.nodeAttr.set("shape", "box")

        self.expect("""digraph "First graph" {node [shape="box"];}""")

    def testMultipleAdd(self):
        n = gv.Node("First")
        self.g.add(n)
        self.g.add(n)

        self.expect("""digraph "First graph" {First;}""")

    def testAddIdenticalNodes(self):
        self.g.add(gv.Node("First"))
        self.g.add(gv.Node("First"))

        self.expect("""digraph "First graph" {First;First;}""")

    def testSubGraph(self):
        sg = gv.Subgraph("sub")
        sg.add(gv.Node("First"))
        self.g.add(sg)

        self.expect("""digraph "First graph" {subgraph sub{First;};}""")


class ChainedConstruction(ConstructionTestCase(gv.Graph)):
    def testChainedAdds(self):
        self.g.add(gv.Node("First")).add(gv.Node("Second"))

        self.expect("""digraph "First graph" {First;Second;}""")

    def testChainedSetAndAdd(self):
        self.g.set("dir", "LR").add(gv.Node("First"))

        self.expect("""digraph "First graph" {dir="LR";First;}""")

    def testAttributedNode(self):
        self.g.add(gv.Node("First").set("shape", "box").set("color", "red"))

        self.expect("""digraph "First graph" {First[color="red",shape="box"];}""")

    def testAttributedLink(self):
        n1 = gv.Node("First")
        n2 = gv.Node("Second")
        self.g.add(n1).add(n2).add(gv.Link(n1, n2).set("color", "red"))

        self.expect("""digraph "First graph" {First;Second;First->Second[color="red"];}""")

    def testLinkTo(self):
        self.g.add(gv.Node("First").linkTo(gv.Node("Second")))

        self.expect("""digraph "First graph" {First;Second;First->Second;}""")

    def testAttributedLinkTo(self):
        self.g.add(gv.Node("First").linkTo(gv.Node("Second")).set("color", "red"))

        self.expect("""digraph "First graph" {First;Second;First->Second[color="red"];}""")
