"""
    Theme support
    =============
    
    Creating themes for Apydia is fairly easy if you know how to use
    [Genshi](http://genshi.edgewall.org/ "Genshi - Trac") (and it's limited 
    XInclude implementation). If you don't, take a look at the bundled 
    "elixir"- and "apydia"-themes to get an idea.
    
    Each theme must have a ``theme.ini``-file with one or more of the
    following options in it's ``[theme]``-section:
    
    <table>
        <tr>
            <th>Option</th>
            <th>Description</th>
        </tr>
        <tr>
            <td><tt>title</tt></td>
            <td>The name of the project for which documention is
                generated.</td>
        </tr>
        <tr>
            <td><tt>inherits</tt></td>
            <td>The parent theme the current theme will inherit. You might
                want to set this to <tt>default</tt> and modify the default
                theme. You can omit it, if you prefer, and create all
                templates and CSS from scratch.</td>
        </tr>
        <tr>
            <td><tt>description</tt></td>
            <td>Short description about the theme. This is currently
                not used at all.</td>
        </tr>
        <tr>
            <td><tt>pygments_style</tt></td>
            <td>The identifier of the Pygments style to use for syntax
                highlighting.</td>
        </tr>
        <tr>
            <td><tt>pygments_style_css</tt></td>
            <td>Custom filename for the CSS-file generated by Pygments.
                If left out it will default to the value of
                <tt>pygments_style</tt> appended by <tt>-pygments.css</tt>.
            </td>
        </tr>
        <tr>
            <td><tt>resources</tt></td>
            <td>Whitespace separated list for resources to be copied. May
                contain wildcards (see Python's <tt>glob</tt>-module for
                details). Apydia will try to locate the specified files
                relative to the theme's base directory, then the parent
                theme's directory and so forth.</td>
        </tr>
    </table>
    
    Example:
    --------
    
        [theme]
        title = Apydia
        inherits = default
        description = "API Reference Theme for Apydia"
        pygments_style = apydia
        pygments_style_css = css/pygments.css
        resources = css/default.css css/apydia.css images/*
    
"""

import os
from glob import glob
from pkg_resources import iter_entry_points, DistributionNotFound
from ConfigParser import RawConfigParser, NoOptionError, NoSectionError

__all__ = ["Theme", "ThemeNotFoundError", "InheritanceError"]


# Filename for theme descriptors
DESCRIPTOR_NAME = "theme.ini"

# Fallback Pygments theme, in case no pygments style is specified by the theme
DEFAULT_PYGMENTS_STYLE = "apydiadefault"


class ThemeNotFoundError(StandardError):
    """
        A ``ThemeNotFoundError`` will be raised if the specified theme is not
        available.
    """
    pass

class InheritanceError(StandardError):
    """
        Exception to be raised when there's any problem with inheritance.
        Currently ``InheritanceError``s will only be raised if a theme is
        configured to inherit from itself or from one of it's descendants.
    """
    pass

class Theme(object):
    
    """
        The ``Theme``-class represents and loads themes. It also knows where
        the theme's and it parent theme's resources can be found.
    """
    
    themes = dict()
    
    def __init__(self, name, module):
        self.name = name
        self.module = module
        self.parent = None
        self.resource_patterns = []
        self._resources = None
        
        cp = RawConfigParser()
        cp.read(os.path.join(self.path, DESCRIPTOR_NAME))
        try:
            self.parent = Theme.load(cp.get("theme", "inherits"))
            self.find_loops()
        except (NoOptionError, NoSectionError):
            pass # no parent theme
        
        try:
            patterns = cp.get("theme", "resources")
            self.resource_patterns += patterns.split()
        except (NoOptionError, NoSectionError):
            pass # no resources defined
        
        try:
            self.pygments_style = cp.get("theme", "pygments_style")
        except NoOptionError:
            if self.parent:
                self.pygments_style = self.parent.pygments_style
            else:
                self.pygments_style = DEFAULT_PYGMENTS_STYLE
            pass # no pygments style given
            
        
        try:
            self.pygments_style_css = cp.get("theme", "pygments_style_css")
        except NoOptionError:
            if self.parent:
                self.pygments_style_css = self.parent.pygments_style_css
            else:
                self.pygments_style_css = self.pygments_style + "-pygments.css"
    
    @classmethod
    def load(cls, name):
        """
            Loads and returns the theme with the identifier
            equal to the value of ``name``.
        """
        
        if name in cls.themes:
            return cls.themes[name]
        module = None
        for entrypoint in iter_entry_points("apydia.themes", name):
            try:
                module = entrypoint.load()
            except DistributionNotFound, msg:
                warnings.warn("DistributionNotFound: %s" % msg)
        if not module:
            raise ThemeNotFoundError, "No such theme \"%s\" can be found." % name
        else:
            cls.themes[name] = Theme(name, module)
            return cls.themes[name]
    
    @property
    def resources(self):
        if self._resources is None:
            self._resources = self.find_all_resources(self.resource_patterns)
        return self._resources
    
    def find_all_resources(self, patterns):
        """ Find all required resource-files for this theme. """
        resources = []
        for pattern in patterns:
            for template_dir in self.template_dirs:
                found = glob(os.path.join(template_dir, pattern))
                if found:
                    for path in found:
                        resources.append((path, path[len(template_dir)+1:]))
                    break
        return resources
    
    @property
    def template_dirs(self):
        """
            Returns a list of this theme's and it parents'
            base directories. 
        """
        if self.parent:
            return [self.path] + self.parent.template_dirs
        else:
            return [self.path]
    
    @property
    def path(self):
        """ Returns this theme's base directory. """
        return os.path.realpath(sorted(self.module.__path__,
                                       cmp=lambda a,b:len(a)-len(b))[0])
    
    def find_loops(self):
        """
            Raises an ``InheritanceError`` if this theme inherits
            itself or one of it's descendants.
        """
        theme = self
        while theme.parent:
            if self == theme.parent:
                raise InheritanceError, "The theme inherits itself, which is nonsense"
            theme = theme.parent
